/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage.ldap.idm.store.ldap;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.naming.AuthenticationException;
import javax.naming.Binding;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NameAlreadyBoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.Control;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.PagedResultsControl;
import javax.naming.ldap.PagedResultsResponseControl;
import javax.naming.ldap.StartTlsRequest;
import javax.naming.ldap.StartTlsResponse;
import org.jboss.logging.Logger;
import org.keycloak.common.util.Time;
import org.keycloak.models.LDAPConstants;
import org.keycloak.models.ModelException;
import org.keycloak.storage.ldap.LDAPConfig;
import org.keycloak.storage.ldap.idm.model.LDAPDn;
import org.keycloak.storage.ldap.idm.query.internal.LDAPQuery;
import org.keycloak.storage.ldap.idm.store.ldap.LDAPUtil;
import org.keycloak.storage.ldap.mappers.LDAPOperationDecorator;

public class LDAPOperationManager {
    private static final Logger logger = Logger.getLogger(LDAPOperationManager.class);
    private static final Logger perfLogger = Logger.getLogger(LDAPOperationManager.class, (String)"perf");
    private final LDAPConfig config;
    private final Map<String, Object> connectionProperties;

    public LDAPOperationManager(LDAPConfig config) throws NamingException {
        this.config = config;
        this.connectionProperties = Collections.unmodifiableMap(this.createConnectionProperties());
    }

    public void modifyAttribute(String dn, Attribute attribute) {
        ModificationItem[] mods = new ModificationItem[]{new ModificationItem(2, attribute)};
        this.modifyAttributes(dn, mods, null);
    }

    public void modifyAttributes(String dn, NamingEnumeration<Attribute> attributes) {
        try {
            ArrayList<ModificationItem> modItems = new ArrayList<ModificationItem>();
            while (attributes.hasMore()) {
                ModificationItem modItem = new ModificationItem(2, attributes.next());
                modItems.add(modItem);
            }
            this.modifyAttributes(dn, modItems.toArray(new ModificationItem[0]), null);
        }
        catch (NamingException ne) {
            throw new ModelException("Could not modify attributes on entry from DN [" + dn + "]", (Throwable)ne);
        }
    }

    public void removeAttribute(String dn, Attribute attribute) {
        ModificationItem[] mods = new ModificationItem[]{new ModificationItem(3, attribute)};
        this.modifyAttributes(dn, mods, null);
    }

    public void addAttribute(String dn, Attribute attribute) {
        ModificationItem[] mods = new ModificationItem[]{new ModificationItem(1, attribute)};
        this.modifyAttributes(dn, mods, null);
    }

    public void removeEntry(final String entryDn) {
        try {
            this.execute(new LdapOperation<SearchResult>(){

                @Override
                public SearchResult execute(LdapContext context) throws NamingException {
                    if (logger.isTraceEnabled()) {
                        logger.tracef("Removing entry with DN [%s]", (Object)entryDn);
                    }
                    LDAPOperationManager.this.destroySubcontext(context, entryDn);
                    return null;
                }

                public String toString() {
                    return "LdapOperation: remove\n" + " dn: " + entryDn;
                }
            });
        }
        catch (NamingException e) {
            throw new ModelException("Could not remove entry from DN [" + entryDn + "]", (Throwable)e);
        }
    }

    public String renameEntry(final String oldDn, final String newDn, final boolean fallback) {
        try {
            String newNonConflictingDn = this.execute(new LdapOperation<String>(){

                @Override
                public String execute(LdapContext context) throws NamingException {
                    String dn = newDn;
                    int max = 5;
                    for (int i = 0; i < max; ++i) {
                        try {
                            context.rename(new LdapName(oldDn), new LdapName(dn));
                            return dn;
                        }
                        catch (NameAlreadyBoundException ex) {
                            if (!fallback) {
                                throw ex;
                            }
                            String failedDn = dn;
                            if (i >= max) {
                                logger.warnf("Failed all fallbacks for renaming [%s]", (Object)oldDn);
                                throw ex;
                            }
                            dn = LDAPOperationManager.this.findNextDNForFallback(newDn, i);
                            logger.warnf("Failed to rename DN [%s] to [%s]. Will try to fallback to DN [%s]", (Object)oldDn, (Object)failedDn, (Object)dn);
                            continue;
                        }
                    }
                    throw new ModelException("Could not rename entry from DN [" + oldDn + "] to new DN [" + newDn + "]. All fallbacks failed");
                }

                public String toString() {
                    return "LdapOperation: renameEntry\n" + " oldDn: " + oldDn + "\n" + " newDn: " + newDn;
                }
            });
            return newNonConflictingDn;
        }
        catch (NamingException e) {
            throw new ModelException("Could not rename entry from DN [" + oldDn + "] to new DN [" + newDn + "]", (Throwable)e);
        }
    }

    private String findNextDNForFallback(String newDn, int counter) {
        LDAPDn dn = LDAPDn.fromString(newDn);
        String rdnAttrName = dn.getFirstRdnAttrName();
        String rdnAttrVal = dn.getFirstRdnAttrValue();
        LDAPDn parentDn = dn.getParentDn();
        parentDn.addFirst(rdnAttrName, rdnAttrVal + counter);
        return parentDn.toString();
    }

    public List<SearchResult> search(final String baseDN, final String filter, final Collection<String> returningAttributes, final int searchScope) throws NamingException {
        final ArrayList result = new ArrayList();
        final SearchControls cons = this.getSearchControls(returningAttributes, searchScope);
        try {
            return this.execute(new LdapOperation<List<SearchResult>>(){

                @Override
                public List<SearchResult> execute(LdapContext context) throws NamingException {
                    NamingEnumeration<SearchResult> search = context.search((Name)new LdapName(baseDN), filter, cons);
                    while (search.hasMoreElements()) {
                        result.add(search.nextElement());
                    }
                    search.close();
                    return result;
                }

                public String toString() {
                    return "LdapOperation: search\n" + " baseDn: " + baseDN + "\n" + " filter: " + filter + "\n" + " searchScope: " + searchScope + "\n" + " returningAttrs: " + returningAttributes + "\n" + " resultSize: " + result.size();
                }
            });
        }
        catch (NamingException e) {
            logger.errorf((Throwable)e, "Could not query server using DN [%s] and filter [%s]", (Object)baseDN, (Object)filter);
            throw e;
        }
    }

    public List<SearchResult> searchPaginated(final String baseDN, final String filter, final LDAPQuery identityQuery) throws NamingException {
        final ArrayList result = new ArrayList();
        final SearchControls cons = this.getSearchControls(identityQuery.getReturningLdapAttributes(), identityQuery.getSearchScope());
        if (identityQuery.getPaginationContext() == null) {
            LdapContext ldapContext = this.createLdapContext();
            identityQuery.initPagination(ldapContext);
        }
        try {
            return this.execute(new LdapOperation<List<SearchResult>>(){

                @Override
                public List<SearchResult> execute(LdapContext context) throws NamingException {
                    try {
                        byte[] cookie = identityQuery.getPaginationContext().getCookie();
                        PagedResultsControl pagedControls = new PagedResultsControl(identityQuery.getLimit(), cookie, true);
                        context.setRequestControls(new Control[]{pagedControls});
                        NamingEnumeration<SearchResult> search = context.search((Name)new LdapName(baseDN), filter, cons);
                        while (search.hasMoreElements()) {
                            result.add(search.nextElement());
                        }
                        search.close();
                        Control[] responseControls = context.getResponseControls();
                        if (responseControls != null) {
                            for (Control respControl : responseControls) {
                                if (!(respControl instanceof PagedResultsResponseControl)) continue;
                                PagedResultsResponseControl prrc = (PagedResultsResponseControl)respControl;
                                cookie = prrc.getCookie();
                                identityQuery.getPaginationContext().setCookie(cookie);
                            }
                        }
                        return result;
                    }
                    catch (IOException ioe) {
                        logger.errorf((Throwable)ioe, "Could not query server with paginated query using DN [%s], filter [%s]", (Object)baseDN, (Object)filter);
                        throw new NamingException(ioe.getMessage());
                    }
                }

                public String toString() {
                    return "LdapOperation: searchPaginated\n" + " baseDn: " + baseDN + "\n" + " filter: " + filter + "\n" + " searchScope: " + identityQuery.getSearchScope() + "\n" + " returningAttrs: " + identityQuery.getReturningLdapAttributes() + "\n" + " limit: " + identityQuery.getLimit() + "\n" + " resultSize: " + result.size();
                }
            }, identityQuery.getPaginationContext().getLdapContext(), null);
        }
        catch (NamingException e) {
            logger.errorf((Throwable)e, "Could not query server using DN [%s] and filter [%s]", (Object)baseDN, (Object)filter);
            throw e;
        }
    }

    private SearchControls getSearchControls(Collection<String> returningAttributes, int searchScope) {
        SearchControls cons = new SearchControls();
        cons.setSearchScope(searchScope);
        cons.setReturningObjFlag(false);
        returningAttributes = this.getReturningAttributes(returningAttributes);
        cons.setReturningAttributes(returningAttributes.toArray(new String[returningAttributes.size()]));
        return cons;
    }

    public String getFilterById(String id) {
        String filter = null;
        if (this.config.isObjectGUID()) {
            final String strObjectGUID = "<GUID=" + id + ">";
            try {
                Attributes attributes = this.execute(new LdapOperation<Attributes>(){

                    @Override
                    public Attributes execute(LdapContext context) throws NamingException {
                        return context.getAttributes(strObjectGUID);
                    }

                    public String toString() {
                        return "LdapOperation: GUIDResolve\n" + " strObjectGUID: " + strObjectGUID;
                    }
                });
                byte[] objectGUID = (byte[])attributes.get("objectGUID").get();
                filter = "(&(objectClass=*)(" + this.getUuidAttributeName() + "=" + LDAPUtil.convertObjectGUIToByteString(objectGUID) + "))";
            }
            catch (NamingException ne) {
                filter = null;
            }
        }
        if (filter == null) {
            filter = "(&(objectClass=*)(" + this.getUuidAttributeName() + "=" + id + "))";
        }
        if (logger.isTraceEnabled()) {
            logger.tracef("Using filter for lookup user by LDAP ID: %s", filter);
        }
        return filter;
    }

    public SearchResult lookupById(final String baseDN, String id, final Collection<String> returningAttributes) {
        final String filter = this.getFilterById(id);
        try {
            final SearchControls cons = this.getSearchControls(returningAttributes, this.config.getSearchScope());
            return this.execute(new LdapOperation<SearchResult>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public SearchResult execute(LdapContext context) throws NamingException {
                    try (NamingEnumeration<SearchResult> search = context.search((Name)new LdapName(baseDN), filter, cons);){
                        if (search.hasMoreElements()) {
                            SearchResult searchResult = search.next();
                            return searchResult;
                        }
                    }
                    return null;
                }

                public String toString() {
                    return "LdapOperation: lookupById\n" + " baseDN: " + baseDN + "\n" + " filter: " + filter + "\n" + " searchScope: " + cons.getSearchScope() + "\n" + " returningAttrs: " + returningAttributes;
                }
            });
        }
        catch (NamingException e) {
            throw new ModelException("Could not query server using DN [" + baseDN + "] and filter [" + filter + "]", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void destroySubcontext(LdapContext context, String dn) {
        try {
            NamingEnumeration<Binding> enumeration = null;
            try {
                enumeration = context.listBindings(new LdapName(dn));
                while (enumeration.hasMore()) {
                    Binding binding = enumeration.next();
                    String name = binding.getNameInNamespace();
                    this.destroySubcontext(context, name);
                }
                context.unbind(new LdapName(dn));
            }
            finally {
                try {
                    enumeration.close();
                }
                catch (Exception exception) {}
            }
        }
        catch (Exception e) {
            throw new ModelException("Could not unbind DN [" + dn + "]", (Throwable)e);
        }
    }

    public void authenticate(String dn, String password) throws AuthenticationException {
        InitialContext authCtx = null;
        try {
            if (password == null || password.isEmpty()) {
                throw new AuthenticationException("Empty password used");
            }
            Hashtable<String, Object> env = new Hashtable<String, Object>(this.connectionProperties);
            env.put("com.sun.jndi.ldap.connect.pool", "false");
            if (!this.config.isStartTls()) {
                env.put("java.naming.security.authentication", this.config.getAuthType());
                env.put("java.naming.security.principal", dn);
                env.put("java.naming.security.credentials", password);
            }
            authCtx = new InitialLdapContext(env, null);
            this.startTLS((LdapContext)((Object)authCtx), this.config.getAuthType(), dn, password);
        }
        catch (AuthenticationException ae) {
            if (logger.isDebugEnabled()) {
                logger.debugf((Throwable)ae, "Authentication failed for DN [%s]", (Object)dn);
            }
            throw ae;
        }
        catch (Exception e) {
            logger.errorf((Throwable)e, "Unexpected exception when validating password of DN [%s]", (Object)dn);
            throw new AuthenticationException("Unexpected exception when validating password of user");
        }
        finally {
            if (authCtx != null) {
                try {
                    authCtx.close();
                }
                catch (NamingException namingException) {}
            }
        }
    }

    private void startTLS(LdapContext ldapContext, String authType, String bindDN, String bindCredentials) throws NamingException {
        if (this.config.isStartTls()) {
            try {
                StartTlsResponse tls = (StartTlsResponse)ldapContext.extendedOperation(new StartTlsRequest());
                tls.negotiate();
                char[] bindCredential = null;
                ldapContext.addToEnvironment("java.naming.security.authentication", authType);
                if (bindCredentials != null) {
                    bindCredential = bindCredentials.toCharArray();
                }
                if (!"none".equals(authType)) {
                    ldapContext.addToEnvironment("java.naming.security.principal", bindDN);
                    ldapContext.addToEnvironment("java.naming.security.credentials", bindCredential);
                }
            }
            catch (Exception e) {
                logger.error((Object)"Could not negotiate TLS", (Throwable)e);
            }
            ldapContext.lookup("");
        }
    }

    public void modifyAttributesNaming(final String dn, final ModificationItem[] mods, LDAPOperationDecorator decorator) throws NamingException {
        if (logger.isTraceEnabled()) {
            logger.tracef("Modifying attributes for entry [%s]: [", (Object)dn);
            for (ModificationItem item : mods) {
                Object values = item.getAttribute().size() > 0 ? item.getAttribute().get() : "No values";
                String attrName = item.getAttribute().getID().toUpperCase();
                if (attrName.contains("PASSWORD") || attrName.contains("UNICODEPWD")) {
                    values = "********************";
                }
                logger.tracef("  Op [%s]: %s = %s", item.getModificationOp(), (Object)item.getAttribute().getID(), values);
            }
            logger.tracef("]", new Object[0]);
        }
        this.execute(new LdapOperation<Void>(){

            @Override
            public Void execute(LdapContext context) throws NamingException {
                context.modifyAttributes(new LdapName(dn), mods);
                return null;
            }

            public String toString() {
                return "LdapOperation: modify\n" + " dn: " + dn + "\n" + " modificationsSize: " + mods.length;
            }
        }, null, decorator);
    }

    public void modifyAttributes(String dn, ModificationItem[] mods, LDAPOperationDecorator decorator) {
        try {
            this.modifyAttributesNaming(dn, mods, decorator);
        }
        catch (NamingException e) {
            throw new ModelException("Could not modify attribute for DN [" + dn + "]", (Throwable)e);
        }
    }

    public void createSubContext(final String name, final Attributes attributes) {
        try {
            if (logger.isTraceEnabled()) {
                logger.tracef("Creating entry [%s] with attributes: [", (Object)name);
                NamingEnumeration<? extends Attribute> all = attributes.getAll();
                while (all.hasMore()) {
                    Attribute attribute = all.next();
                    String attrName = attribute.getID().toUpperCase();
                    Object attrVal = attribute.get();
                    if (attrName.contains("PASSWORD") || attrName.contains("UNICODEPWD")) {
                        attrVal = "********************";
                    }
                    logger.tracef("  %s = %s", (Object)attribute.getID(), attrVal);
                }
                logger.tracef("]", new Object[0]);
            }
            this.execute(new LdapOperation<Void>(){

                @Override
                public Void execute(LdapContext context) throws NamingException {
                    DirContext subcontext = context.createSubcontext(new LdapName(name), attributes);
                    subcontext.close();
                    return null;
                }

                public String toString() {
                    return "LdapOperation: create\n" + " dn: " + name + "\n" + " attributesSize: " + attributes.size();
                }
            });
        }
        catch (NamingException e) {
            throw new ModelException("Error creating subcontext [" + name + "]", (Throwable)e);
        }
    }

    private String getUuidAttributeName() {
        return this.config.getUuidLDAPAttributeName();
    }

    public Attributes getAttributes(String entryUUID, String baseDN, Set<String> returningAttributes) {
        SearchResult search = this.lookupById(baseDN, entryUUID, returningAttributes);
        if (search == null) {
            throw new ModelException("Couldn't find item with ID [" + entryUUID + " under base DN [" + baseDN + "]");
        }
        return search.getAttributes();
    }

    public String decodeEntryUUID(Object entryUUID) {
        String id = this.config.isObjectGUID() && entryUUID instanceof byte[] ? LDAPUtil.decodeObjectGUID((byte[])entryUUID) : entryUUID.toString();
        return id;
    }

    private LdapContext createLdapContext() throws NamingException {
        if (!this.config.isStartTls()) {
            return new InitialLdapContext(new Hashtable<String, Object>(this.connectionProperties), null);
        }
        InitialLdapContext ldapContext = new InitialLdapContext(new Hashtable<String, Object>(this.connectionProperties), null);
        this.startTLS(ldapContext, this.config.getAuthType(), this.config.getBindDN(), this.config.getBindCredential());
        return ldapContext;
    }

    private Map<String, Object> createConnectionProperties() {
        Properties additionalProperties;
        String readTimeout;
        String connectionTimeout;
        String url;
        HashMap<String, Object> env = new HashMap<String, Object>();
        env.put("java.naming.factory.initial", this.config.getFactoryName());
        if (!this.config.isStartTls()) {
            String authType = this.config.getAuthType();
            env.put("java.naming.security.authentication", authType);
            String bindDN = this.config.getBindDN();
            char[] bindCredential = null;
            if (this.config.getBindCredential() != null) {
                bindCredential = this.config.getBindCredential().toCharArray();
            }
            if (!"none".equals(authType)) {
                env.put("java.naming.security.principal", bindDN);
                env.put("java.naming.security.credentials", bindCredential);
            }
        }
        if ((url = this.config.getConnectionUrl()) != null) {
            env.put("java.naming.provider.url", url);
        } else {
            logger.warn((Object)"LDAP URL is null. LDAPOperationManager won't work correctly");
        }
        String useTruststoreSpi = this.config.getUseTruststoreSpi();
        LDAPConstants.setTruststoreSpiIfNeeded((String)useTruststoreSpi, (String)url, env);
        String connectionPooling = this.config.getConnectionPooling();
        if (connectionPooling != null) {
            env.put("com.sun.jndi.ldap.connect.pool", connectionPooling);
        }
        if ((connectionTimeout = this.config.getConnectionTimeout()) != null && !connectionTimeout.isEmpty()) {
            env.put("com.sun.jndi.ldap.connect.timeout", connectionTimeout);
        }
        if ((readTimeout = this.config.getReadTimeout()) != null && !readTimeout.isEmpty()) {
            env.put("com.sun.jndi.ldap.read.timeout", readTimeout);
        }
        if ((additionalProperties = this.config.getAdditionalConnectionProperties()) != null) {
            for (Object object : additionalProperties.keySet()) {
                env.put(object.toString(), additionalProperties.getProperty(object.toString()));
            }
        }
        StringBuilder binaryAttrsBuilder = new StringBuilder();
        if (this.config.isObjectGUID()) {
            binaryAttrsBuilder.append("objectGUID").append(" ");
        }
        for (String attrName : this.config.getBinaryAttributeNames()) {
            binaryAttrsBuilder.append(attrName).append(" ");
        }
        String string = binaryAttrsBuilder.toString().trim();
        if (!string.isEmpty()) {
            env.put("java.naming.ldap.attributes.binary", string);
        }
        if (logger.isDebugEnabled()) {
            HashMap<String, Object> copyEnv = new HashMap<String, Object>(env);
            if (copyEnv.containsKey("java.naming.security.credentials")) {
                copyEnv.put("java.naming.security.credentials", "**************************************");
            }
            logger.debugf("Creating LdapContext using properties: [%s]", copyEnv);
        }
        return env;
    }

    private <R> R execute(LdapOperation<R> operation) throws NamingException {
        return this.execute(operation, null, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <R> R execute(LdapOperation<R> operation, LdapContext context, LDAPOperationDecorator decorator) throws NamingException {
        boolean manageContext = context == null;
        Long start = null;
        try {
            if (perfLogger.isDebugEnabled()) {
                start = Time.currentTimeMillis();
            }
            if (manageContext) {
                context = this.createLdapContext();
            }
            if (decorator != null) {
                decorator.beforeLDAPOperation(context, operation);
            }
            R r = operation.execute(context);
            return r;
        }
        finally {
            if (context != null && manageContext) {
                try {
                    context.close();
                }
                catch (NamingException ne) {
                    logger.error((Object)"Could not close Ldap context.", (Throwable)ne);
                }
            }
            if (perfLogger.isDebugEnabled()) {
                long took = Time.currentTimeMillis() - start;
                if (took > 100L) {
                    perfLogger.debugf("\n%s\ntook: %d ms\n", (Object)operation.toString(), (Object)took);
                } else if (perfLogger.isTraceEnabled()) {
                    perfLogger.tracef("\n%s\ntook: %d ms\n", (Object)operation.toString(), (Object)took);
                }
            }
        }
    }

    private Set<String> getReturningAttributes(Collection<String> returningAttributes) {
        HashSet<String> result = new HashSet<String>();
        result.addAll(returningAttributes);
        result.add(this.getUuidAttributeName());
        result.add("objectclass");
        return result;
    }

    public static interface LdapOperation<R> {
        public R execute(LdapContext var1) throws NamingException;
    }
}

