"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.source = void 0;

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _eventKit = require("event-kit");

var _relayNetworkLayerManager = _interopRequireDefault(require("../relay-network-layer-manager"));

var _author = _interopRequireWildcard(require("./author"));

var _keytarStrategy = require("../shared/keytar-strategy");

var _modelObserver = _interopRequireDefault(require("./model-observer"));

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj["default"] = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

// This is a guess about what a reasonable value is. Can adjust if performance is poor.
const MAX_COMMITS = 5000;
const source = {
  PENDING: Symbol('pending'),
  GITLOG: Symbol('git log'),
  GITHUBAPI: Symbol('github API')
};
exports.source = source;

class GraphQLCache {
  // One hour
  constructor() {
    this.bySlug = new Map();
  }

  get(remote) {
    const slug = remote.getSlug();
    const {
      ts,
      data
    } = this.bySlug.get(slug) || {
      ts: -Infinity,
      data: {}
    };

    if (Date.now() - ts > this.constructor.MAX_AGE_MS) {
      this.bySlug["delete"](slug);
      return null;
    }

    return data;
  }

  set(remote, data) {
    this.bySlug.set(remote.getSlug(), {
      ts: Date.now(),
      data
    });
  }

}

_defineProperty(GraphQLCache, "MAX_AGE_MS", 3.6e6);

class UserStore {
  constructor({
    repository,
    login,
    config
  }) {
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable(); // TODO: [ku 3/2018] Consider using Dexie (indexDB wrapper) like Desktop and persist users across sessions

    this.allUsers = new Map();
    this.excludedUsers = new Set();
    this.users = [];
    this.committer = _author.nullAuthor;
    this.last = {
      source: source.PENDING,
      repository: null,
      excludedUsers: this.excludedUsers
    };
    this.cache = new GraphQLCache();
    this.repositoryObserver = new _modelObserver["default"]({
      fetchData: r => (0, _yubikiri["default"])({
        committer: r.getCommitter(),
        authors: r.getAuthors({
          max: MAX_COMMITS
        }),
        remotes: r.getRemotes()
      }),
      didUpdate: () => this.loadUsers()
    });
    this.repositoryObserver.setActiveModel(repository);
    this.loginObserver = new _modelObserver["default"]({
      didUpdate: () => this.loadUsers()
    });
    this.loginObserver.setActiveModel(login);
    this.subs.add(config.observe('github.excludedUsers', value => {
      this.excludedUsers = new Set((value || '').split(/\s*,\s*/).filter(each => each.length > 0));
      return this.loadUsers();
    }));
  }

  dispose() {
    this.subs.dispose();
    this.emitter.dispose();
  }

  async loadUsers() {
    const data = this.repositoryObserver.getActiveModelData();

    if (!data) {
      return;
    }

    this.setCommitter(data.committer);
    const githubRemotes = Array.from(data.remotes).filter(remote => remote.isGithubRepo());

    if (githubRemotes.length > 0) {
      await this.loadUsersFromGraphQL(githubRemotes);
    } else {
      this.addUsers(data.authors, source.GITLOG);
    } // if for whatever reason, no committers can be added, fall back to
    // using git log committers as the last resort


    if (this.allUsers.size === 0) {
      this.addUsers(data.authors, source.GITLOG);
    }
  }

  loadUsersFromGraphQL(remotes) {
    return Promise.all(Array.from(remotes, remote => this.loadMentionableUsers(remote)));
  }

  async getToken(loginModel, loginAccount) {
    if (!loginModel) {
      return null;
    }

    const token = await loginModel.getToken(loginAccount);

    if (token === _keytarStrategy.UNAUTHENTICATED || token === _keytarStrategy.INSUFFICIENT || token instanceof Error) {
      return null;
    }

    return token;
  }

  async loadMentionableUsers(remote) {
    const cached = this.cache.get(remote);

    if (cached !== null) {
      this.addUsers(cached, source.GITHUBAPI);
      return;
    }

    const endpoint = remote.getEndpoint();
    const token = await this.getToken(this.loginObserver.getActiveModel(), endpoint.getLoginAccount());

    if (!token) {
      return;
    }

    const fetchQuery = _relayNetworkLayerManager["default"].getFetchQuery(endpoint, token);

    let hasMore = true;
    let cursor = null;
    const remoteUsers = [];

    while (hasMore) {
      const response = await fetchQuery({
        name: 'GetMentionableUsers',
        text: `
          query GetMentionableUsers($owner: String!, $name: String!, $first: Int!, $after: String) {
            repository(owner: $owner, name: $name) {
              mentionableUsers(first: $first, after: $after) {
                nodes {
                  login
                  email
                  name
                }
                pageInfo {
                  hasNextPage
                  endCursor
                }
              }
            }
          }
        `
      }, {
        owner: remote.getOwner(),
        name: remote.getRepo(),
        first: 100,
        after: cursor
      });
      /* istanbul ignore if */

      if (response.errors && response.errors.length > 1) {
        // eslint-disable-next-line no-console
        console.error(`Error fetching mentionable users:\n${response.errors.map(e => e.message).join('\n')}`);
      }

      if (!response.data || !response.data.repository) {
        break;
      }

      const connection = response.data.repository.mentionableUsers;
      const authors = connection.nodes.map(node => {
        if (node.email === '') {
          node.email = `${node.login}@users.noreply.github.com`;
        }

        return new _author["default"](node.email, node.name, node.login);
      });
      this.addUsers(authors, source.GITHUBAPI);
      remoteUsers.push(...authors);
      cursor = connection.pageInfo.endCursor;
      hasMore = connection.pageInfo.hasNextPage;
    }

    this.cache.set(remote, remoteUsers);
  }

  addUsers(users, nextSource) {
    let changed = false;

    if (nextSource !== this.last.source || this.repositoryObserver.getActiveModel() !== this.last.repository || this.excludedUsers !== this.last.excludedUsers) {
      changed = true;
      this.allUsers.clear();
    }

    for (const author of users) {
      if (!this.allUsers.has(author.getEmail())) {
        changed = true;
      }

      this.allUsers.set(author.getEmail(), author);
    }

    if (changed) {
      this.finalize();
    }

    this.last.source = nextSource;
    this.last.repository = this.repositoryObserver.getActiveModel();
    this.last.excludedUsers = this.excludedUsers;
  }

  finalize() {
    // TODO: [ku 3/2018] consider sorting based on most recent authors or commit frequency
    const users = [];

    for (const author of this.allUsers.values()) {
      if (author.matches(this.committer)) {
        continue;
      }

      if (author.isNoReply()) {
        continue;
      }

      if (this.excludedUsers.has(author.getEmail())) {
        continue;
      }

      users.push(author);
    }

    users.sort(_author["default"].compare);
    this.users = users;
    this.didUpdate();
  }

  setRepository(repository) {
    this.repositoryObserver.setActiveModel(repository);
  }

  setLoginModel(login) {
    this.loginObserver.setActiveModel(login);
  }

  setCommitter(committer) {
    const changed = !this.committer.matches(committer);
    this.committer = committer;

    if (changed) {
      this.finalize();
    }
  }

  didUpdate() {
    this.emitter.emit('did-update', this.getUsers());
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  getUsers() {
    return this.users;
  }

}

exports["default"] = UserStore;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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