"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _crypto = _interopRequireDefault(require("crypto"));

var _eventKit = require("event-kit");

var _yubikiri = _interopRequireDefault(require("yubikiri"));

var _whatsMyLine = require("whats-my-line");

var _file = _interopRequireDefault(require("../models/patch/file"));

var _observeModel = _interopRequireDefault(require("../views/observe-model"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CommentPositioningContainer extends _react["default"].Component {
  constructor(props) {
    super(props);

    _defineProperty(this, "fetchData", (localRepository, commentPaths, prCommitSha) => {
      const promises = {};

      for (const commentPath of commentPaths) {
        promises[commentPath] = localRepository.getDiffsForFilePath(commentPath, prCommitSha)["catch"](() => []);
      }

      return (0, _yubikiri["default"])(promises);
    });

    this.state = {
      translationsByFile: new Map()
    };
    this.subs = new _eventKit.CompositeDisposable();
  }

  static getDerivedStateFromProps(props, state) {
    const prevPaths = new Set(state.translationsByFile.keys());
    let changed = false;

    for (const thread of props.commentThreads) {
      const relPath = thread.comments[0].path;
      const commentPath = (0, _helpers.toNativePathSep)(relPath);
      let existing = state.translationsByFile.get(commentPath);

      if (!existing) {
        existing = new FileTranslation(relPath);
        state.translationsByFile.set(commentPath, existing);
        changed = true;
      }

      existing.addCommentThread(thread);
      prevPaths["delete"](commentPath);
    }

    for (const oldPath of prevPaths) {
      state.translationsByFile.deleted(oldPath);
      changed = true;
    }

    if (changed) {
      return {
        translationsByFile: state.translationsByFile
      };
    } else {
      return null;
    }
  }

  componentWillUnmount() {
    this.subs.dispose();
  }

  render() {
    const commentPaths = [...this.state.translationsByFile.keys()];
    return _react["default"].createElement(_observeModel["default"], {
      model: this.props.localRepository,
      fetchData: this.fetchData,
      fetchParams: [commentPaths, this.props.prCommitSha]
    }, diffsByPath => {
      if (diffsByPath === null) {
        return this.props.children(null);
      }

      for (const commentPath of commentPaths) {
        this.state.translationsByFile.get(commentPath).updateIfNecessary({
          multiFilePatch: this.props.multiFilePatch,
          diffs: diffsByPath[commentPath] || [],
          diffPositionFn: this.props.diffPositionToFilePosition,
          translatePositionFn: this.props.translateLinesGivenDiff
        });
      }

      return this.props.children(this.state.translationsByFile);
    });
  }

}

exports["default"] = CommentPositioningContainer;

_defineProperty(CommentPositioningContainer, "propTypes", {
  localRepository: _propTypes["default"].object.isRequired,
  multiFilePatch: _propTypes["default"].object.isRequired,
  commentThreads: _propTypes["default"].arrayOf(_propTypes["default"].shape({
    comments: _propTypes["default"].arrayOf(_propTypes["default"].shape({
      position: _propTypes["default"].number,
      path: _propTypes["default"].string.isRequired
    })).isRequired
  })),
  prCommitSha: _propTypes["default"].string.isRequired,
  children: _propTypes["default"].func.isRequired,
  // For unit test injection
  translateLinesGivenDiff: _propTypes["default"].func,
  diffPositionToFilePosition: _propTypes["default"].func
});

_defineProperty(CommentPositioningContainer, "defaultProps", {
  translateLinesGivenDiff: _whatsMyLine.translateLinesGivenDiff,
  diffPositionToFilePosition: _whatsMyLine.diffPositionToFilePosition,
  didTranslate:
  /* istanbul ignore next */
  () => {}
});

class FileTranslation {
  constructor(relPath) {
    this.relPath = relPath;
    this.nativeRelPath = (0, _helpers.toNativePathSep)(relPath);
    this.rawPositions = new Set();
    this.diffToFilePosition = new Map();
    this.removed = false;
    this.fileTranslations = null;
    this.digest = null;
    this.last = {
      multiFilePatch: null,
      diffs: null
    };
  }

  addCommentThread(thread) {
    this.rawPositions.add(thread.comments[0].position);
  }

  updateIfNecessary({
    multiFilePatch,
    diffs,
    diffPositionFn,
    translatePositionFn
  }) {
    if (this.last.multiFilePatch === multiFilePatch && this.last.diffs === diffs) {
      return false;
    }

    this.last.multiFilePatch = multiFilePatch;
    this.last.diffs = diffs;
    return this.update({
      multiFilePatch,
      diffs,
      diffPositionFn,
      translatePositionFn
    });
  }

  update({
    multiFilePatch,
    diffs,
    diffPositionFn,
    translatePositionFn
  }) {
    const filePatch = multiFilePatch.getPatchForPath(this.nativeRelPath); // Comment on a file that used to exist in a PR but no longer does. Skip silently.

    if (!filePatch) {
      this.diffToFilePosition = new Map();
      this.removed = false;
      this.fileTranslations = null;
      return;
    } // This comment was left on a file that was too large to parse.


    if (!filePatch.getRenderStatus().isVisible()) {
      this.diffToFilePosition = new Map();
      this.removed = true;
      this.fileTranslations = null;
      return;
    }

    this.diffToFilePosition = diffPositionFn(this.rawPositions, filePatch.getRawContentPatch());
    this.removed = false;
    let contentChangeDiff;

    if (diffs.length === 1) {
      contentChangeDiff = diffs[0];
    } else if (diffs.length === 2) {
      const [diff1, diff2] = diffs;

      if (diff1.oldMode === _file["default"].modes.SYMLINK || diff1.newMode === _file["default"].modes.SYMLINK) {
        contentChangeDiff = diff2;
      } else {
        contentChangeDiff = diff1;
      }
    }

    if (contentChangeDiff) {
      const filePositions = [...this.diffToFilePosition.values()];
      this.fileTranslations = translatePositionFn(filePositions, contentChangeDiff);

      const hash = _crypto["default"].createHash('sha256');

      hash.update(JSON.stringify(Array.from(this.fileTranslations.entries())));
      this.digest = hash.digest('hex');
    } else {
      this.fileTranslations = null;
      this.digest = null;
    }
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImNvbW1lbnQtcG9zaXRpb25pbmctY29udGFpbmVyLmpzIl0sIm5hbWVzIjpbIkNvbW1lbnRQb3NpdGlvbmluZ0NvbnRhaW5lciIsIlJlYWN0IiwiQ29tcG9uZW50IiwiY29uc3RydWN0b3IiLCJwcm9wcyIsImxvY2FsUmVwb3NpdG9yeSIsImNvbW1lbnRQYXRocyIsInByQ29tbWl0U2hhIiwicHJvbWlzZXMiLCJjb21tZW50UGF0aCIsImdldERpZmZzRm9yRmlsZVBhdGgiLCJzdGF0ZSIsInRyYW5zbGF0aW9uc0J5RmlsZSIsIk1hcCIsInN1YnMiLCJDb21wb3NpdGVEaXNwb3NhYmxlIiwiZ2V0RGVyaXZlZFN0YXRlRnJvbVByb3BzIiwicHJldlBhdGhzIiwiU2V0Iiwia2V5cyIsImNoYW5nZWQiLCJ0aHJlYWQiLCJjb21tZW50VGhyZWFkcyIsInJlbFBhdGgiLCJjb21tZW50cyIsInBhdGgiLCJleGlzdGluZyIsImdldCIsIkZpbGVUcmFuc2xhdGlvbiIsInNldCIsImFkZENvbW1lbnRUaHJlYWQiLCJvbGRQYXRoIiwiZGVsZXRlZCIsImNvbXBvbmVudFdpbGxVbm1vdW50IiwiZGlzcG9zZSIsInJlbmRlciIsImZldGNoRGF0YSIsImRpZmZzQnlQYXRoIiwiY2hpbGRyZW4iLCJ1cGRhdGVJZk5lY2Vzc2FyeSIsIm11bHRpRmlsZVBhdGNoIiwiZGlmZnMiLCJkaWZmUG9zaXRpb25GbiIsImRpZmZQb3NpdGlvblRvRmlsZVBvc2l0aW9uIiwidHJhbnNsYXRlUG9zaXRpb25GbiIsInRyYW5zbGF0ZUxpbmVzR2l2ZW5EaWZmIiwiUHJvcFR5cGVzIiwib2JqZWN0IiwiaXNSZXF1aXJlZCIsImFycmF5T2YiLCJzaGFwZSIsInBvc2l0aW9uIiwibnVtYmVyIiwic3RyaW5nIiwiZnVuYyIsImRpZFRyYW5zbGF0ZSIsIm5hdGl2ZVJlbFBhdGgiLCJyYXdQb3NpdGlvbnMiLCJkaWZmVG9GaWxlUG9zaXRpb24iLCJyZW1vdmVkIiwiZmlsZVRyYW5zbGF0aW9ucyIsImRpZ2VzdCIsImxhc3QiLCJhZGQiLCJ1cGRhdGUiLCJmaWxlUGF0Y2giLCJnZXRQYXRjaEZvclBhdGgiLCJnZXRSZW5kZXJTdGF0dXMiLCJpc1Zpc2libGUiLCJnZXRSYXdDb250ZW50UGF0Y2giLCJjb250ZW50Q2hhbmdlRGlmZiIsImxlbmd0aCIsImRpZmYxIiwiZGlmZjIiLCJvbGRNb2RlIiwiRmlsZSIsIm1vZGVzIiwiU1lNTElOSyIsIm5ld01vZGUiLCJmaWxlUG9zaXRpb25zIiwidmFsdWVzIiwiaGFzaCIsImNyeXB0byIsImNyZWF0ZUhhc2giLCJKU09OIiwic3RyaW5naWZ5IiwiQXJyYXkiLCJmcm9tIiwiZW50cmllcyJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUVBOztBQUNBOztBQUNBOzs7Ozs7QUFFZSxNQUFNQSwyQkFBTixTQUEwQ0Msa0JBQU1DLFNBQWhELENBQTBEO0FBd0J2RUMsRUFBQUEsV0FBVyxDQUFDQyxLQUFELEVBQVE7QUFDakIsVUFBTUEsS0FBTjs7QUFEaUIsdUNBd0VQLENBQUNDLGVBQUQsRUFBa0JDLFlBQWxCLEVBQWdDQyxXQUFoQyxLQUFnRDtBQUMxRCxZQUFNQyxRQUFRLEdBQUcsRUFBakI7O0FBQ0EsV0FBSyxNQUFNQyxXQUFYLElBQTBCSCxZQUExQixFQUF3QztBQUN0Q0UsUUFBQUEsUUFBUSxDQUFDQyxXQUFELENBQVIsR0FBd0JKLGVBQWUsQ0FBQ0ssbUJBQWhCLENBQW9DRCxXQUFwQyxFQUFpREYsV0FBakQsV0FBb0UsTUFBTSxFQUExRSxDQUF4QjtBQUNEOztBQUNELGFBQU8sMEJBQVNDLFFBQVQsQ0FBUDtBQUNELEtBOUVrQjs7QUFHakIsU0FBS0csS0FBTCxHQUFhO0FBQUNDLE1BQUFBLGtCQUFrQixFQUFFLElBQUlDLEdBQUo7QUFBckIsS0FBYjtBQUNBLFNBQUtDLElBQUwsR0FBWSxJQUFJQyw2QkFBSixFQUFaO0FBQ0Q7O0FBRUQsU0FBT0Msd0JBQVAsQ0FBZ0NaLEtBQWhDLEVBQXVDTyxLQUF2QyxFQUE4QztBQUM1QyxVQUFNTSxTQUFTLEdBQUcsSUFBSUMsR0FBSixDQUFRUCxLQUFLLENBQUNDLGtCQUFOLENBQXlCTyxJQUF6QixFQUFSLENBQWxCO0FBQ0EsUUFBSUMsT0FBTyxHQUFHLEtBQWQ7O0FBRUEsU0FBSyxNQUFNQyxNQUFYLElBQXFCakIsS0FBSyxDQUFDa0IsY0FBM0IsRUFBMkM7QUFDekMsWUFBTUMsT0FBTyxHQUFHRixNQUFNLENBQUNHLFFBQVAsQ0FBZ0IsQ0FBaEIsRUFBbUJDLElBQW5DO0FBQ0EsWUFBTWhCLFdBQVcsR0FBRyw4QkFBZ0JjLE9BQWhCLENBQXBCO0FBRUEsVUFBSUcsUUFBUSxHQUFHZixLQUFLLENBQUNDLGtCQUFOLENBQXlCZSxHQUF6QixDQUE2QmxCLFdBQTdCLENBQWY7O0FBQ0EsVUFBSSxDQUFDaUIsUUFBTCxFQUFlO0FBQ2JBLFFBQUFBLFFBQVEsR0FBRyxJQUFJRSxlQUFKLENBQW9CTCxPQUFwQixDQUFYO0FBQ0FaLFFBQUFBLEtBQUssQ0FBQ0Msa0JBQU4sQ0FBeUJpQixHQUF6QixDQUE2QnBCLFdBQTdCLEVBQTBDaUIsUUFBMUM7QUFDQU4sUUFBQUEsT0FBTyxHQUFHLElBQVY7QUFDRDs7QUFDRE0sTUFBQUEsUUFBUSxDQUFDSSxnQkFBVCxDQUEwQlQsTUFBMUI7QUFFQUosTUFBQUEsU0FBUyxVQUFULENBQWlCUixXQUFqQjtBQUNEOztBQUVELFNBQUssTUFBTXNCLE9BQVgsSUFBc0JkLFNBQXRCLEVBQWlDO0FBQy9CTixNQUFBQSxLQUFLLENBQUNDLGtCQUFOLENBQXlCb0IsT0FBekIsQ0FBaUNELE9BQWpDO0FBQ0FYLE1BQUFBLE9BQU8sR0FBRyxJQUFWO0FBQ0Q7O0FBRUQsUUFBSUEsT0FBSixFQUFhO0FBQ1gsYUFBTztBQUFDUixRQUFBQSxrQkFBa0IsRUFBRUQsS0FBSyxDQUFDQztBQUEzQixPQUFQO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsYUFBTyxJQUFQO0FBQ0Q7QUFDRjs7QUFFRHFCLEVBQUFBLG9CQUFvQixHQUFHO0FBQ3JCLFNBQUtuQixJQUFMLENBQVVvQixPQUFWO0FBQ0Q7O0FBRURDLEVBQUFBLE1BQU0sR0FBRztBQUNQLFVBQU03QixZQUFZLEdBQUcsQ0FBQyxHQUFHLEtBQUtLLEtBQUwsQ0FBV0Msa0JBQVgsQ0FBOEJPLElBQTlCLEVBQUosQ0FBckI7QUFFQSxXQUNFLGdDQUFDLHdCQUFEO0FBQ0UsTUFBQSxLQUFLLEVBQUUsS0FBS2YsS0FBTCxDQUFXQyxlQURwQjtBQUVFLE1BQUEsU0FBUyxFQUFFLEtBQUsrQixTQUZsQjtBQUdFLE1BQUEsV0FBVyxFQUFFLENBQUM5QixZQUFELEVBQWUsS0FBS0YsS0FBTCxDQUFXRyxXQUExQjtBQUhmLE9BS0c4QixXQUFXLElBQUk7QUFDZCxVQUFJQSxXQUFXLEtBQUssSUFBcEIsRUFBMEI7QUFDeEIsZUFBTyxLQUFLakMsS0FBTCxDQUFXa0MsUUFBWCxDQUFvQixJQUFwQixDQUFQO0FBQ0Q7O0FBRUQsV0FBSyxNQUFNN0IsV0FBWCxJQUEwQkgsWUFBMUIsRUFBd0M7QUFDdEMsYUFBS0ssS0FBTCxDQUFXQyxrQkFBWCxDQUE4QmUsR0FBOUIsQ0FBa0NsQixXQUFsQyxFQUErQzhCLGlCQUEvQyxDQUFpRTtBQUMvREMsVUFBQUEsY0FBYyxFQUFFLEtBQUtwQyxLQUFMLENBQVdvQyxjQURvQztBQUUvREMsVUFBQUEsS0FBSyxFQUFFSixXQUFXLENBQUM1QixXQUFELENBQVgsSUFBNEIsRUFGNEI7QUFHL0RpQyxVQUFBQSxjQUFjLEVBQUUsS0FBS3RDLEtBQUwsQ0FBV3VDLDBCQUhvQztBQUkvREMsVUFBQUEsbUJBQW1CLEVBQUUsS0FBS3hDLEtBQUwsQ0FBV3lDO0FBSitCLFNBQWpFO0FBTUQ7O0FBRUQsYUFBTyxLQUFLekMsS0FBTCxDQUFXa0MsUUFBWCxDQUFvQixLQUFLM0IsS0FBTCxDQUFXQyxrQkFBL0IsQ0FBUDtBQUNELEtBcEJILENBREY7QUF5QkQ7O0FBOUZzRTs7OztnQkFBcERaLDJCLGVBQ0E7QUFDakJLLEVBQUFBLGVBQWUsRUFBRXlDLHNCQUFVQyxNQUFWLENBQWlCQyxVQURqQjtBQUVqQlIsRUFBQUEsY0FBYyxFQUFFTSxzQkFBVUMsTUFBVixDQUFpQkMsVUFGaEI7QUFHakIxQixFQUFBQSxjQUFjLEVBQUV3QixzQkFBVUcsT0FBVixDQUFrQkgsc0JBQVVJLEtBQVYsQ0FBZ0I7QUFDaEQxQixJQUFBQSxRQUFRLEVBQUVzQixzQkFBVUcsT0FBVixDQUFrQkgsc0JBQVVJLEtBQVYsQ0FBZ0I7QUFDMUNDLE1BQUFBLFFBQVEsRUFBRUwsc0JBQVVNLE1BRHNCO0FBRTFDM0IsTUFBQUEsSUFBSSxFQUFFcUIsc0JBQVVPLE1BQVYsQ0FBaUJMO0FBRm1CLEtBQWhCLENBQWxCLEVBR05BO0FBSjRDLEdBQWhCLENBQWxCLENBSEM7QUFTakJ6QyxFQUFBQSxXQUFXLEVBQUV1QyxzQkFBVU8sTUFBVixDQUFpQkwsVUFUYjtBQVVqQlYsRUFBQUEsUUFBUSxFQUFFUSxzQkFBVVEsSUFBVixDQUFlTixVQVZSO0FBWWpCO0FBQ0FILEVBQUFBLHVCQUF1QixFQUFFQyxzQkFBVVEsSUFibEI7QUFjakJYLEVBQUFBLDBCQUEwQixFQUFFRyxzQkFBVVE7QUFkckIsQzs7Z0JBREF0RCwyQixrQkFrQkc7QUFDcEI2QyxFQUFBQSx1QkFBdUIsRUFBdkJBLG9DQURvQjtBQUVwQkYsRUFBQUEsMEJBQTBCLEVBQTFCQSx1Q0FGb0I7QUFHcEJZLEVBQUFBLFlBQVk7QUFBRTtBQUEyQixRQUFNLENBQUU7QUFIN0IsQzs7QUF1RnhCLE1BQU0zQixlQUFOLENBQXNCO0FBQ3BCekIsRUFBQUEsV0FBVyxDQUFDb0IsT0FBRCxFQUFVO0FBQ25CLFNBQUtBLE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtpQyxhQUFMLEdBQXFCLDhCQUFnQmpDLE9BQWhCLENBQXJCO0FBRUEsU0FBS2tDLFlBQUwsR0FBb0IsSUFBSXZDLEdBQUosRUFBcEI7QUFDQSxTQUFLd0Msa0JBQUwsR0FBMEIsSUFBSTdDLEdBQUosRUFBMUI7QUFDQSxTQUFLOEMsT0FBTCxHQUFlLEtBQWY7QUFDQSxTQUFLQyxnQkFBTCxHQUF3QixJQUF4QjtBQUNBLFNBQUtDLE1BQUwsR0FBYyxJQUFkO0FBRUEsU0FBS0MsSUFBTCxHQUFZO0FBQUN0QixNQUFBQSxjQUFjLEVBQUUsSUFBakI7QUFBdUJDLE1BQUFBLEtBQUssRUFBRTtBQUE5QixLQUFaO0FBQ0Q7O0FBRURYLEVBQUFBLGdCQUFnQixDQUFDVCxNQUFELEVBQVM7QUFDdkIsU0FBS29DLFlBQUwsQ0FBa0JNLEdBQWxCLENBQXNCMUMsTUFBTSxDQUFDRyxRQUFQLENBQWdCLENBQWhCLEVBQW1CMkIsUUFBekM7QUFDRDs7QUFFRFosRUFBQUEsaUJBQWlCLENBQUM7QUFBQ0MsSUFBQUEsY0FBRDtBQUFpQkMsSUFBQUEsS0FBakI7QUFBd0JDLElBQUFBLGNBQXhCO0FBQXdDRSxJQUFBQTtBQUF4QyxHQUFELEVBQStEO0FBQzlFLFFBQ0UsS0FBS2tCLElBQUwsQ0FBVXRCLGNBQVYsS0FBNkJBLGNBQTdCLElBQ0EsS0FBS3NCLElBQUwsQ0FBVXJCLEtBQVYsS0FBb0JBLEtBRnRCLEVBR0U7QUFDQSxhQUFPLEtBQVA7QUFDRDs7QUFFRCxTQUFLcUIsSUFBTCxDQUFVdEIsY0FBVixHQUEyQkEsY0FBM0I7QUFDQSxTQUFLc0IsSUFBTCxDQUFVckIsS0FBVixHQUFrQkEsS0FBbEI7QUFFQSxXQUFPLEtBQUt1QixNQUFMLENBQVk7QUFBQ3hCLE1BQUFBLGNBQUQ7QUFBaUJDLE1BQUFBLEtBQWpCO0FBQXdCQyxNQUFBQSxjQUF4QjtBQUF3Q0UsTUFBQUE7QUFBeEMsS0FBWixDQUFQO0FBQ0Q7O0FBRURvQixFQUFBQSxNQUFNLENBQUM7QUFBQ3hCLElBQUFBLGNBQUQ7QUFBaUJDLElBQUFBLEtBQWpCO0FBQXdCQyxJQUFBQSxjQUF4QjtBQUF3Q0UsSUFBQUE7QUFBeEMsR0FBRCxFQUErRDtBQUNuRSxVQUFNcUIsU0FBUyxHQUFHekIsY0FBYyxDQUFDMEIsZUFBZixDQUErQixLQUFLVixhQUFwQyxDQUFsQixDQURtRSxDQUVuRTs7QUFDQSxRQUFJLENBQUNTLFNBQUwsRUFBZ0I7QUFDZCxXQUFLUCxrQkFBTCxHQUEwQixJQUFJN0MsR0FBSixFQUExQjtBQUNBLFdBQUs4QyxPQUFMLEdBQWUsS0FBZjtBQUNBLFdBQUtDLGdCQUFMLEdBQXdCLElBQXhCO0FBRUE7QUFDRCxLQVRrRSxDQVduRTs7O0FBQ0EsUUFBSSxDQUFDSyxTQUFTLENBQUNFLGVBQVYsR0FBNEJDLFNBQTVCLEVBQUwsRUFBOEM7QUFDNUMsV0FBS1Ysa0JBQUwsR0FBMEIsSUFBSTdDLEdBQUosRUFBMUI7QUFDQSxXQUFLOEMsT0FBTCxHQUFlLElBQWY7QUFDQSxXQUFLQyxnQkFBTCxHQUF3QixJQUF4QjtBQUVBO0FBQ0Q7O0FBRUQsU0FBS0Ysa0JBQUwsR0FBMEJoQixjQUFjLENBQUMsS0FBS2UsWUFBTixFQUFvQlEsU0FBUyxDQUFDSSxrQkFBVixFQUFwQixDQUF4QztBQUNBLFNBQUtWLE9BQUwsR0FBZSxLQUFmO0FBRUEsUUFBSVcsaUJBQUo7O0FBQ0EsUUFBSTdCLEtBQUssQ0FBQzhCLE1BQU4sS0FBaUIsQ0FBckIsRUFBd0I7QUFDdEJELE1BQUFBLGlCQUFpQixHQUFHN0IsS0FBSyxDQUFDLENBQUQsQ0FBekI7QUFDRCxLQUZELE1BRU8sSUFBSUEsS0FBSyxDQUFDOEIsTUFBTixLQUFpQixDQUFyQixFQUF3QjtBQUM3QixZQUFNLENBQUNDLEtBQUQsRUFBUUMsS0FBUixJQUFpQmhDLEtBQXZCOztBQUNBLFVBQUkrQixLQUFLLENBQUNFLE9BQU4sS0FBa0JDLGlCQUFLQyxLQUFMLENBQVdDLE9BQTdCLElBQXdDTCxLQUFLLENBQUNNLE9BQU4sS0FBa0JILGlCQUFLQyxLQUFMLENBQVdDLE9BQXpFLEVBQWtGO0FBQ2hGUCxRQUFBQSxpQkFBaUIsR0FBR0csS0FBcEI7QUFDRCxPQUZELE1BRU87QUFDTEgsUUFBQUEsaUJBQWlCLEdBQUdFLEtBQXBCO0FBQ0Q7QUFDRjs7QUFFRCxRQUFJRixpQkFBSixFQUF1QjtBQUNyQixZQUFNUyxhQUFhLEdBQUcsQ0FBQyxHQUFHLEtBQUtyQixrQkFBTCxDQUF3QnNCLE1BQXhCLEVBQUosQ0FBdEI7QUFDQSxXQUFLcEIsZ0JBQUwsR0FBd0JoQixtQkFBbUIsQ0FBQ21DLGFBQUQsRUFBZ0JULGlCQUFoQixDQUEzQzs7QUFFQSxZQUFNVyxJQUFJLEdBQUdDLG1CQUFPQyxVQUFQLENBQWtCLFFBQWxCLENBQWI7O0FBQ0FGLE1BQUFBLElBQUksQ0FBQ2pCLE1BQUwsQ0FBWW9CLElBQUksQ0FBQ0MsU0FBTCxDQUFlQyxLQUFLLENBQUNDLElBQU4sQ0FBVyxLQUFLM0IsZ0JBQUwsQ0FBc0I0QixPQUF0QixFQUFYLENBQWYsQ0FBWjtBQUNBLFdBQUszQixNQUFMLEdBQWNvQixJQUFJLENBQUNwQixNQUFMLENBQVksS0FBWixDQUFkO0FBQ0QsS0FQRCxNQU9PO0FBQ0wsV0FBS0QsZ0JBQUwsR0FBd0IsSUFBeEI7QUFDQSxXQUFLQyxNQUFMLEdBQWMsSUFBZDtBQUNEO0FBQ0Y7O0FBOUVtQiIsInNvdXJjZVJvb3QiOiIvYnVpbGQvYXRvbS9zcmMvYXRvbS0xLjQwLjEvb3V0L2FwcC9ub2RlX21vZHVsZXMvZ2l0aHViIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IFJlYWN0IGZyb20gJ3JlYWN0JztcbmltcG9ydCBQcm9wVHlwZXMgZnJvbSAncHJvcC10eXBlcyc7XG5pbXBvcnQgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQge0NvbXBvc2l0ZURpc3Bvc2FibGV9IGZyb20gJ2V2ZW50LWtpdCc7XG5pbXBvcnQgeXViaWtpcmkgZnJvbSAneXViaWtpcmknO1xuaW1wb3J0IHt0cmFuc2xhdGVMaW5lc0dpdmVuRGlmZiwgZGlmZlBvc2l0aW9uVG9GaWxlUG9zaXRpb259IGZyb20gJ3doYXRzLW15LWxpbmUnO1xuXG5pbXBvcnQgRmlsZSBmcm9tICcuLi9tb2RlbHMvcGF0Y2gvZmlsZSc7XG5pbXBvcnQgT2JzZXJ2ZU1vZGVsIGZyb20gJy4uL3ZpZXdzL29ic2VydmUtbW9kZWwnO1xuaW1wb3J0IHt0b05hdGl2ZVBhdGhTZXB9IGZyb20gJy4uL2hlbHBlcnMnO1xuXG5leHBvcnQgZGVmYXVsdCBjbGFzcyBDb21tZW50UG9zaXRpb25pbmdDb250YWluZXIgZXh0ZW5kcyBSZWFjdC5Db21wb25lbnQge1xuICBzdGF0aWMgcHJvcFR5cGVzID0ge1xuICAgIGxvY2FsUmVwb3NpdG9yeTogUHJvcFR5cGVzLm9iamVjdC5pc1JlcXVpcmVkLFxuICAgIG11bHRpRmlsZVBhdGNoOiBQcm9wVHlwZXMub2JqZWN0LmlzUmVxdWlyZWQsXG4gICAgY29tbWVudFRocmVhZHM6IFByb3BUeXBlcy5hcnJheU9mKFByb3BUeXBlcy5zaGFwZSh7XG4gICAgICBjb21tZW50czogUHJvcFR5cGVzLmFycmF5T2YoUHJvcFR5cGVzLnNoYXBlKHtcbiAgICAgICAgcG9zaXRpb246IFByb3BUeXBlcy5udW1iZXIsXG4gICAgICAgIHBhdGg6IFByb3BUeXBlcy5zdHJpbmcuaXNSZXF1aXJlZCxcbiAgICAgIH0pKS5pc1JlcXVpcmVkLFxuICAgIH0pKSxcbiAgICBwckNvbW1pdFNoYTogUHJvcFR5cGVzLnN0cmluZy5pc1JlcXVpcmVkLFxuICAgIGNoaWxkcmVuOiBQcm9wVHlwZXMuZnVuYy5pc1JlcXVpcmVkLFxuXG4gICAgLy8gRm9yIHVuaXQgdGVzdCBpbmplY3Rpb25cbiAgICB0cmFuc2xhdGVMaW5lc0dpdmVuRGlmZjogUHJvcFR5cGVzLmZ1bmMsXG4gICAgZGlmZlBvc2l0aW9uVG9GaWxlUG9zaXRpb246IFByb3BUeXBlcy5mdW5jLFxuICB9XG5cbiAgc3RhdGljIGRlZmF1bHRQcm9wcyA9IHtcbiAgICB0cmFuc2xhdGVMaW5lc0dpdmVuRGlmZixcbiAgICBkaWZmUG9zaXRpb25Ub0ZpbGVQb3NpdGlvbixcbiAgICBkaWRUcmFuc2xhdGU6IC8qIGlzdGFuYnVsIGlnbm9yZSBuZXh0ICovICgpID0+IHt9LFxuICB9XG5cbiAgY29uc3RydWN0b3IocHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG5cbiAgICB0aGlzLnN0YXRlID0ge3RyYW5zbGF0aW9uc0J5RmlsZTogbmV3IE1hcCgpfTtcbiAgICB0aGlzLnN1YnMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZSgpO1xuICB9XG5cbiAgc3RhdGljIGdldERlcml2ZWRTdGF0ZUZyb21Qcm9wcyhwcm9wcywgc3RhdGUpIHtcbiAgICBjb25zdCBwcmV2UGF0aHMgPSBuZXcgU2V0KHN0YXRlLnRyYW5zbGF0aW9uc0J5RmlsZS5rZXlzKCkpO1xuICAgIGxldCBjaGFuZ2VkID0gZmFsc2U7XG5cbiAgICBmb3IgKGNvbnN0IHRocmVhZCBvZiBwcm9wcy5jb21tZW50VGhyZWFkcykge1xuICAgICAgY29uc3QgcmVsUGF0aCA9IHRocmVhZC5jb21tZW50c1swXS5wYXRoO1xuICAgICAgY29uc3QgY29tbWVudFBhdGggPSB0b05hdGl2ZVBhdGhTZXAocmVsUGF0aCk7XG5cbiAgICAgIGxldCBleGlzdGluZyA9IHN0YXRlLnRyYW5zbGF0aW9uc0J5RmlsZS5nZXQoY29tbWVudFBhdGgpO1xuICAgICAgaWYgKCFleGlzdGluZykge1xuICAgICAgICBleGlzdGluZyA9IG5ldyBGaWxlVHJhbnNsYXRpb24ocmVsUGF0aCk7XG4gICAgICAgIHN0YXRlLnRyYW5zbGF0aW9uc0J5RmlsZS5zZXQoY29tbWVudFBhdGgsIGV4aXN0aW5nKTtcbiAgICAgICAgY2hhbmdlZCA9IHRydWU7XG4gICAgICB9XG4gICAgICBleGlzdGluZy5hZGRDb21tZW50VGhyZWFkKHRocmVhZCk7XG5cbiAgICAgIHByZXZQYXRocy5kZWxldGUoY29tbWVudFBhdGgpO1xuICAgIH1cblxuICAgIGZvciAoY29uc3Qgb2xkUGF0aCBvZiBwcmV2UGF0aHMpIHtcbiAgICAgIHN0YXRlLnRyYW5zbGF0aW9uc0J5RmlsZS5kZWxldGVkKG9sZFBhdGgpO1xuICAgICAgY2hhbmdlZCA9IHRydWU7XG4gICAgfVxuXG4gICAgaWYgKGNoYW5nZWQpIHtcbiAgICAgIHJldHVybiB7dHJhbnNsYXRpb25zQnlGaWxlOiBzdGF0ZS50cmFuc2xhdGlvbnNCeUZpbGV9O1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gbnVsbDtcbiAgICB9XG4gIH1cblxuICBjb21wb25lbnRXaWxsVW5tb3VudCgpIHtcbiAgICB0aGlzLnN1YnMuZGlzcG9zZSgpO1xuICB9XG5cbiAgcmVuZGVyKCkge1xuICAgIGNvbnN0IGNvbW1lbnRQYXRocyA9IFsuLi50aGlzLnN0YXRlLnRyYW5zbGF0aW9uc0J5RmlsZS5rZXlzKCldO1xuXG4gICAgcmV0dXJuIChcbiAgICAgIDxPYnNlcnZlTW9kZWxcbiAgICAgICAgbW9kZWw9e3RoaXMucHJvcHMubG9jYWxSZXBvc2l0b3J5fVxuICAgICAgICBmZXRjaERhdGE9e3RoaXMuZmV0Y2hEYXRhfVxuICAgICAgICBmZXRjaFBhcmFtcz17W2NvbW1lbnRQYXRocywgdGhpcy5wcm9wcy5wckNvbW1pdFNoYV19PlxuXG4gICAgICAgIHtkaWZmc0J5UGF0aCA9PiB7XG4gICAgICAgICAgaWYgKGRpZmZzQnlQYXRoID09PSBudWxsKSB7XG4gICAgICAgICAgICByZXR1cm4gdGhpcy5wcm9wcy5jaGlsZHJlbihudWxsKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBmb3IgKGNvbnN0IGNvbW1lbnRQYXRoIG9mIGNvbW1lbnRQYXRocykge1xuICAgICAgICAgICAgdGhpcy5zdGF0ZS50cmFuc2xhdGlvbnNCeUZpbGUuZ2V0KGNvbW1lbnRQYXRoKS51cGRhdGVJZk5lY2Vzc2FyeSh7XG4gICAgICAgICAgICAgIG11bHRpRmlsZVBhdGNoOiB0aGlzLnByb3BzLm11bHRpRmlsZVBhdGNoLFxuICAgICAgICAgICAgICBkaWZmczogZGlmZnNCeVBhdGhbY29tbWVudFBhdGhdIHx8IFtdLFxuICAgICAgICAgICAgICBkaWZmUG9zaXRpb25GbjogdGhpcy5wcm9wcy5kaWZmUG9zaXRpb25Ub0ZpbGVQb3NpdGlvbixcbiAgICAgICAgICAgICAgdHJhbnNsYXRlUG9zaXRpb25GbjogdGhpcy5wcm9wcy50cmFuc2xhdGVMaW5lc0dpdmVuRGlmZixcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHJldHVybiB0aGlzLnByb3BzLmNoaWxkcmVuKHRoaXMuc3RhdGUudHJhbnNsYXRpb25zQnlGaWxlKTtcbiAgICAgICAgfX1cblxuICAgICAgPC9PYnNlcnZlTW9kZWw+XG4gICAgKTtcbiAgfVxuXG4gIGZldGNoRGF0YSA9IChsb2NhbFJlcG9zaXRvcnksIGNvbW1lbnRQYXRocywgcHJDb21taXRTaGEpID0+IHtcbiAgICBjb25zdCBwcm9taXNlcyA9IHt9O1xuICAgIGZvciAoY29uc3QgY29tbWVudFBhdGggb2YgY29tbWVudFBhdGhzKSB7XG4gICAgICBwcm9taXNlc1tjb21tZW50UGF0aF0gPSBsb2NhbFJlcG9zaXRvcnkuZ2V0RGlmZnNGb3JGaWxlUGF0aChjb21tZW50UGF0aCwgcHJDb21taXRTaGEpLmNhdGNoKCgpID0+IFtdKTtcbiAgICB9XG4gICAgcmV0dXJuIHl1YmlraXJpKHByb21pc2VzKTtcbiAgfVxufVxuXG5jbGFzcyBGaWxlVHJhbnNsYXRpb24ge1xuICBjb25zdHJ1Y3RvcihyZWxQYXRoKSB7XG4gICAgdGhpcy5yZWxQYXRoID0gcmVsUGF0aDtcbiAgICB0aGlzLm5hdGl2ZVJlbFBhdGggPSB0b05hdGl2ZVBhdGhTZXAocmVsUGF0aCk7XG5cbiAgICB0aGlzLnJhd1Bvc2l0aW9ucyA9IG5ldyBTZXQoKTtcbiAgICB0aGlzLmRpZmZUb0ZpbGVQb3NpdGlvbiA9IG5ldyBNYXAoKTtcbiAgICB0aGlzLnJlbW92ZWQgPSBmYWxzZTtcbiAgICB0aGlzLmZpbGVUcmFuc2xhdGlvbnMgPSBudWxsO1xuICAgIHRoaXMuZGlnZXN0ID0gbnVsbDtcblxuICAgIHRoaXMubGFzdCA9IHttdWx0aUZpbGVQYXRjaDogbnVsbCwgZGlmZnM6IG51bGx9O1xuICB9XG5cbiAgYWRkQ29tbWVudFRocmVhZCh0aHJlYWQpIHtcbiAgICB0aGlzLnJhd1Bvc2l0aW9ucy5hZGQodGhyZWFkLmNvbW1lbnRzWzBdLnBvc2l0aW9uKTtcbiAgfVxuXG4gIHVwZGF0ZUlmTmVjZXNzYXJ5KHttdWx0aUZpbGVQYXRjaCwgZGlmZnMsIGRpZmZQb3NpdGlvbkZuLCB0cmFuc2xhdGVQb3NpdGlvbkZufSkge1xuICAgIGlmIChcbiAgICAgIHRoaXMubGFzdC5tdWx0aUZpbGVQYXRjaCA9PT0gbXVsdGlGaWxlUGF0Y2ggJiZcbiAgICAgIHRoaXMubGFzdC5kaWZmcyA9PT0gZGlmZnNcbiAgICApIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICB0aGlzLmxhc3QubXVsdGlGaWxlUGF0Y2ggPSBtdWx0aUZpbGVQYXRjaDtcbiAgICB0aGlzLmxhc3QuZGlmZnMgPSBkaWZmcztcblxuICAgIHJldHVybiB0aGlzLnVwZGF0ZSh7bXVsdGlGaWxlUGF0Y2gsIGRpZmZzLCBkaWZmUG9zaXRpb25GbiwgdHJhbnNsYXRlUG9zaXRpb25Gbn0pO1xuICB9XG5cbiAgdXBkYXRlKHttdWx0aUZpbGVQYXRjaCwgZGlmZnMsIGRpZmZQb3NpdGlvbkZuLCB0cmFuc2xhdGVQb3NpdGlvbkZufSkge1xuICAgIGNvbnN0IGZpbGVQYXRjaCA9IG11bHRpRmlsZVBhdGNoLmdldFBhdGNoRm9yUGF0aCh0aGlzLm5hdGl2ZVJlbFBhdGgpO1xuICAgIC8vIENvbW1lbnQgb24gYSBmaWxlIHRoYXQgdXNlZCB0byBleGlzdCBpbiBhIFBSIGJ1dCBubyBsb25nZXIgZG9lcy4gU2tpcCBzaWxlbnRseS5cbiAgICBpZiAoIWZpbGVQYXRjaCkge1xuICAgICAgdGhpcy5kaWZmVG9GaWxlUG9zaXRpb24gPSBuZXcgTWFwKCk7XG4gICAgICB0aGlzLnJlbW92ZWQgPSBmYWxzZTtcbiAgICAgIHRoaXMuZmlsZVRyYW5zbGF0aW9ucyA9IG51bGw7XG5cbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICAvLyBUaGlzIGNvbW1lbnQgd2FzIGxlZnQgb24gYSBmaWxlIHRoYXQgd2FzIHRvbyBsYXJnZSB0byBwYXJzZS5cbiAgICBpZiAoIWZpbGVQYXRjaC5nZXRSZW5kZXJTdGF0dXMoKS5pc1Zpc2libGUoKSkge1xuICAgICAgdGhpcy5kaWZmVG9GaWxlUG9zaXRpb24gPSBuZXcgTWFwKCk7XG4gICAgICB0aGlzLnJlbW92ZWQgPSB0cnVlO1xuICAgICAgdGhpcy5maWxlVHJhbnNsYXRpb25zID0gbnVsbDtcblxuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMuZGlmZlRvRmlsZVBvc2l0aW9uID0gZGlmZlBvc2l0aW9uRm4odGhpcy5yYXdQb3NpdGlvbnMsIGZpbGVQYXRjaC5nZXRSYXdDb250ZW50UGF0Y2goKSk7XG4gICAgdGhpcy5yZW1vdmVkID0gZmFsc2U7XG5cbiAgICBsZXQgY29udGVudENoYW5nZURpZmY7XG4gICAgaWYgKGRpZmZzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgY29udGVudENoYW5nZURpZmYgPSBkaWZmc1swXTtcbiAgICB9IGVsc2UgaWYgKGRpZmZzLmxlbmd0aCA9PT0gMikge1xuICAgICAgY29uc3QgW2RpZmYxLCBkaWZmMl0gPSBkaWZmcztcbiAgICAgIGlmIChkaWZmMS5vbGRNb2RlID09PSBGaWxlLm1vZGVzLlNZTUxJTksgfHwgZGlmZjEubmV3TW9kZSA9PT0gRmlsZS5tb2Rlcy5TWU1MSU5LKSB7XG4gICAgICAgIGNvbnRlbnRDaGFuZ2VEaWZmID0gZGlmZjI7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBjb250ZW50Q2hhbmdlRGlmZiA9IGRpZmYxO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmIChjb250ZW50Q2hhbmdlRGlmZikge1xuICAgICAgY29uc3QgZmlsZVBvc2l0aW9ucyA9IFsuLi50aGlzLmRpZmZUb0ZpbGVQb3NpdGlvbi52YWx1ZXMoKV07XG4gICAgICB0aGlzLmZpbGVUcmFuc2xhdGlvbnMgPSB0cmFuc2xhdGVQb3NpdGlvbkZuKGZpbGVQb3NpdGlvbnMsIGNvbnRlbnRDaGFuZ2VEaWZmKTtcblxuICAgICAgY29uc3QgaGFzaCA9IGNyeXB0by5jcmVhdGVIYXNoKCdzaGEyNTYnKTtcbiAgICAgIGhhc2gudXBkYXRlKEpTT04uc3RyaW5naWZ5KEFycmF5LmZyb20odGhpcy5maWxlVHJhbnNsYXRpb25zLmVudHJpZXMoKSkpKTtcbiAgICAgIHRoaXMuZGlnZXN0ID0gaGFzaC5kaWdlc3QoJ2hleCcpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLmZpbGVUcmFuc2xhdGlvbnMgPSBudWxsO1xuICAgICAgdGhpcy5kaWdlc3QgPSBudWxsO1xuICAgIH1cbiAgfVxufVxuIl19