(function() {
  var CompositeDisposable, CorrectionsView, SpellCheckTask, SpellCheckView, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SpellCheckTask = require('./spell-check-task');

  CorrectionsView = null;

  module.exports = SpellCheckView = (function() {
    function SpellCheckView(editor, spellCheckModule, manager) {
      this.editor = editor;
      this.spellCheckModule = spellCheckModule;
      this.manager = manager;
      this.makeCorrection = bind(this.makeCorrection, this);
      this.addContextMenuEntries = bind(this.addContextMenuEntries, this);
      this.disposables = new CompositeDisposable;
      this.initializeMarkerLayer();
      this.taskWrapper = new SpellCheckTask(this.manager);
      this.correctMisspellingCommand = atom.commands.add(atom.views.getView(this.editor), 'spell-check:correct-misspelling', (function(_this) {
        return function() {
          var marker, ref;
          if (marker = _this.markerLayer.findMarkers({
            containsBufferPosition: _this.editor.getCursorBufferPosition()
          })[0]) {
            if (CorrectionsView == null) {
              CorrectionsView = require('./corrections-view');
            }
            if ((ref = _this.correctionsView) != null) {
              ref.destroy();
            }
            _this.correctionsView = new CorrectionsView(_this.editor, _this.getCorrections(marker), marker, _this, _this.updateMisspellings);
            return _this.correctionsView.attach();
          }
        };
      })(this));
      atom.views.getView(this.editor).addEventListener('contextmenu', this.addContextMenuEntries);
      this.disposables.add(this.editor.onDidChangePath((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(this.editor.onDidChangeGrammar((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('editor.fontSize', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('spell-check.grammars', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.subscribeToBuffer();
      this.disposables.add(this.editor.onDidDestroy(this.destroy.bind(this)));
    }

    SpellCheckView.prototype.initializeMarkerLayer = function() {
      this.markerLayer = this.editor.addMarkerLayer({
        maintainHistory: false
      });
      return this.markerLayerDecoration = this.editor.decorateMarkerLayer(this.markerLayer, {
        type: 'highlight',
        "class": 'spell-check-misspelling',
        deprecatedRegionClass: 'misspelling'
      });
    };

    SpellCheckView.prototype.destroy = function() {
      var ref;
      this.unsubscribeFromBuffer();
      this.disposables.dispose();
      this.taskWrapper.terminate();
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      this.correctMisspellingCommand.dispose();
      if ((ref = this.correctionsView) != null) {
        ref.destroy();
      }
      return this.clearContextMenuEntries();
    };

    SpellCheckView.prototype.unsubscribeFromBuffer = function() {
      this.destroyMarkers();
      if (this.buffer != null) {
        this.bufferDisposable.dispose();
        return this.buffer = null;
      }
    };

    SpellCheckView.prototype.subscribeToBuffer = function() {
      this.unsubscribeFromBuffer();
      if (this.spellCheckCurrentGrammar()) {
        this.buffer = this.editor.getBuffer();
        this.bufferDisposable = new CompositeDisposable(this.buffer.onDidStopChanging((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this), this.editor.onDidTokenize((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this))));
        return this.updateMisspellings();
      }
    };

    SpellCheckView.prototype.spellCheckCurrentGrammar = function() {
      var grammar;
      grammar = this.editor.getGrammar().scopeName;
      return _.contains(atom.config.get('spell-check.grammars'), grammar);
    };

    SpellCheckView.prototype.destroyMarkers = function() {
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      return this.initializeMarkerLayer();
    };

    SpellCheckView.prototype.addMarkers = function(misspellings) {
      var i, len, misspelling, results, scope;
      results = [];
      for (i = 0, len = misspellings.length; i < len; i++) {
        misspelling = misspellings[i];
        scope = this.editor.scopeDescriptorForBufferPosition(misspelling[0]);
        if (!this.scopeIsExcluded(scope)) {
          results.push(this.markerLayer.markBufferRange(misspelling, {
            invalidate: 'touch'
          }));
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    SpellCheckView.prototype.updateMisspellings = function() {
      return this.taskWrapper.start(this.editor, (function(_this) {
        return function(misspellings) {
          _this.destroyMarkers();
          if (_this.buffer != null) {
            return _this.addMarkers(misspellings);
          }
        };
      })(this));
    };

    SpellCheckView.prototype.getCorrections = function(marker) {
      var args, misspelling, projectPath, ref, ref1, ref2, relativePath;
      projectPath = null;
      relativePath = null;
      if ((ref = this.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
        ref2 = atom.project.relativizePath(this.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
      }
      args = {
        projectPath: projectPath,
        relativePath: relativePath
      };
      misspelling = this.editor.getTextInBufferRange(marker.getBufferRange());
      return this.manager.suggest(args, misspelling);
    };

    SpellCheckView.prototype.addContextMenuEntries = function(mouseEvent) {
      var commandName, contextMenuEntry, correction, correctionIndex, corrections, currentBufferPosition, currentScreenPosition, i, len, marker;
      this.clearContextMenuEntries();
      currentScreenPosition = atom.views.getView(this.editor).component.screenPositionForMouseEvent(mouseEvent);
      currentBufferPosition = this.editor.bufferPositionForScreenPosition(currentScreenPosition);
      if (marker = this.markerLayer.findMarkers({
        containsBufferPosition: currentBufferPosition
      })[0]) {
        corrections = this.getCorrections(marker);
        if (corrections.length > 0) {
          this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
          correctionIndex = 0;
          for (i = 0, len = corrections.length; i < len; i++) {
            correction = corrections[i];
            contextMenuEntry = {};
            commandName = 'spell-check:correct-misspelling-' + correctionIndex;
            contextMenuEntry.command = (function(_this) {
              return function(correction, contextMenuEntry) {
                return atom.commands.add(atom.views.getView(_this.editor), commandName, function() {
                  _this.makeCorrection(correction, marker);
                  return _this.clearContextMenuEntries();
                });
              };
            })(this)(correction, contextMenuEntry);
            contextMenuEntry.menuItem = atom.contextMenu.add({
              'atom-text-editor': [
                {
                  label: correction.label,
                  command: commandName
                }
              ]
            });
            this.spellCheckModule.contextMenuEntries.push(contextMenuEntry);
            correctionIndex++;
          }
          return this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
        }
      }
    };

    SpellCheckView.prototype.makeCorrection = function(correction, marker) {
      var args, projectPath, ref, ref1, ref2, relativePath;
      if (correction.isSuggestion) {
        this.editor.setSelectedBufferRange(marker.getBufferRange());
        return this.editor.insertText(correction.suggestion);
      } else {
        projectPath = null;
        relativePath = null;
        if ((ref = this.editor.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
          ref2 = atom.project.relativizePath(this.editor.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
        }
        args = {
          id: this.id,
          projectPath: projectPath,
          relativePath: relativePath
        };
        correction.plugin.add(args, correction);
        return this.updateMisspellings.bind(this)();
      }
    };

    SpellCheckView.prototype.clearContextMenuEntries = function() {
      var entry, i, len, ref, ref1, ref2;
      ref = this.spellCheckModule.contextMenuEntries;
      for (i = 0, len = ref.length; i < len; i++) {
        entry = ref[i];
        if ((ref1 = entry.command) != null) {
          ref1.dispose();
        }
        if ((ref2 = entry.menuItem) != null) {
          ref2.dispose();
        }
      }
      return this.spellCheckModule.contextMenuEntries = [];
    };

    SpellCheckView.prototype.scopeIsExcluded = function(scopeDescriptor, excludedScopes) {
      return this.spellCheckModule.excludedScopeRegexLists.some(function(regexList) {
        return scopeDescriptor.scopes.some(function(scopeName) {
          return regexList.every(function(regex) {
            return regex.test(scopeName);
          });
        });
      });
    };

    return SpellCheckView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
