import { fromCompare } from './Ord';
import { semigroupOrdering } from './Ordering';
import { fromEquals } from './Setoid';
export const URI = 'Pair';
/**
 * @data
 * @constructor Pair
 * @since 1.0.0
 */
export class Pair {
    constructor(fst, snd) {
        this.fst = fst;
        this.snd = snd;
    }
    /** Map a function over the first field of a pair */
    first(f) {
        return new Pair(f(this.fst), this.snd);
    }
    /** Map a function over the second field of a pair */
    second(f) {
        return new Pair(this.fst, f(this.snd));
    }
    /** Swaps the elements in a pair */
    swap() {
        return new Pair(this.snd, this.fst);
    }
    map(f) {
        return new Pair(f(this.fst), f(this.snd));
    }
    ap(fab) {
        return new Pair(fab.fst(this.fst), fab.snd(this.snd));
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    reduce(b, f) {
        return f(f(b, this.fst), this.snd);
    }
    extract() {
        return this.fst;
    }
    extend(f) {
        return new Pair(f(this), f(this.swap()));
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    return {
        show: p => `new Pair(${S.show(p.fst)}, ${S.show(p.snd)})`
    };
};
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new Pair(a, a);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return M.concat(f(fa.fst), f(fa.snd));
};
const foldr = (fa, b, f) => {
    return f(fa.fst, f(fa.snd, b));
};
const extract = (fa) => {
    return fa.extract();
};
const extend = (fa, f) => {
    return fa.extend(f);
};
/**
 * @since 1.0.0
 */
export const getSetoid = (S) => {
    return fromEquals((x, y) => S.equals(x.fst, y.fst) && S.equals(x.snd, y.snd));
};
/**
 * @since 1.0.0
 */
export const getOrd = (O) => {
    return fromCompare((x, y) => semigroupOrdering.concat(O.compare(x.fst, y.fst), O.compare(x.snd, y.snd)));
};
/**
 * @since 1.0.0
 */
export const getSemigroup = (S) => {
    return {
        concat: (x, y) => new Pair(S.concat(x.fst, y.fst), S.concat(x.snd, y.snd))
    };
};
/**
 * @since 1.0.0
 */
export const getMonoid = (M) => {
    return Object.assign({}, getSemigroup(M), { empty: new Pair(M.empty, M.empty) });
};
const traverse = (F) => (ta, f) => {
    return F.ap(F.map(f(ta.fst), (b1) => (b2) => new Pair(b1, b2)), f(ta.snd));
};
const sequence = (F) => (ta) => {
    return F.ap(F.map(ta.fst, (a1) => (a2) => new Pair(a1, a2)), ta.snd);
};
/**
 * @since 1.0.0
 */
export const pair = {
    URI,
    map,
    of,
    ap,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    extend,
    extract
};
