import { identity, not, toString } from './function';
import { getDualMonoid } from './Monoid';
import { fromCompare } from './Ord';
import { fromEquals } from './Setoid';
export const URI = 'Option';
export class None {
    constructor() {
        this._tag = 'None';
    }
    /**
     * Takes a function `f` and an `Option` of `A`. Maps `f` either on `None` or `Some`, Option's data constructors. If it
     * maps on `Some` then it will apply the `f` on `Some`'s value, if it maps on `None` it will return `None`.
     *
     * @example
     * import { some } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(some(1).map(n => n * 2), some(2))
     */
    map(f) {
        return none;
    }
    /**
     * Maps `f` over this `Option`'s value. If the value returned from `f` is null or undefined, returns `None`
     *
     * @example
     * import { none, some } from 'fp-ts/lib/Option'
     *
     * interface Foo {
     *   bar?: {
     *     baz?: string
     *   }
     * }
     *
     * assert.deepStrictEqual(
     *   some<Foo>({ bar: { baz: 'quux' } })
     *     .mapNullable(foo => foo.bar)
     *     .mapNullable(bar => bar.baz),
     *   some('quux')
     * )
     * assert.deepStrictEqual(
     *   some<Foo>({ bar: {} })
     *     .mapNullable(foo => foo.bar)
     *     .mapNullable(bar => bar.baz),
     *   none
     * )
     * assert.deepStrictEqual(
     *   some<Foo>({})
     *     .mapNullable(foo => foo.bar)
     *     .mapNullable(bar => bar.baz),
     *   none
     * )
     */
    mapNullable(f) {
        return none;
    }
    /**
     * `ap`, some may also call it "apply". Takes a function `fab` that is in the context of `Option`, and applies that
     * function to this `Option`'s value. If the `Option` calling `ap` is `none` it will return `none`.
     *
     * @example
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(some(2).ap(some((x: number) => x + 1)), some(3))
     * assert.deepStrictEqual(none.ap(some((x: number) => x + 1)), none)
     */
    ap(fab) {
        return none;
    }
    /**
     * Flipped version of `ap`
     *
     * @example
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(some((x: number) => x + 1).ap_(some(2)), some(3))
     * assert.deepStrictEqual(none.ap_(some(2)), none)
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /**
     * Returns the result of applying f to this `Option`'s value if this `Option` is nonempty. Returns `None` if this
     * `Option` is empty. Slightly different from `map` in that `f` is expected to return an `Option` (which could be
     * `None`)
     */
    chain(f) {
        return none;
    }
    reduce(b, f) {
        return b;
    }
    /**
     * `alt` short for alternative, takes another `Option`. If this `Option` is a `Some` type then it will be returned, if
     * it is a `None` then it will return the next `Some` if it exist. If both are `None` then it will return `none`.
     *
     * @example
     * import { Option, some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(some(2).alt(some(4)), some(2))
     * const fa: Option<number> = none
     * assert.deepStrictEqual(fa.alt(some(4)), some(4))
     */
    alt(fa) {
        return fa;
    }
    /**
     * Lazy version of `alt`
     *
     * @example
     * import { some } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(some(1).orElse(() => some(2)), some(1))
     *
     * @since 1.6.0
     */
    orElse(fa) {
        return fa();
    }
    extend(f) {
        return none;
    }
    /**
     * Applies a function to each case in the data structure
     *
     * @example
     * import { none, some } from 'fp-ts/lib/Option'
     *
     * assert.strictEqual(some(1).fold('none', a => `some: ${a}`), 'some: 1')
     * assert.strictEqual(none.fold('none', a => `some: ${a}`), 'none')
     */
    fold(b, onSome) {
        return b;
    }
    /** Lazy version of `fold` */
    foldL(onNone, onSome) {
        return onNone();
    }
    /**
     * Returns the value from this `Some` or the given argument if this is a `None`
     *
     * @example
     * import { Option, none, some } from 'fp-ts/lib/Option'
     *
     * assert.strictEqual(some(1).getOrElse(0), 1)
     * const fa: Option<number> = none
     * assert.strictEqual(fa.getOrElse(0), 0)
     */
    getOrElse(a) {
        return a;
    }
    /** Lazy version of `getOrElse` */
    getOrElseL(f) {
        return f();
    }
    /** Returns the value from this `Some` or `null` if this is a `None` */
    toNullable() {
        return null;
    }
    /** Returns the value from this `Some` or `undefined` if this is a `None` */
    toUndefined() {
        return undefined;
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return 'none';
    }
    /** Returns `true` if the option has an element that is equal (as determined by `S`) to `a`, `false` otherwise */
    contains(S, a) {
        return false;
    }
    /** Returns `true` if the option is `None`, `false` otherwise */
    isNone() {
        return true;
    }
    /** Returns `true` if the option is an instance of `Some`, `false` otherwise */
    isSome() {
        return false;
    }
    /**
     * Returns `true` if this option is non empty and the predicate `p` returns `true` when applied to this Option's value
     */
    exists(p) {
        return false;
    }
    filter(p) {
        return none;
    }
    /**
     * Use `filter` instead.
     * Returns this option refined as `Option<B>` if it is non empty and the `refinement` returns `true` when applied to
     * this Option's value. Otherwise returns `None`
     * @since 1.3.0
     * @deprecated
     */
    refine(refinement) {
        return none;
    }
}
None.value = new None();
/**
 * @since 1.0.0
 */
export const none = None.value;
export class Some {
    constructor(value) {
        this.value = value;
        this._tag = 'Some';
    }
    map(f) {
        return new Some(f(this.value));
    }
    mapNullable(f) {
        return fromNullable(f(this.value));
    }
    ap(fab) {
        return fab.isNone() ? none : new Some(fab.value(this.value));
    }
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return f(this.value);
    }
    reduce(b, f) {
        return f(b, this.value);
    }
    alt(fa) {
        return this;
    }
    orElse(fa) {
        return this;
    }
    extend(f) {
        return new Some(f(this));
    }
    fold(b, onSome) {
        return onSome(this.value);
    }
    foldL(onNone, onSome) {
        return onSome(this.value);
    }
    getOrElse(a) {
        return this.value;
    }
    getOrElseL(f) {
        return this.value;
    }
    toNullable() {
        return this.value;
    }
    toUndefined() {
        return this.value;
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `some(${toString(this.value)})`;
    }
    contains(S, a) {
        return S.equals(this.value, a);
    }
    isNone() {
        return false;
    }
    isSome() {
        return true;
    }
    exists(p) {
        return p(this.value);
    }
    filter(p) {
        return this.exists(p) ? this : none;
    }
    refine(refinement) {
        return this.filter(refinement);
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    return {
        show: oa => oa.fold('none', a => `some(${S.show(a)})`)
    };
};
/**
 * @example
 * import { none, some, getSetoid } from 'fp-ts/lib/Option'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * const S = getSetoid(setoidNumber)
 * assert.strictEqual(S.equals(none, none), true)
 * assert.strictEqual(S.equals(none, some(1)), false)
 * assert.strictEqual(S.equals(some(1), none), false)
 * assert.strictEqual(S.equals(some(1), some(2)), false)
 * assert.strictEqual(S.equals(some(1), some(1)), true)
 *
 * @since 1.0.0
 */
export const getSetoid = (S) => {
    return fromEquals((x, y) => (x.isNone() ? y.isNone() : y.isNone() ? false : S.equals(x.value, y.value)));
};
/**
 * The `Ord` instance allows `Option` values to be compared with
 * `compare`, whenever there is an `Ord` instance for
 * the type the `Option` contains.
 *
 * `None` is considered to be less than any `Some` value.
 *
 *
 * @example
 * import { none, some, getOrd } from 'fp-ts/lib/Option'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * const O = getOrd(ordNumber)
 * assert.strictEqual(O.compare(none, none), 0)
 * assert.strictEqual(O.compare(none, some(1)), -1)
 * assert.strictEqual(O.compare(some(1), none), 1)
 * assert.strictEqual(O.compare(some(1), some(2)), -1)
 * assert.strictEqual(O.compare(some(1), some(1)), 0)
 *
 * @since 1.2.0
 */
export const getOrd = (O) => {
    return fromCompare((x, y) => (x.isSome() ? (y.isSome() ? O.compare(x.value, y.value) : 1) : -1));
};
const map = (fa, f) => {
    return fa.map(f);
};
/**
 * @since 1.0.0
 */
export const some = (a) => {
    return new Some(a);
};
const of = some;
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.isNone() ? M.empty : f(fa.value);
};
const foldr = (fa, b, f) => {
    return fa.isNone() ? b : f(fa.value, b);
};
const traverse = (F) => (ta, f) => {
    return ta.isNone() ? F.of(none) : F.map(f(ta.value), some);
};
const sequence = (F) => (ta) => {
    return ta.isNone() ? F.of(none) : F.map(ta.value, some);
};
const alt = (fx, fy) => {
    return fx.alt(fy);
};
const extend = (ea, f) => {
    return ea.extend(f);
};
const zero = () => {
    return none;
};
/**
 * `Apply` semigroup
 *
 * | x       | y       | concat(x, y)       |
 * | ------- | ------- | ------------------ |
 * | none    | none    | none               |
 * | some(a) | none    | none               |
 * | none    | some(a) | none               |
 * | some(a) | some(b) | some(concat(a, b)) |
 *
 * @example
 * import { getApplySemigroup, some, none } from 'fp-ts/lib/Option'
 * import { semigroupSum } from 'fp-ts/lib/Semigroup'
 *
 * const S = getApplySemigroup(semigroupSum)
 * assert.deepStrictEqual(S.concat(none, none), none)
 * assert.deepStrictEqual(S.concat(some(1), none), none)
 * assert.deepStrictEqual(S.concat(none, some(1)), none)
 * assert.deepStrictEqual(S.concat(some(1), some(2)), some(3))
 *
 * @since 1.7.0
 */
export const getApplySemigroup = (S) => {
    return {
        concat: (x, y) => (x.isSome() && y.isSome() ? some(S.concat(x.value, y.value)) : none)
    };
};
/**
 * @since 1.7.0
 */
export const getApplyMonoid = (M) => {
    return Object.assign({}, getApplySemigroup(M), { empty: some(M.empty) });
};
/**
 * Monoid returning the left-most non-`None` value
 *
 * | x       | y       | concat(x, y) |
 * | ------- | ------- | ------------ |
 * | none    | none    | none         |
 * | some(a) | none    | some(a)      |
 * | none    | some(a) | some(a)      |
 * | some(a) | some(b) | some(a)      |
 *
 * @example
 * import { getFirstMonoid, some, none } from 'fp-ts/lib/Option'
 *
 * const M = getFirstMonoid<number>()
 * assert.deepStrictEqual(M.concat(none, none), none)
 * assert.deepStrictEqual(M.concat(some(1), none), some(1))
 * assert.deepStrictEqual(M.concat(none, some(1)), some(1))
 * assert.deepStrictEqual(M.concat(some(1), some(2)), some(1))
 *
 * @since 1.0.0
 */
export const getFirstMonoid = () => {
    return {
        concat: alt,
        empty: none
    };
};
/**
 * Monoid returning the right-most non-`None` value
 *
 * | x       | y       | concat(x, y) |
 * | ------- | ------- | ------------ |
 * | none    | none    | none         |
 * | some(a) | none    | some(a)      |
 * | none    | some(a) | some(a)      |
 * | some(a) | some(b) | some(b)      |
 *
 * @example
 * import { getLastMonoid, some, none } from 'fp-ts/lib/Option'
 *
 * const M = getLastMonoid<number>()
 * assert.deepStrictEqual(M.concat(none, none), none)
 * assert.deepStrictEqual(M.concat(some(1), none), some(1))
 * assert.deepStrictEqual(M.concat(none, some(1)), some(1))
 * assert.deepStrictEqual(M.concat(some(1), some(2)), some(2))
 *
 * @since 1.0.0
 */
export const getLastMonoid = () => {
    return getDualMonoid(getFirstMonoid());
};
/**
 * Monoid returning the left-most non-`None` value. If both operands are `Some`s then the inner values are
 * appended using the provided `Semigroup`
 *
 * | x       | y       | concat(x, y)       |
 * | ------- | ------- | ------------------ |
 * | none    | none    | none               |
 * | some(a) | none    | some(a)            |
 * | none    | some(a) | some(a)            |
 * | some(a) | some(b) | some(concat(a, b)) |
 *
 * @example
 * import { getMonoid, some, none } from 'fp-ts/lib/Option'
 * import { semigroupSum } from 'fp-ts/lib/Semigroup'
 *
 * const M = getMonoid(semigroupSum)
 * assert.deepStrictEqual(M.concat(none, none), none)
 * assert.deepStrictEqual(M.concat(some(1), none), some(1))
 * assert.deepStrictEqual(M.concat(none, some(1)), some(1))
 * assert.deepStrictEqual(M.concat(some(1), some(2)), some(3))
 *
 * @since 1.0.0
 */
export const getMonoid = (S) => {
    return {
        concat: (x, y) => (x.isNone() ? y : y.isNone() ? x : some(S.concat(x.value, y.value))),
        empty: none
    };
};
/**
 * Constructs a new `Option` from a nullable type. If the value is `null` or `undefined`, returns `None`, otherwise
 * returns the value wrapped in a `Some`
 *
 * @example
 * import { none, some, fromNullable } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(fromNullable(undefined), none)
 * assert.deepStrictEqual(fromNullable(null), none)
 * assert.deepStrictEqual(fromNullable(1), some(1))
 *
 * @since 1.0.0
 */
export const fromNullable = (a) => {
    return a == null ? none : new Some(a);
};
export function fromPredicate(predicate) {
    return a => (predicate(a) ? some(a) : none);
}
/**
 * Transforms an exception into an `Option`. If `f` throws, returns `None`, otherwise returns the output wrapped in
 * `Some`
 *
 * @example
 * import { none, some, tryCatch } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(
 *   tryCatch(() => {
 *     throw new Error()
 *   }),
 *   none
 * )
 * assert.deepStrictEqual(tryCatch(() => 1), some(1))
 *
 * @since 1.0.0
 */
export const tryCatch = (f) => {
    try {
        return some(f());
    }
    catch (e) {
        return none;
    }
};
/**
 * Constructs a new `Option` from a `Either`. If the value is a `Left`, returns `None`, otherwise returns the inner
 * value wrapped in a `Some`
 *
 * @example
 * import { none, some, fromEither } from 'fp-ts/lib/Option'
 * import { left, right } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(fromEither(left(1)), none)
 * assert.deepStrictEqual(fromEither(right(1)), some(1))
 *
 * @since 1.0.0
 */
export const fromEither = (fa) => {
    return fa.isLeft() ? none : some(fa.value);
};
/**
 * Returns `true` if the option is an instance of `Some`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isSome = (fa) => {
    return fa.isSome();
};
/**
 * Returns `true` if the option is `None`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isNone = (fa) => {
    return fa.isNone();
};
/**
 * Use `fromPredicate` instead.
 * Refinement version of `fromPredicate`
 *
 * @since 1.3.0
 * @deprecated
 */
export const fromRefinement = (refinement) => (a) => {
    return refinement(a) ? some(a) : none;
};
/**
 * Returns a refinement from a prism.
 * This function ensures that a custom type guard definition is type-safe.
 *
 * ```ts
 * import { some, none, getRefinement } from 'fp-ts/lib/Option'
 *
 * type A = { type: 'A' }
 * type B = { type: 'B' }
 * type C = A | B
 *
 * const isA = (c: C): c is A => c.type === 'B' // <= typo but typescript doesn't complain
 * const isA = getRefinement<C, A>(c => (c.type === 'B' ? some(c) : none)) // static error: Type '"B"' is not assignable to type '"A"'
 * ```
 *
 * @since 1.7.0
 */
export const getRefinement = (getOption) => {
    return (a) => getOption(a).isSome();
};
const compact = (fa) => fa.chain(identity);
const separate = (fa) => {
    if (fa.isNone()) {
        return {
            left: none,
            right: none
        };
    }
    const e = fa.value;
    if (e.isLeft()) {
        return {
            left: some(e.value),
            right: none
        };
    }
    return {
        left: none,
        right: some(e.value)
    };
};
const filter = (fa, p) => fa.filter(p);
const filterMap = chain;
const partitionMap = (fa, f) => separate(fa.map(f));
const partition = (fa, p) => ({
    left: fa.filter(not(p)),
    right: fa.filter(p)
});
const wither = (F) => (fa, f) => fa.isNone() ? F.of(fa) : f(fa.value);
const wilt = (F) => (fa, f) => {
    if (fa.isNone()) {
        return F.of({
            left: none,
            right: none
        });
    }
    return F.map(f(fa.value), e => {
        if (e.isLeft()) {
            return {
                left: some(e.value),
                right: none
            };
        }
        return {
            left: none,
            right: some(e.value)
        };
    });
};
/**
 * @since 1.0.0
 */
export const option = {
    URI,
    map,
    of,
    ap,
    chain,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    zero,
    alt,
    extend,
    compact,
    separate,
    filter,
    filterMap,
    partition,
    partitionMap,
    wither,
    wilt
};
