"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.REMOVED = exports.DEFERRED = exports.COLLAPSED = exports.EXPANDED = void 0;

var _atom = require("atom");

var _hunk = _interopRequireDefault(require("./hunk"));

var _region = require("./region");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const EXPANDED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(expanded)';
  },

  isVisible() {
    return true;
  },

  isExpandable() {
    return false;
  }

};
exports.EXPANDED = EXPANDED;
const COLLAPSED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(collapsed)';
  },

  isVisible() {
    return false;
  },

  isExpandable() {
    return true;
  }

};
exports.COLLAPSED = COLLAPSED;
const DEFERRED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(deferred)';
  },

  isVisible() {
    return false;
  },

  isExpandable() {
    return true;
  }

};
exports.DEFERRED = DEFERRED;
const REMOVED = {
  /* istanbul ignore next */
  toString() {
    return 'RenderStatus(removed)';
  },

  isVisible() {
    return false;
  },

  isExpandable() {
    return false;
  }

};
exports.REMOVED = REMOVED;

class Patch {
  static createNull() {
    return new NullPatch();
  }

  static createHiddenPatch(marker, renderStatus, showFn) {
    return new HiddenPatch(marker, renderStatus, showFn);
  }

  constructor({
    status,
    hunks,
    marker
  }) {
    this.status = status;
    this.hunks = hunks;
    this.marker = marker;
    this.changedLineCount = this.getHunks().reduce((acc, hunk) => acc + hunk.changedLineCount(), 0);
  }

  getStatus() {
    return this.status;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    const startPoint = this.getMarker().getRange().start;
    return _atom.Range.fromObject([startPoint, startPoint]);
  }

  getHunks() {
    return this.hunks;
  }

  getChangedLineCount() {
    return this.changedLineCount;
  }

  containsRow(row) {
    return this.marker.getRange().intersectsRow(row);
  }

  destroyMarkers() {
    this.marker.destroy();

    for (const hunk of this.hunks) {
      hunk.destroyMarkers();
    }
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;

    for (const hunk of this.hunks) {
      hunk.updateMarkers(map);
    }
  }

  getMaxLineNumberWidth() {
    const lastHunk = this.hunks[this.hunks.length - 1];
    return lastHunk ? lastHunk.getMaxLineNumberWidth() : 0;
  }

  clone(opts = {}) {
    return new this.constructor({
      status: opts.status !== undefined ? opts.status : this.getStatus(),
      hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
      marker: opts.marker !== undefined ? opts.marker : this.getMarker()
    });
  }
  /* Return the set of Markers owned by this Patch that butt up against the patch's beginning. */


  getStartingMarkers() {
    const markers = [this.marker];

    if (this.hunks.length > 0) {
      const firstHunk = this.hunks[0];
      markers.push(firstHunk.getMarker());

      if (firstHunk.getRegions().length > 0) {
        const firstRegion = firstHunk.getRegions()[0];
        markers.push(firstRegion.getMarker());
      }
    }

    return markers;
  }
  /* Return the set of Markers owned by this Patch that end at the patch's end position. */


  getEndingMarkers() {
    const markers = [this.marker];

    if (this.hunks.length > 0) {
      const lastHunk = this.hunks[this.hunks.length - 1];
      markers.push(lastHunk.getMarker());

      if (lastHunk.getRegions().length > 0) {
        const lastRegion = lastHunk.getRegions()[lastHunk.getRegions().length - 1];
        markers.push(lastRegion.getMarker());
      }
    }

    return markers;
  }

  buildStagePatchForLines(originalBuffer, nextPatchBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextPatchBuffer);
    const hunks = [];
    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let selectedDeletionRowCount = 0;
      let noNewlineRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref of region.intersectRows(rowSet, true)) {
          const {
            intersection,
            gap
          } = _ref;
          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: omit from new buffer
                builder.remove(intersection);
              } else {
                // Selected addition: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
              }
            },
            deletion: () => {
              if (gap) {
                // Unselected deletion: convert to context row
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
              } else {
                // Selected deletion: include in new patch
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                selectedDeletionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
            },
            nonewline: () => {
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
              noNewlineRowCount += intersection.getRowCount();
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line
        builder.markHunkRange(hunk.getRange());
        const {
          regions,
          marker
        } = builder.latestHunkWasIncluded();
        const newStartRow = hunk.getNewStartRow() + newRowDelta;
        const newRowCount = marker.getRange().getRowCount() - selectedDeletionRowCount - noNewlineRowCount;
        hunks.push(new _hunk["default"]({
          oldStartRow: hunk.getOldStartRow(),
          oldRowCount: hunk.getOldRowCount(),
          newStartRow,
          newRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));
        newRowDelta += newRowCount - hunk.getNewRowCount();
      } else {
        newRowDelta += hunk.getOldRowCount() - hunk.getNewRowCount();
        builder.latestHunkWasDiscarded();
      }
    }

    const marker = nextPatchBuffer.markRange(this.constructor.layerName, [[0, 0], [nextPatchBuffer.getBuffer().getLastRow() - 1, Infinity]], {
      invalidate: 'never',
      exclusive: false
    });
    const wholeFile = rowSet.size === this.changedLineCount;
    const status = this.getStatus() === 'deleted' && !wholeFile ? 'modified' : this.getStatus();
    return this.clone({
      hunks,
      status,
      marker
    });
  }

  buildUnstagePatchForLines(originalBuffer, nextPatchBuffer, rowSet) {
    const originalBaseOffset = this.getMarker().getRange().start.row;
    const builder = new BufferBuilder(originalBuffer, originalBaseOffset, nextPatchBuffer);
    const hunks = [];
    let newRowDelta = 0;

    for (const hunk of this.getHunks()) {
      let atLeastOneSelectedChange = false;
      let contextRowCount = 0;
      let additionRowCount = 0;
      let deletionRowCount = 0;

      for (const region of hunk.getRegions()) {
        for (const _ref2 of region.intersectRows(rowSet, true)) {
          const {
            intersection,
            gap
          } = _ref2;
          region.when({
            addition: () => {
              if (gap) {
                // Unselected addition: become a context line.
                builder.append(intersection);
                builder.markRegion(intersection, _region.Unchanged);
                contextRowCount += intersection.getRowCount();
              } else {
                // Selected addition: become a deletion.
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Deletion);
                deletionRowCount += intersection.getRowCount();
              }
            },
            deletion: () => {
              if (gap) {
                // Non-selected deletion: omit from new buffer.
                builder.remove(intersection);
              } else {
                // Selected deletion: becomes an addition
                atLeastOneSelectedChange = true;
                builder.append(intersection);
                builder.markRegion(intersection, _region.Addition);
                additionRowCount += intersection.getRowCount();
              }
            },
            unchanged: () => {
              // Untouched context line: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.Unchanged);
              contextRowCount += intersection.getRowCount();
            },
            nonewline: () => {
              // Nonewline marker: include in new patch.
              builder.append(intersection);
              builder.markRegion(intersection, _region.NoNewline);
            }
          });
        }
      }

      if (atLeastOneSelectedChange) {
        // Hunk contains at least one selected line
        builder.markHunkRange(hunk.getRange());
        const {
          marker,
          regions
        } = builder.latestHunkWasIncluded();
        hunks.push(new _hunk["default"]({
          oldStartRow: hunk.getNewStartRow(),
          oldRowCount: contextRowCount + deletionRowCount,
          newStartRow: hunk.getNewStartRow() + newRowDelta,
          newRowCount: contextRowCount + additionRowCount,
          sectionHeading: hunk.getSectionHeading(),
          marker,
          regions
        }));
      } else {
        builder.latestHunkWasDiscarded();
      } // (contextRowCount + additionRowCount) - (contextRowCount + deletionRowCount)


      newRowDelta += additionRowCount - deletionRowCount;
    }

    const wholeFile = rowSet.size === this.changedLineCount;
    let status = this.getStatus();

    if (this.getStatus() === 'added') {
      status = wholeFile ? 'deleted' : 'modified';
    } else if (this.getStatus() === 'deleted') {
      status = 'added';
    }

    const marker = nextPatchBuffer.markRange(this.constructor.layerName, [[0, 0], [nextPatchBuffer.getBuffer().getLastRow(), Infinity]], {
      invalidate: 'never',
      exclusive: false
    });
    return this.clone({
      hunks,
      status,
      marker
    });
  }

  getFirstChangeRange() {
    const firstHunk = this.getHunks()[0];

    if (!firstHunk) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstChange = firstHunk.getChanges()[0];

    if (!firstChange) {
      return _atom.Range.fromObject([[0, 0], [0, 0]]);
    }

    const firstRow = firstChange.getStartBufferRow();
    return _atom.Range.fromObject([[firstRow, 0], [firstRow, Infinity]]);
  }

  toStringIn(buffer) {
    return this.getHunks().reduce((str, hunk) => str + hunk.toStringIn(buffer), '');
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(Patch marker=${this.marker.id}`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    inspectString += '\n';

    for (const hunk of this.hunks) {
      inspectString += hunk.inspect({
        indent: options.indent + 2
      });
    }

    inspectString += `${indentation})\n`;
    return inspectString;
  }

  isPresent() {
    return true;
  }

  getRenderStatus() {
    return EXPANDED;
  }

}

exports["default"] = Patch;

_defineProperty(Patch, "layerName", 'patch');

class HiddenPatch extends Patch {
  constructor(marker, renderStatus, showFn) {
    super({
      status: null,
      hunks: [],
      marker
    });
    this.renderStatus = renderStatus;
    this.show = showFn;
  }

  getInsertionPoint() {
    return this.getRange().end;
  }

  getRenderStatus() {
    return this.renderStatus;
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    return `${indentation}(HiddenPatch marker=${this.marker.id})\n`;
  }

}

class NullPatch {
  constructor() {
    const buffer = new _atom.TextBuffer();
    this.marker = buffer.markRange([[0, 0], [0, 0]]);
  }

  getStatus() {
    return null;
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getStartRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  getHunks() {
    return [];
  }

  getChangedLineCount() {
    return 0;
  }

  containsRow() {
    return false;
  }

  getMaxLineNumberWidth() {
    return 0;
  }

  clone(opts = {}) {
    if (opts.status === undefined && opts.hunks === undefined && opts.marker === undefined && opts.renderStatus === undefined) {
      return this;
    } else {
      return new Patch({
        status: opts.status !== undefined ? opts.status : this.getStatus(),
        hunks: opts.hunks !== undefined ? opts.hunks : this.getHunks(),
        marker: opts.marker !== undefined ? opts.marker : this.getMarker(),
        renderStatus: opts.renderStatus !== undefined ? opts.renderStatus : this.getRenderStatus()
      });
    }
  }

  getStartingMarkers() {
    return [];
  }

  getEndingMarkers() {
    return [];
  }

  buildStagePatchForLines() {
    return this;
  }

  buildUnstagePatchForLines() {
    return this;
  }

  getFirstChangeRange() {
    return _atom.Range.fromObject([[0, 0], [0, 0]]);
  }

  updateMarkers() {}

  toStringIn() {
    return '';
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    return `${indentation}(NullPatch)\n`;
  }

  isPresent() {
    return false;
  }

  getRenderStatus() {
    return EXPANDED;
  }

}

class BufferBuilder {
  constructor(original, originalBaseOffset, nextPatchBuffer) {
    this.originalBuffer = original;
    this.nextPatchBuffer = nextPatchBuffer; // The ranges provided to builder methods are expected to be valid within the original buffer. Account for
    // the position of the Patch within its original TextBuffer, and any existing content already on the next
    // TextBuffer.

    this.offset = this.nextPatchBuffer.getBuffer().getLastRow() - originalBaseOffset;
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    this.lastOffset = 0;
  }

  append(range) {
    this.hunkBufferText += this.originalBuffer.getTextInRange(range) + '\n';
    this.hunkRowCount += range.getRowCount();
  }

  remove(range) {
    this.offset -= range.getRowCount();
  }

  markRegion(range, RegionKind) {
    const finalRange = this.offset !== 0 ? range.translate([this.offset, 0], [this.offset, 0]) : range; // Collapse consecutive ranges of the same RegionKind into one continuous region.

    const lastRegion = this.hunkRegions[this.hunkRegions.length - 1];

    if (lastRegion && lastRegion.RegionKind === RegionKind && finalRange.start.row - lastRegion.range.end.row === 1) {
      lastRegion.range.end = finalRange.end;
    } else {
      this.hunkRegions.push({
        RegionKind,
        range: finalRange
      });
    }
  }

  markHunkRange(range) {
    let finalRange = range;

    if (this.hunkStartOffset !== 0 || this.offset !== 0) {
      finalRange = finalRange.translate([this.hunkStartOffset, 0], [this.offset, 0]);
    }

    this.hunkRange = finalRange;
  }

  latestHunkWasIncluded() {
    this.nextPatchBuffer.buffer.append(this.hunkBufferText, {
      normalizeLineEndings: false
    });
    const regions = this.hunkRegions.map(({
      RegionKind,
      range
    }) => {
      const regionMarker = this.nextPatchBuffer.markRange(RegionKind.layerName, range, {
        invalidate: 'never',
        exclusive: false
      });
      return new RegionKind(regionMarker);
    });
    const marker = this.nextPatchBuffer.markRange('hunk', this.hunkRange, {
      invalidate: 'never',
      exclusive: false
    });
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    return {
      regions,
      marker
    };
  }

  latestHunkWasDiscarded() {
    this.offset -= this.hunkRowCount;
    this.hunkBufferText = '';
    this.hunkRowCount = 0;
    this.hunkStartOffset = this.offset;
    this.hunkRegions = [];
    this.hunkRange = null;
    return {
      regions: [],
      marker: null
    };
  }

}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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