/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import * as tslib_1 from "tslib";
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { NG_BASE_DEF } from '../render3/fields';
import { compileComponent as render3CompileComponent, compileDirective as render3CompileDirective } from '../render3/jit/directive';
import { compilePipe as render3CompilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
import { noop } from '../util/noop';
import { fillProperties } from '../util/property';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export var Directive = makeDecorator('Directive', function (dir) {
    if (dir === void 0) { dir = {}; }
    return dir;
}, undefined, undefined, function (type, meta) { return SWITCH_COMPILE_DIRECTIVE(type, meta); });
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export var Component = makeDecorator('Component', function (c) {
    if (c === void 0) { c = {}; }
    return (tslib_1.__assign({ changeDetection: ChangeDetectionStrategy.Default }, c));
}, Directive, undefined, function (type, meta) { return SWITCH_COMPILE_COMPONENT(type, meta); });
/**
 * @Annotation
 * @publicApi
 */
export var Pipe = makeDecorator('Pipe', function (p) { return (tslib_1.__assign({ pure: true }, p)); }, undefined, undefined, function (type, meta) { return SWITCH_COMPILE_PIPE(type, meta); });
var initializeBaseDef = function (target) {
    var constructor = target.constructor;
    var inheritedBaseDef = constructor.ngBaseDef;
    var baseDef = constructor.ngBaseDef = {
        inputs: {},
        outputs: {},
        declaredInputs: {},
    };
    if (inheritedBaseDef) {
        fillProperties(baseDef.inputs, inheritedBaseDef.inputs);
        fillProperties(baseDef.outputs, inheritedBaseDef.outputs);
        fillProperties(baseDef.declaredInputs, inheritedBaseDef.declaredInputs);
    }
};
var ɵ0 = initializeBaseDef;
/**
 * Does the work of creating the `ngBaseDef` property for the @Input and @Output decorators.
 * @param key "inputs" or "outputs"
 */
var updateBaseDefFromIOProp = function (getProp) {
    return function (target, name) {
        var args = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            args[_i - 2] = arguments[_i];
        }
        var constructor = target.constructor;
        if (!constructor.hasOwnProperty(NG_BASE_DEF)) {
            initializeBaseDef(target);
        }
        var baseDef = constructor.ngBaseDef;
        var defProp = getProp(baseDef);
        defProp[name] = args[0];
    };
};
var ɵ1 = updateBaseDefFromIOProp;
/**
 * @Annotation
 * @publicApi
 */
export var Input = makePropDecorator('Input', function (bindingPropertyName) { return ({ bindingPropertyName: bindingPropertyName }); }, undefined, updateBaseDefFromIOProp(function (baseDef) { return baseDef.inputs || {}; }));
/**
 * @Annotation
 * @publicApi
 */
export var Output = makePropDecorator('Output', function (bindingPropertyName) { return ({ bindingPropertyName: bindingPropertyName }); }, undefined, updateBaseDefFromIOProp(function (baseDef) { return baseDef.outputs || {}; }));
/**
 * @Annotation
 * @publicApi
 */
export var HostBinding = makePropDecorator('HostBinding', function (hostPropertyName) { return ({ hostPropertyName: hostPropertyName }); });
/**
 * Binds a CSS event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *  }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 * ```
 *
 * @Annotation
 * @publicApi
 */
export var HostListener = makePropDecorator('HostListener', function (eventName, args) { return ({ eventName: eventName, args: args }); });
export var SWITCH_COMPILE_COMPONENT__POST_R3__ = render3CompileComponent;
export var SWITCH_COMPILE_DIRECTIVE__POST_R3__ = render3CompileDirective;
export var SWITCH_COMPILE_PIPE__POST_R3__ = render3CompilePipe;
var SWITCH_COMPILE_COMPONENT__PRE_R3__ = noop;
var SWITCH_COMPILE_DIRECTIVE__PRE_R3__ = noop;
var SWITCH_COMPILE_PIPE__PRE_R3__ = noop;
var SWITCH_COMPILE_COMPONENT = SWITCH_COMPILE_COMPONENT__PRE_R3__;
var SWITCH_COMPILE_DIRECTIVE = SWITCH_COMPILE_DIRECTIVE__PRE_R3__;
var SWITCH_COMPILE_PIPE = SWITCH_COMPILE_PIPE__PRE_R3__;
export { ɵ0, ɵ1 };
//# sourceMappingURL=data:application/json;base64,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