"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Array_1 = require("./Array");
var function_1 = require("./function");
var Setoid_1 = require("./Setoid");
exports.URI = 'Tree';
/**
 * @since 1.6.0
 */
var Tree = /** @class */ (function () {
    function Tree(value, forest) {
        this.value = value;
        this.forest = forest;
    }
    Tree.prototype.map = function (f) {
        return new Tree(f(this.value), this.forest.map(function (tree) { return tree.map(f); }));
    };
    Tree.prototype.ap = function (fab) {
        var _this = this;
        return fab.chain(function (f) { return _this.map(f); }); // <- derived
    };
    /**
     * Flipped version of `ap`
     * @since 1.6.0
     */
    Tree.prototype.ap_ = function (fb) {
        return fb.ap(this);
    };
    Tree.prototype.chain = function (f) {
        var _a = f(this.value), value = _a.value, forest = _a.forest;
        return new Tree(value, function_1.concat(forest, this.forest.map(function (t) { return t.chain(f); })));
    };
    Tree.prototype.extract = function () {
        return this.value;
    };
    Tree.prototype.extend = function (f) {
        return new Tree(f(this), this.forest.map(function (t) { return t.extend(f); }));
    };
    Tree.prototype.reduce = function (b, f) {
        var r = f(b, this.value);
        var len = this.forest.length;
        for (var i = 0; i < len; i++) {
            r = this.forest[i].reduce(r, f);
        }
        return r;
    };
    Tree.prototype.inspect = function () {
        return this.toString();
    };
    Tree.prototype.toString = function () {
        return "new Tree(" + function_1.toString(this.value) + ", " + function_1.toString(this.forest) + ")";
    };
    return Tree;
}());
exports.Tree = Tree;
/**
 * @since 1.17.0
 */
exports.getShow = function (S) {
    var show = function (t) {
        return "new Tree(" + S.show(t.value) + ", [" + t.forest.map(show).join(', ') + "])";
    };
    return {
        show: show
    };
};
var map = function (fa, f) {
    return fa.map(f);
};
var of = function (a) {
    return new Tree(a, Array_1.empty);
};
var ap = function (fab, fa) {
    return fa.ap(fab);
};
var chain = function (fa, f) {
    return fa.chain(f);
};
var extract = function (fa) {
    return fa.extract();
};
var extend = function (fa, f) {
    return fa.extend(f);
};
var reduce = function (fa, b, f) {
    return fa.reduce(b, f);
};
var foldMap = function (M) { return function (fa, f) {
    return fa.reduce(M.empty, function (acc, a) { return M.concat(acc, f(a)); });
}; };
var foldr = function (fa, b, f) {
    var r = b;
    var len = fa.forest.length;
    for (var i = len - 1; i >= 0; i--) {
        r = foldr(fa.forest[i], r, f);
    }
    return f(fa.value, r);
};
function traverse(F) {
    var traverseF = Array_1.traverse(F);
    var r = function (ta, f) {
        return F.ap(F.map(f(ta.value), function (value) { return function (forest) { return new Tree(value, forest); }; }), traverseF(ta.forest, function (t) { return r(t, f); }));
    };
    return r;
}
function sequence(F) {
    var traverseF = traverse(F);
    return function (ta) { return traverseF(ta, function_1.identity); };
}
/**
 * @since 1.6.0
 */
exports.getSetoid = function (S) {
    var SA;
    var R = Setoid_1.fromEquals(function (x, y) { return S.equals(x.value, y.value) && SA.equals(x.forest, y.forest); });
    SA = Array_1.getSetoid(R);
    return R;
};
/**
 * @since 1.6.0
 */
exports.tree = {
    URI: exports.URI,
    map: map,
    of: of,
    ap: ap,
    chain: chain,
    reduce: reduce,
    foldMap: foldMap,
    foldr: foldr,
    traverse: traverse,
    sequence: sequence,
    extract: extract,
    extend: extend
};
var draw = function (indentation, forest) {
    var r = '';
    var len = forest.length;
    var tree;
    for (var i = 0; i < len; i++) {
        tree = forest[i];
        var isLast = i === len - 1;
        r += indentation + (isLast ? '└' : '├') + '─ ' + tree.value;
        r += draw(indentation + (len > 1 && !isLast ? '│  ' : '   '), tree.forest);
    }
    return r;
};
/**
 * Neat 2-dimensional drawing of a forest
 *
 * @since 1.6.0
 */
exports.drawForest = function (forest) {
    return draw('\n', forest);
};
/**
 * Neat 2-dimensional drawing of a tree
 *
 * @example
 * import { Tree, drawTree, tree } from 'fp-ts/lib/Tree'
 *
 * const fa = new Tree('a', [
 *   tree.of('b'),
 *   tree.of('c'),
 *   new Tree('d', [tree.of('e'), tree.of('f')])
 * ])
 *
 * assert.strictEqual(drawTree(fa), `a
 * ├─ b
 * ├─ c
 * └─ d
 *    ├─ e
 *    └─ f`)
 *
 *
 * @since 1.6.0
 */
exports.drawTree = function (tree) {
    return tree.value + exports.drawForest(tree.forest);
};
/**
 * Build a tree from a seed value
 *
 * @since 1.6.0
 */
exports.unfoldTree = function (b, f) {
    var _a = f(b), a = _a[0], bs = _a[1];
    return new Tree(a, exports.unfoldForest(bs, f));
};
/**
 * Build a tree from a seed value
 *
 * @since 1.6.0
 */
exports.unfoldForest = function (bs, f) {
    return bs.map(function (b) { return exports.unfoldTree(b, f); });
};
function unfoldTreeM(M) {
    var unfoldForestMM = unfoldForestM(M);
    return function (b, f) { return M.chain(f(b), function (_a) {
        var a = _a[0], bs = _a[1];
        return M.chain(unfoldForestMM(bs, f), function (ts) { return M.of(new Tree(a, ts)); });
    }); };
}
exports.unfoldTreeM = unfoldTreeM;
function unfoldForestM(M) {
    var traverseM = Array_1.traverse(M);
    var unfoldTree;
    return function (bs, f) {
        // tslint:disable-next-line
        if (unfoldTree === undefined) {
            unfoldTree = unfoldTreeM(M);
        }
        return traverseM(bs, function (b) { return unfoldTree(b, f); });
    };
}
exports.unfoldForestM = unfoldForestM;
/**
 * @since 1.14.0
 */
function elem(S) {
    var go = function (a, fa) {
        if (S.equals(a, fa.value)) {
            return true;
        }
        return fa.forest.some(function (tree) { return go(a, tree); });
    };
    return go;
}
exports.elem = elem;
