import { Injectable, Component, Input, ViewEncapsulation, ViewChild, ComponentRef, TemplateRef, ComponentFactoryResolver, ViewContainerRef, ChangeDetectorRef, Directive, Renderer2, NgModule, InjectionToken } from '@angular/core';
import { ReplaySubject } from 'rxjs';
import { CommonModule } from '@angular/common';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIActions {
}
BlockUIActions.START = 'start';
BlockUIActions.STOP = 'stop';
BlockUIActions.UPDATE = 'update';
BlockUIActions.RESET = 'reset';
BlockUIActions.UNSUBSCRIBE = 'unsubscribe';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ BlockUIDefaultName = `block-ui-main`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIInstanceService {
    constructor() {
        this.blockUISettings = {};
        this.blockUIInstances = {};
        this.blockUISubject = new ReplaySubject();
        this.blockUIObservable = this.blockUISubject.asObservable();
        this.blockUIObservable.subscribe(this.blockUIMiddleware.bind(this));
    }
    /**
     * @return {?}
     */
    getSettings() {
        return this.blockUISettings;
    }
    /**
     * @param {?=} settings
     * @return {?}
     */
    updateSettings(settings = {}) {
        this.blockUISettings = Object.assign({}, this.blockUISettings, settings);
    }
    /**
     * @param {?=} name
     * @return {?}
     */
    decorate(name = BlockUIDefaultName) {
        const /** @type {?} */ blockUI = /** @type {?} */ ({
            name,
            isActive: false,
            blockCount: 0,
            start: this.dispatch(this.blockUISubject, BlockUIActions.START, name),
            update: this.dispatch(this.blockUISubject, BlockUIActions.UPDATE, name),
            stop: this.dispatch(this.blockUISubject, BlockUIActions.STOP, name),
            reset: this.dispatch(this.blockUISubject, BlockUIActions.RESET, name),
            unsubscribe: this.dispatch(this.blockUISubject, BlockUIActions.UNSUBSCRIBE, name)
        });
        this.blockUIInstances[name] = this.blockUIInstances[name] || blockUI;
        return blockUI;
    }
    /**
     * @return {?}
     */
    observe() {
        return this.blockUIObservable;
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    blockUIMiddleware({ action, name }) {
        let /** @type {?} */ isActive = null;
        switch (action) {
            case (BlockUIActions.START):
                isActive = true;
                break;
            case (BlockUIActions.STOP):
            case (BlockUIActions.RESET):
                isActive = false;
                break;
        }
        if (isActive !== null) {
            this.blockUIInstances[name].isActive = isActive;
        }
    }
    /**
     * @param {?} subject
     * @param {?} action
     * @param {?=} name
     * @return {?}
     */
    dispatch(subject, action, name = BlockUIDefaultName) {
        return (message) => {
            subject.next({
                name,
                action,
                message
            });
        };
    }
}
BlockUIInstanceService.decorators = [
    { type: Injectable },
];
/** @nocollapse */
BlockUIInstanceService.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIComponent {
    /**
     * @param {?} blockUI
     */
    constructor(blockUI) {
        this.blockUI = blockUI;
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.name = this.name || BlockUIDefaultName;
        this.template = this.template || this.blockUI.blockUISettings.template;
    }
}
BlockUIComponent.decorators = [
    { type: Component, args: [{
                selector: 'block-ui',
                template: `
    <ng-content></ng-content>
    <block-ui-content
      [name]="name"
      [message]="message"
      [template]="template"
      [delayStart]="delayStart"
      [delayStop]="delayStop"
    >
    </block-ui-content>
  `,
                encapsulation: ViewEncapsulation.None
            },] },
];
/** @nocollapse */
BlockUIComponent.ctorParameters = () => [
    { type: BlockUIInstanceService }
];
BlockUIComponent.propDecorators = {
    name: [{ type: Input }],
    message: [{ type: Input }],
    delayStart: [{ type: Input }],
    delayStop: [{ type: Input }],
    template: [{ type: Input }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
// Spinner style - https://github.com/lukehaas/css-loaders
const /** @type {?} */ styles = `
.block-ui-wrapper {
  display: none;
  position: fixed;
  height: 100%;
  width: 100%;
  top: 0;
  left: 0;
  background: rgba(0, 0, 0, 0.70);
  z-index: 30000;
  cursor: wait;
}

.block-ui-wrapper.block-ui-wrapper--element {
  position: absolute;
}

.block-ui-wrapper.active {
  display: block;
}

.block-ui-wrapper.block-ui-main {
  position: fixed;
}

.block-ui-spinner,
.block-ui-template {
  position: absolute;
  top: 40%;
  margin: 0 auto;
  left: 0;
  right: 0;
  transform: translateY(-50%);
}

.block-ui-spinner > .message {
  font-size: 1.3em;
  text-align: center;
  color: #fff;
}

.block-ui__element {
  position: relative;
}

.loader,
.loader:after {
  border-radius: 50%;
  width: 10em;
  height: 10em;
}
.loader {
  margin: 7px auto;
  font-size: 5px;
  position: relative;
  text-indent: -9999em;
  border-top: 1.1em solid rgba(255, 255, 255, 0.2);
  border-right: 1.1em solid rgba(255, 255, 255, 0.2);
  border-bottom: 1.1em solid rgba(255, 255, 255, 0.2);
  border-left: 1.1em solid #ffffff;
  -webkit-transform: translateZ(0);
  -ms-transform: translateZ(0);
  transform: translateZ(0);
  -webkit-animation: load8 1.1s infinite linear;
  animation: load8 1.1s infinite linear;
}

@-webkit-keyframes load8 {
  0% {
    -webkit-transform: rotate(0deg);
    transform: rotate(0deg);
  }
  100% {
    -webkit-transform: rotate(360deg);
    transform: rotate(360deg);
  }
}

@keyframes load8 {
  0% {
    -webkit-transform: rotate(0deg);
    transform: rotate(0deg);
  }
  100% {
    -webkit-transform: rotate(360deg);
    transform: rotate(360deg);
  }
}
`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ template = `
<div class="block-ui-wrapper {{name}} {{className}}" [ngClass]="{ 'active': active }">
  <div class="block-ui-spinner" *ngIf="!templateCmp">
    <div class="loader"></div>
    <div *ngIf="message || defaultMessage" class="message">
      {{ message || defaultMessage }}
    </div>
  </div>
  <ng-template *ngIf="templateCmp" #templateOutlet></ng-template>
</div>
`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIContentComponent {
    /**
     * @param {?} blockUI
     * @param {?} resolver
     * @param {?} changeDetectionRef
     */
    constructor(blockUI, resolver, changeDetectionRef) {
        this.blockUI = blockUI;
        this.resolver = resolver;
        this.changeDetectionRef = changeDetectionRef;
        this.name = BlockUIDefaultName;
        this.delayStart = 0;
        this.delayStop = 0;
        this.state = { startTimeout: null, stopTimeout: null, blockCount: 0 };
        this.active = false;
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.settings = this.blockUI.getSettings();
        this.blockUISubscription = this.subscribeToBlockUI(this.blockUI.observe());
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        try {
            if (!this.templateCmp) {
                return false;
            }
            if (this.templateCmp instanceof TemplateRef) {
                this.templateOutlet.createEmbeddedView(this.templateCmp);
            }
            else {
                const /** @type {?} */ templateComp = this.resolver.resolveComponentFactory(this.templateCmp);
                this.templateCompRef = this.templateOutlet.createComponent(templateComp);
                this.updateBlockTemplate(this.message);
            }
        }
        catch (/** @type {?} */ error) {
            console.error('ng-block-ui:', error);
        }
    }
    /**
     * @return {?}
     */
    ngAfterViewChecked() {
        this.changeDetectionRef.detectChanges();
    }
    /**
     * @param {?} blockUI$
     * @return {?}
     */
    subscribeToBlockUI(blockUI$) {
        return blockUI$
            .subscribe(event => this.onDispatchedEvent(event));
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onDispatchedEvent(event) {
        switch (event.action) {
            case (BlockUIActions.START):
                this.onStart(event);
                break;
            case (BlockUIActions.STOP):
                this.onStop(event);
                break;
            case (BlockUIActions.UPDATE):
                this.onUpdate(event);
                break;
            case (BlockUIActions.RESET):
                this.onReset();
                break;
            case (BlockUIActions.UNSUBSCRIBE):
                this.onStop(event);
                this.onUnsubscribe(event.name);
                break;
        }
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onStart({ name, message }) {
        if (name === this.name) {
            const /** @type {?} */ delay = this.delayStart || this.settings.delayStart || 0;
            if (delay) {
                if (this.state.startTimeout === null) {
                    this.state.startTimeout = setTimeout(() => {
                        this.showBlock(message);
                    }, delay);
                }
                this.state.blockCount++;
            }
            else {
                this.showBlock(message);
            }
            this.updateInstanceBlockCount();
        }
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onStop({ name, action }) {
        if (name === this.name) {
            if (this.state.blockCount > 1) {
                this.state.blockCount--;
            }
            else {
                if (!this.active) {
                    this.clearState();
                }
                else {
                    const /** @type {?} */ delay = this.delayStop || this.settings.delayStop || 0;
                    if (delay) {
                        if (this.state.stopTimeout === null) {
                            this.state.stopTimeout = setTimeout(() => {
                                this.hideBlock();
                            }, delay);
                        }
                    }
                    else {
                        this.hideBlock();
                    }
                }
            }
            this.updateInstanceBlockCount();
        }
    }
    /**
     * @return {?}
     */
    onReset() {
        this.hideBlock();
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onUpdate({ name, message }) {
        if (name === this.name) {
            this.active = true;
            this.message = message || this.defaultMessage || this.settings.message;
            this.updateBlockTemplate(this.message);
            this.changeDetectionRef.detectChanges();
        }
    }
    /**
     * @param {?} message
     * @return {?}
     */
    showBlock(message) {
        this.active = true;
        this.message = message || this.defaultMessage || this.settings.message;
        this.updateBlockTemplate(this.message);
        this.changeDetectionRef.detectChanges();
    }
    /**
     * @return {?}
     */
    hideBlock() {
        this.clearState();
        this.active = false;
        this.changeDetectionRef.detectChanges();
    }
    /**
     * @return {?}
     */
    clearState() {
        this.state.startTimeout != null && clearTimeout(this.state.startTimeout);
        this.state.stopTimeout != null && clearTimeout(this.state.stopTimeout);
        this.state.blockCount = 0;
        this.state.startTimeout = null;
        this.state.stopTimeout = null;
        this.updateInstanceBlockCount();
    }
    /**
     * @param {?} msg
     * @return {?}
     */
    updateBlockTemplate(msg) {
        if (this.templateCompRef && this.templateCompRef instanceof ComponentRef) {
            this.templateCompRef.instance.message = msg;
        }
    }
    /**
     * @param {?} name
     * @return {?}
     */
    onUnsubscribe(name) {
        if (this.blockUISubscription && name === this.name) {
            this.blockUISubscription.unsubscribe();
        }
    }
    /**
     * @return {?}
     */
    updateInstanceBlockCount() {
        if (this.blockUI.blockUIInstances[this.name]) {
            this.blockUI.blockUIInstances[this.name].blockCount = this.state.blockCount;
        }
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        this.onUnsubscribe(this.name);
    }
}
BlockUIContentComponent.decorators = [
    { type: Component, args: [{
                selector: 'block-ui-content',
                template: template,
                styles: [styles],
                // TODO: Find how to bundle styles for npm
                encapsulation: ViewEncapsulation.None
            },] },
];
/** @nocollapse */
BlockUIContentComponent.ctorParameters = () => [
    { type: BlockUIInstanceService },
    { type: ComponentFactoryResolver },
    { type: ChangeDetectorRef }
];
BlockUIContentComponent.propDecorators = {
    name: [{ type: Input }],
    delayStart: [{ type: Input }],
    delayStop: [{ type: Input }],
    defaultMessage: [{ type: Input, args: ['message',] }],
    templateCmp: [{ type: Input, args: ['template',] }],
    templateOutlet: [{ type: ViewChild, args: ['templateOutlet', { read: ViewContainerRef },] }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIService {
    /**
     * @param {?} blockUIInstance
     */
    constructor(blockUIInstance) {
        this.blockUIInstance = blockUIInstance;
    }
    /**
     * Starts blocking for given BlockUI instance or instances
     * @param {?} target
     * @param {?=} message
     * @return {?}
     */
    start(target, message) {
        this.dispatch(target, BlockUIActions.START, message);
    }
    /**
     * Stops blocking for given BlockUI instance or instances
     * @param {?} target
     * @return {?}
     */
    stop(target) {
        this.dispatch(target, BlockUIActions.STOP);
    }
    /**
     * Unsubscribes for given BlockUI instance or instances
     * @param {?} target
     * @return {?}
     */
    unsubscribe(target) {
        this.dispatch(target, BlockUIActions.UNSUBSCRIBE);
    }
    /**
     * Checks if BlockUI is actively blocking
     * @param {?=} target
     * @return {?}
     */
    isActive(target = null) {
        const /** @type {?} */ targets = target ? this.toArray(target) : null;
        const /** @type {?} */ instances = this.blockUIInstance.blockUIInstances;
        return Object.keys(instances).some((key) => {
            if (!targets) {
                return instances[key].isActive;
            }
            return targets.indexOf(instances[key].name) >= 0 && instances[key].isActive;
        });
    }
    /**
     * @param {?=} target
     * @param {?=} type
     * @param {?=} message
     * @return {?}
     */
    dispatch(target = [], type, message) {
        const /** @type {?} */ instances = this.toArray(target);
        instances.forEach(i => this.blockUIInstance.decorate(i)[type](message));
    }
    /**
     * @param {?=} target
     * @return {?}
     */
    toArray(target = []) {
        return typeof target === 'string' ? [target] : target;
    }
}
BlockUIService.decorators = [
    { type: Injectable },
];
/** @nocollapse */
BlockUIService.ctorParameters = () => [
    { type: BlockUIInstanceService }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIDirective {
    /**
     * @param {?} blockUIService
     * @param {?} viewRef
     * @param {?} templateRef
     * @param {?} renderer
     * @param {?} componentFactoryResolver
     */
    constructor(blockUIService, viewRef, templateRef, renderer, componentFactoryResolver) {
        this.blockUIService = blockUIService;
        this.viewRef = viewRef;
        this.templateRef = templateRef;
        this.renderer = renderer;
        this.componentFactoryResolver = componentFactoryResolver;
    }
    /**
     * @param {?} name
     * @return {?}
     */
    set blockUI(name) { this.blockTarget = name; }
    ;
    /**
     * @param {?} message
     * @return {?}
     */
    set blockUIMessage(message) { this.message = message; }
    ;
    /**
     * @param {?} template
     * @return {?}
     */
    set blockUITemplate(template) { this.template = template; }
    ;
    /**
     * @param {?} delayStart
     * @return {?}
     */
    set blockUIDelayStart(delayStart) {
        this.delayStart = delayStart ? Number(delayStart) : null;
    }
    ;
    /**
     * @param {?} delayStop
     * @return {?}
     */
    set blockUIDelayStop(delayStop) {
        this.delayStop = delayStop ? Number(delayStop) : null;
    }
    ;
    /**
     * @return {?}
     */
    ngOnInit() {
        try {
            this.viewRef.createEmbeddedView(this.templateRef);
            const /** @type {?} */ parentElement = this.viewRef.element.nativeElement.nextSibling;
            if (parentElement && !this.isComponentInTemplate(parentElement)) {
                this.renderer.addClass(parentElement, 'block-ui__element');
                this.blockUIComponentRef = this.createComponent();
                let /** @type {?} */ blockUIContent = this.findContentNode(this.viewRef.element.nativeElement);
                if (blockUIContent) {
                    const /** @type {?} */ settings = this.blockUIService.getSettings();
                    parentElement.appendChild(blockUIContent);
                    this.blockUIComponentRef.instance.className = 'block-ui-wrapper--element';
                    this.blockUIComponentRef.instance.name = this.blockTarget || BlockUIDefaultName;
                    if (this.message)
                        this.blockUIComponentRef.instance.defaultMessage = this.message;
                    if (this.delayStart)
                        this.blockUIComponentRef.instance.delayStart = this.delayStart;
                    if (this.delayStop)
                        this.blockUIComponentRef.instance.delayStop = this.delayStop;
                    if (this.template || settings.template)
                        this.blockUIComponentRef.instance.templateCmp = this.template || settings.template;
                }
            }
        }
        catch (/** @type {?} */ error) {
            console.error('ng-block-ui:', error);
        }
    }
    /**
     * @param {?} element
     * @return {?}
     */
    isComponentInTemplate(element) {
        let { children } = element || [];
        children = Array.from(children).reverse();
        return children.some((el) => el.localName === 'block-ui');
    }
    /**
     * @param {?} element
     * @return {?}
     */
    findContentNode(element) {
        const { nextSibling } = element;
        return [nextSibling, nextSibling.nextSibling].find((e) => e.localName === 'block-ui-content');
    }
    /**
     * @return {?}
     */
    createComponent() {
        const /** @type {?} */ resolvedBlockUIComponent = this.componentFactoryResolver.resolveComponentFactory(BlockUIContentComponent);
        return this.viewRef.createComponent(resolvedBlockUIComponent);
    }
}
BlockUIDirective.decorators = [
    { type: Directive, args: [{ selector: '[blockUI]' },] },
];
/** @nocollapse */
BlockUIDirective.ctorParameters = () => [
    { type: BlockUIInstanceService },
    { type: ViewContainerRef },
    { type: TemplateRef },
    { type: Renderer2 },
    { type: ComponentFactoryResolver }
];
BlockUIDirective.propDecorators = {
    blockUI: [{ type: Input }],
    blockUIMessage: [{ type: Input }],
    blockUITemplate: [{ type: Input }],
    blockUIDelayStart: [{ type: Input }],
    blockUIDelayStop: [{ type: Input }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ BlockUIServiceInstance = new BlockUIInstanceService();
// Needed for AOT compiling
const /** @type {?} */ BlockUIModuleSettings = new InjectionToken('BlockUIModuleSettings');
/**
 * @param {?} settings
 * @return {?}
 */
function provideInstance(settings) {
    BlockUIServiceInstance.updateSettings(settings);
    return BlockUIServiceInstance;
}
class BlockUIModule {
    /**
     * @param {?=} settings
     * @return {?}
     */
    static forRoot(settings = {}) {
        return {
            ngModule: BlockUIModule,
            providers: [
                {
                    provide: BlockUIModuleSettings,
                    useValue: settings
                },
                {
                    provide: BlockUIInstanceService,
                    useFactory: provideInstance,
                    deps: [BlockUIModuleSettings]
                },
                BlockUIService
            ]
        };
    }
}
BlockUIModule.decorators = [
    { type: NgModule, args: [{
                imports: [
                    CommonModule
                ],
                entryComponents: [
                    BlockUIComponent,
                    BlockUIContentComponent
                ],
                declarations: [
                    BlockUIComponent,
                    BlockUIDirective,
                    BlockUIContentComponent
                ],
                exports: [
                    BlockUIComponent,
                    BlockUIDirective,
                    BlockUIContentComponent
                ]
            },] },
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @param {?=} value
 * @return {?}
 */
function BlockUI(value) {
    return function (target, propertyKey, descriptor) {
        target[propertyKey] = BlockUIServiceInstance.decorate(value);
    };
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

export { BlockUIModule, BlockUIComponent, BlockUIContentComponent, BlockUI, BlockUIService, BlockUIDefaultName as BLOCKUI_DEFAULT, BlockUIModuleSettings as ɵa, provideInstance as ɵb, styles as ɵe, template as ɵd, BlockUIDirective as ɵf, BlockUIInstanceService as ɵc };

//# sourceMappingURL=data:application/json;charset=utf-8;base64,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