"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adoptToHapiOnPreAuthFormat = adoptToHapiOnPreAuthFormat;

var _boom = _interopRequireDefault(require("boom"));

var _router = require("../router");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var ResultType;

(function (ResultType) {
  ResultType["next"] = "next";
  ResultType["redirected"] = "redirected";
  ResultType["rejected"] = "rejected";
})(ResultType || (ResultType = {}));

const preAuthResult = {
  next() {
    return {
      type: ResultType.next
    };
  },

  redirected(url, options = {}) {
    return {
      type: ResultType.redirected,
      url,
      forward: options.forward
    };
  },

  rejected(error, options = {}) {
    return {
      type: ResultType.rejected,
      error,
      statusCode: options.statusCode
    };
  },

  isValid(candidate) {
    return candidate && (candidate.type === ResultType.next || candidate.type === ResultType.rejected || candidate.type === ResultType.redirected);
  },

  isNext(result) {
    return result.type === ResultType.next;
  },

  isRedirected(result) {
    return result.type === ResultType.redirected;
  },

  isRejected(result) {
    return result.type === ResultType.rejected;
  }

};
/**
 * @public
 * A tool set defining an outcome of OnPreAuth interceptor for incoming request.
 */

const toolkit = {
  next: preAuthResult.next,
  redirected: preAuthResult.redirected,
  rejected: preAuthResult.rejected
};
/** @public */

/**
 * @public
 * Adopt custom request interceptor to Hapi lifecycle system.
 * @param fn - an extension point allowing to perform custom logic for
 * incoming HTTP requests.
 */
function adoptToHapiOnPreAuthFormat(fn) {
  return async function interceptPreAuthRequest(request, h) {
    try {
      const result = await fn(_router.KibanaRequest.from(request), toolkit);

      if (!preAuthResult.isValid(result)) {
        throw new Error(`Unexpected result from OnPreAuth. Expected OnPreAuthResult, but given: ${result}.`);
      }

      if (preAuthResult.isNext(result)) {
        return h.continue;
      }

      if (preAuthResult.isRedirected(result)) {
        const {
          url,
          forward
        } = result;

        if (forward) {
          request.setUrl(url); // We should update raw request as well since it can be proxied to the old platform

          request.raw.req.url = url;
          return h.continue;
        }

        return h.redirect(url).takeover();
      }

      const {
        error,
        statusCode
      } = result;
      return _boom.default.boomify(error, {
        statusCode
      });
    } catch (error) {
      return _boom.default.internal(error.message, {
        statusCode: 500
      });
    }
  };
}