"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _commit = require("../commit");

var _branchSet = _interopRequireDefault(require("../branch-set"));

var _remoteSet = _interopRequireDefault(require("../remote-set"));

var _operationStates = require("../operation-states");

var _multiFilePatch = _interopRequireDefault(require("../patch/multi-file-patch"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

/**
 * Map of registered subclasses to allow states to transition to one another without circular dependencies.
 * Subclasses of State should call `State.register` to add themselves here.
 */
const stateConstructors = new Map();
/**
 * Base class for Repository states. Implements default "null" behavior.
 */

class State {
  constructor(repository) {
    this.repository = repository;
  }

  static register(Subclass) {
    stateConstructors.set(Subclass.name, Subclass);
  } // This state has just been entered. Perform any asynchronous initialization that needs to occur.


  start() {
    return Promise.resolve();
  } // State probe predicates ////////////////////////////////////////////////////////////////////////////////////////////
  // Allow external callers to identify which state a Repository is in if necessary.


  isLoadingGuess() {
    return false;
  }

  isAbsentGuess() {
    return false;
  }

  isAbsent() {
    return false;
  }

  isLoading() {
    return false;
  }

  isEmpty() {
    return false;
  }

  isPresent() {
    return false;
  }

  isTooLarge() {
    return false;
  }

  isDestroyed() {
    return false;
  } // Behavior probe predicates /////////////////////////////////////////////////////////////////////////////////////////
  // Determine specific rendering behavior based on the current state.


  isUndetermined() {
    return false;
  }

  showGitTabInit() {
    return false;
  }

  showGitTabInitInProgress() {
    return false;
  }

  showGitTabLoading() {
    return false;
  }

  showStatusBarTiles() {
    return false;
  }

  hasDirectory() {
    return true;
  } // Lifecycle actions /////////////////////////////////////////////////////////////////////////////////////////////////
  // These generally default to rejecting a Promise with an error.


  init() {
    return unsupportedOperationPromise(this, 'init');
  }

  clone(remoteUrl) {
    return unsupportedOperationPromise(this, 'clone');
  }

  destroy() {
    return this.transitionTo('Destroyed');
  }
  /* istanbul ignore next */


  refresh() {} // No-op

  /* istanbul ignore next */


  observeFilesystemChange(events) {
    this.repository.refresh();
  }
  /* istanbul ignore next */


  updateCommitMessageAfterFileSystemChange() {
    // this is only used in unit tests, we don't need no stinkin coverage
    this.repository.refresh();
  } // Git operations ////////////////////////////////////////////////////////////////////////////////////////////////////
  // These default to rejecting a Promise with an error stating that the operation is not supported in the current
  // state.
  // Staging and unstaging


  stageFiles(paths) {
    return unsupportedOperationPromise(this, 'stageFiles');
  }

  unstageFiles(paths) {
    return unsupportedOperationPromise(this, 'unstageFiles');
  }

  stageFilesFromParentCommit(paths) {
    return unsupportedOperationPromise(this, 'stageFilesFromParentCommit');
  }

  applyPatchToIndex(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToIndex');
  }

  applyPatchToWorkdir(patch) {
    return unsupportedOperationPromise(this, 'applyPatchToWorkdir');
  } // Committing


  commit(message, options) {
    return unsupportedOperationPromise(this, 'commit');
  } // Merging


  merge(branchName) {
    return unsupportedOperationPromise(this, 'merge');
  }

  abortMerge() {
    return unsupportedOperationPromise(this, 'abortMerge');
  }

  checkoutSide(side, paths) {
    return unsupportedOperationPromise(this, 'checkoutSide');
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    return unsupportedOperationPromise(this, 'mergeFile');
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    return unsupportedOperationPromise(this, 'writeMergeConflictToIndex');
  } // Checkout


  checkout(revision, options = {}) {
    return unsupportedOperationPromise(this, 'checkout');
  }

  checkoutPathsAtRevision(paths, revision = 'HEAD') {
    return unsupportedOperationPromise(this, 'checkoutPathsAtRevision');
  } // Reset


  undoLastCommit() {
    return unsupportedOperationPromise(this, 'undoLastCommit');
  } // Remote interactions


  fetch(branchName) {
    return unsupportedOperationPromise(this, 'fetch');
  }

  pull(branchName) {
    return unsupportedOperationPromise(this, 'pull');
  }

  push(branchName) {
    return unsupportedOperationPromise(this, 'push');
  } // Configuration


  setConfig(option, value, {
    replaceAll
  } = {}) {
    return unsupportedOperationPromise(this, 'setConfig');
  }

  unsetConfig(option) {
    return unsupportedOperationPromise(this, 'unsetConfig');
  } // Direct blob interactions


  createBlob({
    filePath,
    stdin
  } = {}) {
    return unsupportedOperationPromise(this, 'createBlob');
  }

  expandBlobToFile(absFilePath, sha) {
    return unsupportedOperationPromise(this, 'expandBlobToFile');
  } // Discard history


  createDiscardHistoryBlob() {
    return unsupportedOperationPromise(this, 'createDiscardHistoryBlob');
  }

  updateDiscardHistory() {
    return unsupportedOperationPromise(this, 'updateDiscardHistory');
  }

  storeBeforeAndAfterBlobs(filePaths, isSafe, destructiveAction, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'storeBeforeAndAfterBlobs');
  }

  restoreLastDiscardInTempFiles(isSafe, partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'restoreLastDiscardInTempFiles');
  }

  popDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'popDiscardHistory');
  }

  clearDiscardHistory(partialDiscardFilePath = null) {
    return unsupportedOperationPromise(this, 'clearDiscardHistory');
  }

  discardWorkDirChangesForPaths(paths) {
    return unsupportedOperationPromise(this, 'discardWorkDirChangesForPaths');
  } // Accessors /////////////////////////////////////////////////////////////////////////////////////////////////////////
  // When possible, these default to "empty" results when invoked in states that don't have information available, or
  // fail in a way that's consistent with the requested information not being found.
  // Index queries


  getStatusBundle() {
    return Promise.resolve({
      stagedFiles: {},
      unstagedFiles: {},
      mergeConflictFiles: {},
      branch: {
        oid: null,
        head: null,
        upstream: null,
        aheadBehind: {
          ahead: null,
          behind: null
        }
      }
    });
  }

  getStatusesForChangedFiles() {
    return Promise.resolve({
      stagedFiles: [],
      unstagedFiles: [],
      mergeConflictFiles: []
    });
  }

  getFilePatchForPath(filePath, options = {}) {
    return Promise.resolve(_multiFilePatch["default"].createNull());
  }

  getDiffsForFilePath(filePath, options = {}) {
    return Promise.resolve([]);
  }

  getStagedChangesPatch() {
    return Promise.resolve(_multiFilePatch["default"].createNull());
  }

  readFileFromIndex(filePath) {
    return Promise.reject(new Error(`fatal: Path ${filePath} does not exist (neither on disk nor in the index).`));
  } // Commit access


  getLastCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getCommit() {
    return Promise.resolve(_commit.nullCommit);
  }

  getRecentCommits() {
    return Promise.resolve([]);
  }

  isCommitPushed(sha) {
    return false;
  } // Author information


  getAuthors() {
    return Promise.resolve([]);
  } // Branches


  getBranches() {
    return Promise.resolve(new _branchSet["default"]());
  }

  getHeadDescription() {
    return Promise.resolve('(no repository)');
  } // Merging and rebasing status


  isMerging() {
    return Promise.resolve(false);
  }

  isRebasing() {
    return Promise.resolve(false);
  } // Remotes


  getRemotes() {
    return Promise.resolve(new _remoteSet["default"]([]));
  }

  addRemote() {
    return unsupportedOperationPromise(this, 'addRemote');
  }

  getAheadCount(branchName) {
    return Promise.resolve(null);
  }

  getBehindCount(branchName) {
    return Promise.resolve(null);
  }

  getConfig(option, {
    local
  } = {}) {
    return Promise.resolve(null);
  } // Direct blob access


  getBlobContents(sha) {
    return Promise.reject(new Error(`fatal: Not a valid object name ${sha}`));
  } // Discard history


  hasDiscardHistory(partialDiscardFilePath = null) {
    return false;
  }

  getDiscardHistory(partialDiscardFilePath = null) {
    return [];
  }

  getLastHistorySnapshots(partialDiscardFilePath = null) {
    return null;
  } // Atom repo state


  getOperationStates() {
    return _operationStates.nullOperationStates;
  }

  setCommitMessage(message) {
    return unsupportedOperationPromise(this, 'setCommitMessage');
  }

  getCommitMessage() {
    return '';
  }

  fetchCommitMessageTemplate() {
    return unsupportedOperationPromise(this, 'fetchCommitMessageTemplate');
  } // Cache


  getCache() {
    return null;
  } // Internal //////////////////////////////////////////////////////////////////////////////////////////////////////////
  // Non-delegated methods that provide subclasses with convenient access to Repository properties.


  git() {
    return this.repository.git;
  }

  workdir() {
    return this.repository.getWorkingDirectoryPath();
  } // Call methods on the active Repository state, even if the state has transitioned beneath you.
  // Use this to perform operations within `start()` methods to guard against interrupted state transitions.


  current() {
    return this.repository.state;
  } // pipeline


  executePipelineAction(...args) {
    return this.repository.executePipelineAction(...args);
  } // Return a Promise that will resolve once the state transitions from Loading.


  getLoadPromise() {
    return this.repository.getLoadPromise();
  }

  getRemoteForBranch(branchName) {
    return this.repository.getRemoteForBranch(branchName);
  }

  saveDiscardHistory() {
    return this.repository.saveDiscardHistory();
  } // Initiate a transition to another state.


  transitionTo(stateName, ...payload) {
    const StateConstructor = stateConstructors.get(stateName);
    /* istanbul ignore if */

    if (StateConstructor === undefined) {
      throw new Error(`Attempt to transition to unrecognized state ${stateName}`);
    }

    return this.repository.transition(this, StateConstructor, ...payload);
  } // Event broadcast


  didDestroy() {
    return this.repository.emitter.emit('did-destroy');
  }

  didUpdate() {
    return this.repository.emitter.emit('did-update');
  } // Direct git access
  // Non-delegated git operations for internal use within states.

  /* istanbul ignore next */


  directResolveDotGitDir() {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetConfig(key, options = {}) {
    return Promise.resolve(null);
  }
  /* istanbul ignore next */


  directGetBlobContents() {
    return Promise.reject(new Error('Not a valid object name'));
  }
  /* istanbul ignore next */


  directInit() {
    return Promise.resolve();
  }
  /* istanbul ignore next */


  directClone(remoteUrl, options) {
    return Promise.resolve();
  } // Deferred operations
  // Direct raw git operations to the current state, even if the state has been changed. Use these methods within
  // start() methods.


  resolveDotGitDir() {
    return this.current().directResolveDotGitDir();
  }

  doInit(workdir) {
    return this.current().directInit();
  }

  doClone(remoteUrl, options) {
    return this.current().directClone(remoteUrl, options);
  } // Parse a DiscardHistory payload from the SHA recorded in config.


  async loadHistoryPayload() {
    const historySha = await this.current().directGetConfig('atomGithub.historySha');

    if (!historySha) {
      return {};
    }

    let blob;

    try {
      blob = await this.current().directGetBlobContents(historySha);
    } catch (e) {
      if (/Not a valid object name/.test(e.stdErr)) {
        return {};
      }

      throw e;
    }

    try {
      return JSON.parse(blob);
    } catch (e) {
      return {};
    }
  } // Debugging assistance.


  toString() {
    return this.constructor.name;
  }

}

exports["default"] = State;

function unsupportedOperationPromise(self, opName) {
  return Promise.reject(new Error(`${opName} is not available in ${self} state`));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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