"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autobind = autobind;
exports.extractProps = extractProps;
exports.unusedProps = unusedProps;
exports.getPackageRoot = getPackageRoot;
exports.getAtomHelperPath = getAtomHelperPath;
exports.getDugitePath = getDugitePath;
exports.getSharedModulePath = getSharedModulePath;
exports.isBinary = isBinary;
exports.firstImplementer = firstImplementer;
exports.isValidWorkdir = isValidWorkdir;
exports.fileExists = fileExists;
exports.getTempDir = getTempDir;
exports.isFileExecutable = isFileExecutable;
exports.isFileSymlink = isFileSymlink;
exports.shortenSha = shortenSha;
exports.normalizeGitHelperPath = normalizeGitHelperPath;
exports.toNativePathSep = toNativePathSep;
exports.toGitPathSep = toGitPathSep;
exports.filePathEndsWith = filePathEndsWith;
exports.toSentence = toSentence;
exports.pushAtKey = pushAtKey;
exports.getCommitMessagePath = getCommitMessagePath;
exports.getCommitMessageEditors = getCommitMessageEditors;
exports.getFilePatchPaneItems = getFilePatchPaneItems;
exports.destroyFilePatchPaneItems = destroyFilePatchPaneItems;
exports.destroyEmptyFilePatchPaneItems = destroyEmptyFilePatchPaneItems;
exports.extractCoAuthorsAndRawCommitMessage = extractCoAuthorsAndRawCommitMessage;
exports.createItem = createItem;
exports.equalSets = equalSets;
exports.blankLabel = blankLabel;
exports.renderMarkdown = renderMarkdown;
exports.GHOST_USER = exports.reactionTypeToEmoji = exports.NBSP_CHARACTER = exports.classNameForStatus = exports.CHECK_RUN_PAGE_SIZE = exports.CHECK_SUITE_PAGE_SIZE = exports.PAGINATION_WAIT_TIME_MS = exports.PAGE_SIZE = exports.CO_AUTHOR_REGEX = exports.LINE_ENDING_REGEX = void 0;

var _path = _interopRequireDefault(require("path"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _os = _interopRequireDefault(require("os"));

var _temp = _interopRequireDefault(require("temp"));

var _refHolder = _interopRequireDefault(require("./models/ref-holder"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const LINE_ENDING_REGEX = /\r?\n/;
exports.LINE_ENDING_REGEX = LINE_ENDING_REGEX;
const CO_AUTHOR_REGEX = /^co-authored-by. (.+?) <(.+?)>$/i;
exports.CO_AUTHOR_REGEX = CO_AUTHOR_REGEX;
const PAGE_SIZE = 50;
exports.PAGE_SIZE = PAGE_SIZE;
const PAGINATION_WAIT_TIME_MS = 100;
exports.PAGINATION_WAIT_TIME_MS = PAGINATION_WAIT_TIME_MS;
const CHECK_SUITE_PAGE_SIZE = 10;
exports.CHECK_SUITE_PAGE_SIZE = CHECK_SUITE_PAGE_SIZE;
const CHECK_RUN_PAGE_SIZE = 20;
exports.CHECK_RUN_PAGE_SIZE = CHECK_RUN_PAGE_SIZE;

function autobind(self, ...methods) {
  for (const method of methods) {
    if (typeof self[method] !== 'function') {
      throw new Error(`Unable to autobind method ${method}`);
    }

    self[method] = self[method].bind(self);
  }
} // Extract a subset of props chosen from a propTypes object from a component's props to pass to a different API.
//
// Usage:
//
// ```js
// const apiProps = {
//   zero: PropTypes.number.isRequired,
//   one: PropTypes.string,
//   two: PropTypes.object,
// };
//
// class Component extends React.Component {
//   static propTypes = {
//     ...apiProps,
//     extra: PropTypes.func,
//   }
//
//   action() {
//     const options = extractProps(this.props, apiProps);
//     // options contains zero, one, and two, but not extra
//   }
// }
// ```


function extractProps(props, propTypes, nameMap = {}) {
  return Object.keys(propTypes).reduce((opts, propName) => {
    if (props[propName] !== undefined) {
      const destPropName = nameMap[propName] || propName;
      opts[destPropName] = props[propName];
    }

    return opts;
  }, {});
} // The opposite of extractProps. Return a subset of props that do *not* appear in a component's prop types.


function unusedProps(props, propTypes) {
  return Object.keys(props).reduce((opts, propName) => {
    if (propTypes[propName] === undefined) {
      opts[propName] = props[propName];
    }

    return opts;
  }, {});
}

function getPackageRoot() {
  const {
    resourcePath
  } = atom.getLoadSettings();
  const currentFileWasRequiredFromSnapshot = !_path["default"].isAbsolute(__dirname);

  if (currentFileWasRequiredFromSnapshot) {
    return _path["default"].join(resourcePath, 'node_modules', 'github');
  } else {
    const packageRoot = _path["default"].resolve(__dirname, '..');

    if (_path["default"].extname(resourcePath) === '.asar') {
      if (packageRoot.indexOf(resourcePath) === 0) {
        return _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'github');
      }
    }

    return packageRoot;
  }
}

function getAtomAppName() {
  const match = atom.getVersion().match(/-([A-Za-z]+)(\d+|-)/);

  if (match) {
    const channel = match[1];
    return `Atom ${channel.charAt(0).toUpperCase() + channel.slice(1)} Helper`;
  }

  return 'Atom Helper';
}

function getAtomHelperPath() {
  if (process.platform === 'darwin') {
    const appName = getAtomAppName();
    return _path["default"].resolve(process.resourcesPath, '..', 'Frameworks', `${appName}.app`, 'Contents', 'MacOS', appName);
  } else {
    return process.execPath;
  }
}

let DUGITE_PATH;

function getDugitePath() {
  if (!DUGITE_PATH) {
    DUGITE_PATH = require.resolve('dugite');

    if (!_path["default"].isAbsolute(DUGITE_PATH)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();

      if (_path["default"].extname(resourcePath) === '.asar') {
        DUGITE_PATH = _path["default"].join(`${resourcePath}.unpacked`, 'node_modules', 'dugite');
      } else {
        DUGITE_PATH = _path["default"].join(resourcePath, 'node_modules', 'dugite');
      }
    }
  }

  return DUGITE_PATH;
}

const SHARED_MODULE_PATHS = new Map();

function getSharedModulePath(relPath) {
  let modulePath = SHARED_MODULE_PATHS.get(relPath);

  if (!modulePath) {
    modulePath = require.resolve(_path["default"].join(__dirname, 'shared', relPath));

    if (!_path["default"].isAbsolute(modulePath)) {
      // Assume we're snapshotted
      const {
        resourcePath
      } = atom.getLoadSettings();
      modulePath = _path["default"].join(resourcePath, modulePath);
    }

    SHARED_MODULE_PATHS.set(relPath, modulePath);
  }

  return modulePath;
}

function isBinary(data) {
  for (let i = 0; i < 50; i++) {
    const code = data.charCodeAt(i); // Char code 65533 is the "replacement character";
    // 8 and below are control characters.

    if (code === 65533 || code < 9) {
      return true;
    }
  }

  return false;
}

function descriptorsFromProto(proto) {
  return Object.getOwnPropertyNames(proto).reduce((acc, name) => {
    Object.assign(acc, {
      [name]: Reflect.getOwnPropertyDescriptor(proto, name)
    });
    return acc;
  }, {});
}
/**
 * Takes an array of targets and returns a proxy. The proxy intercepts property accessor calls and
 * returns the value of that property on the first object in `targets` where the target implements that property.
 */


function firstImplementer(...targets) {
  return new Proxy({
    __implementations: targets
  }, {
    get(target, name) {
      if (name === 'getImplementers') {
        return () => targets;
      }

      if (Reflect.has(target, name)) {
        return target[name];
      }

      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        return firstValidTarget[name];
      } else {
        return undefined;
      }
    },

    set(target, name, value) {
      const firstValidTarget = targets.find(t => Reflect.has(t, name));

      if (firstValidTarget) {
        // eslint-disable-next-line no-return-assign
        return firstValidTarget[name] = value;
      } else {
        // eslint-disable-next-line no-return-assign
        return target[name] = value;
      }
    },

    // Used by sinon
    has(target, name) {
      if (name === 'getImplementers') {
        return true;
      }

      return targets.some(t => Reflect.has(t, name));
    },

    // Used by sinon
    getOwnPropertyDescriptor(target, name) {
      const firstValidTarget = targets.find(t => Reflect.getOwnPropertyDescriptor(t, name));
      const compositeOwnPropertyDescriptor = Reflect.getOwnPropertyDescriptor(target, name);

      if (firstValidTarget) {
        return Reflect.getOwnPropertyDescriptor(firstValidTarget, name);
      } else if (compositeOwnPropertyDescriptor) {
        return compositeOwnPropertyDescriptor;
      } else {
        return undefined;
      }
    },

    // Used by sinon
    getPrototypeOf(target) {
      return targets.reduceRight((acc, t) => {
        return Object.create(acc, descriptorsFromProto(Object.getPrototypeOf(t)));
      }, Object.prototype);
    }

  });
}

function isRoot(dir) {
  return _path["default"].resolve(dir, '..') === dir;
}

function isValidWorkdir(dir) {
  return dir !== _os["default"].homedir() && !isRoot(dir);
}

async function fileExists(absoluteFilePath) {
  try {
    await _fsExtra["default"].access(absoluteFilePath);
    return true;
  } catch (e) {
    if (e.code === 'ENOENT') {
      return false;
    }

    throw e;
  }
}

function getTempDir(options = {}) {
  _temp["default"].track();

  return new Promise((resolve, reject) => {
    _temp["default"].mkdir(options, (tempError, folder) => {
      if (tempError) {
        reject(tempError);
        return;
      }

      if (options.symlinkOk) {
        resolve(folder);
      } else {
        _fsExtra["default"].realpath(folder, (realError, rpath) => realError ? reject(realError) : resolve(rpath));
      }
    });
  });
}

async function isFileExecutable(absoluteFilePath) {
  const stat = await _fsExtra["default"].stat(absoluteFilePath);
  return stat.mode & _fsExtra["default"].constants.S_IXUSR; // eslint-disable-line no-bitwise
}

async function isFileSymlink(absoluteFilePath) {
  const stat = await _fsExtra["default"].lstat(absoluteFilePath);
  return stat.isSymbolicLink();
}

function shortenSha(sha) {
  return sha.slice(0, 8);
}

const classNameForStatus = {
  added: 'added',
  deleted: 'removed',
  modified: 'modified',
  typechange: 'modified',
  equivalent: 'ignored'
};
/*
 * Apply any platform-specific munging to a path before presenting it as
 * a git environment variable or option.
 *
 * Convert a Windows-style "C:\foo\bar\baz" path to a "/c/foo/bar/baz" UNIX-y
 * path that the sh.exe used to execute git's credential helpers will
 * understand.
 */

exports.classNameForStatus = classNameForStatus;

function normalizeGitHelperPath(inPath) {
  if (process.platform === 'win32') {
    return inPath.replace(/\\/g, '/').replace(/^([^:]+):/, '/$1');
  } else {
    return inPath;
  }
}
/*
 * On Windows, git commands report paths with / delimiters. Convert them to \-delimited paths
 * so that Atom unifromly treats paths with native path separators.
 */


function toNativePathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split('/').join(_path["default"].sep);
  }
}
/*
 * Convert Windows paths back to /-delimited paths to be presented to git.
 */


function toGitPathSep(rawPath) {
  if (process.platform !== 'win32') {
    return rawPath;
  } else {
    return rawPath.split(_path["default"].sep).join('/');
  }
}

function filePathEndsWith(filePath, ...segments) {
  return filePath.endsWith(_path["default"].join(...segments));
}
/**
 * Turns an array of things @kuychaco cannot eat
 * into a sentence containing things @kuychaco cannot eat
 *
 * ['toast'] => 'toast'
 * ['toast', 'eggs'] => 'toast and eggs'
 * ['toast', 'eggs', 'cheese'] => 'toast, eggs, and cheese'
 *
 * Oxford comma included because you're wrong, shut up.
 */


function toSentence(array) {
  const len = array.length;

  if (len === 1) {
    return `${array[0]}`;
  } else if (len === 2) {
    return `${array[0]} and ${array[1]}`;
  }

  return array.reduce((acc, item, idx) => {
    if (idx === 0) {
      return `${item}`;
    } else if (idx === len - 1) {
      return `${acc}, and ${item}`;
    } else {
      return `${acc}, ${item}`;
    }
  }, '');
}

function pushAtKey(map, key, value) {
  let existing = map.get(key);

  if (!existing) {
    existing = [];
    map.set(key, existing);
  }

  existing.push(value);
} // Repository and workspace helpers


function getCommitMessagePath(repository) {
  return _path["default"].join(repository.getGitDirectoryPath(), 'ATOM_COMMIT_EDITMSG');
}

function getCommitMessageEditors(repository, workspace) {
  if (!repository.isPresent()) {
    return [];
  }

  return workspace.getTextEditors().filter(editor => editor.getPath() === getCommitMessagePath(repository));
}

let ChangedFileItem = null;

function getFilePatchPaneItems({
  onlyStaged,
  empty
} = {}, workspace) {
  if (ChangedFileItem === null) {
    ChangedFileItem = require('./items/changed-file-item')["default"];
  }

  return workspace.getPaneItems().filter(item => {
    const isFilePatchItem = item && item.getRealItem && item.getRealItem() instanceof ChangedFileItem;

    if (onlyStaged) {
      return isFilePatchItem && item.stagingStatus === 'staged';
    } else if (empty) {
      return isFilePatchItem ? item.isEmpty() : false;
    } else {
      return isFilePatchItem;
    }
  });
}

function destroyFilePatchPaneItems({
  onlyStaged
} = {}, workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    onlyStaged
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function destroyEmptyFilePatchPaneItems(workspace) {
  const itemsToDestroy = getFilePatchPaneItems({
    empty: true
  }, workspace);
  itemsToDestroy.forEach(item => item.destroy());
}

function extractCoAuthorsAndRawCommitMessage(commitMessage) {
  const messageLines = [];
  const coAuthors = [];

  for (const line of commitMessage.split(LINE_ENDING_REGEX)) {
    const match = line.match(CO_AUTHOR_REGEX);

    if (match) {
      // eslint-disable-next-line no-unused-vars
      const [_, name, email] = match;
      coAuthors.push({
        name,
        email
      });
    } else {
      messageLines.push(line);
    }
  }

  return {
    message: messageLines.join('\n'),
    coAuthors
  };
} // Atom API pane item manipulation


function createItem(node, componentHolder = null, uri = null, extra = {}) {
  const holder = componentHolder || new _refHolder["default"]();

  const override = _objectSpread({
    getElement: () => node,
    getRealItem: () => holder.getOr(null),
    getRealItemPromise: () => holder.getPromise()
  }, extra);

  if (uri) {
    override.getURI = () => uri;
  }

  if (componentHolder) {
    return new Proxy(override, {
      get(target, name) {
        if (Reflect.has(target, name)) {
          return target[name];
        } // The {value: ...} wrapper prevents .map() from flattening a returned RefHolder.
        // If component[name] is a RefHolder, we want to return that RefHolder as-is.


        const {
          value
        } = holder.map(component => ({
          value: component[name]
        })).getOr({
          value: undefined
        });
        return value;
      },

      set(target, name, value) {
        return holder.map(component => {
          component[name] = value;
          return true;
        }).getOr(true);
      },

      has(target, name) {
        return holder.map(component => Reflect.has(component, name)).getOr(false) || Reflect.has(target, name);
      }

    });
  } else {
    return override;
  }
} // Set functions


function equalSets(left, right) {
  if (left.size !== right.size) {
    return false;
  }

  for (const each of left) {
    if (!right.has(each)) {
      return false;
    }
  }

  return true;
} // Constants


const NBSP_CHARACTER = '\u00a0';
exports.NBSP_CHARACTER = NBSP_CHARACTER;

function blankLabel() {
  return NBSP_CHARACTER;
}

const reactionTypeToEmoji = {
  THUMBS_UP: '👍',
  THUMBS_DOWN: '👎',
  LAUGH: '😆',
  HOORAY: '🎉',
  CONFUSED: '😕',
  HEART: '❤️',
  ROCKET: '🚀',
  EYES: '👀'
}; // Markdown

exports.reactionTypeToEmoji = reactionTypeToEmoji;
let marked = null;
let domPurify = null;

function renderMarkdown(md) {
  if (marked === null) {
    marked = require('marked');

    if (domPurify === null) {
      const createDOMPurify = require('dompurify');

      domPurify = createDOMPurify();
    }

    marked.setOptions({
      silent: true,
      sanitize: true,
      sanitizer: html => domPurify.sanitize(html)
    });
  }

  return marked(md);
}

const GHOST_USER = {
  login: 'ghost',
  avatarUrl: 'https://avatars1.githubusercontent.com/u/10137?v=4',
  url: 'https://github.com/ghost'
};
exports.GHOST_USER = GHOST_USER;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImhlbHBlcnMuanMiXSwibmFtZXMiOlsiTElORV9FTkRJTkdfUkVHRVgiLCJDT19BVVRIT1JfUkVHRVgiLCJQQUdFX1NJWkUiLCJQQUdJTkFUSU9OX1dBSVRfVElNRV9NUyIsIkNIRUNLX1NVSVRFX1BBR0VfU0laRSIsIkNIRUNLX1JVTl9QQUdFX1NJWkUiLCJhdXRvYmluZCIsInNlbGYiLCJtZXRob2RzIiwibWV0aG9kIiwiRXJyb3IiLCJiaW5kIiwiZXh0cmFjdFByb3BzIiwicHJvcHMiLCJwcm9wVHlwZXMiLCJuYW1lTWFwIiwiT2JqZWN0Iiwia2V5cyIsInJlZHVjZSIsIm9wdHMiLCJwcm9wTmFtZSIsInVuZGVmaW5lZCIsImRlc3RQcm9wTmFtZSIsInVudXNlZFByb3BzIiwiZ2V0UGFja2FnZVJvb3QiLCJyZXNvdXJjZVBhdGgiLCJhdG9tIiwiZ2V0TG9hZFNldHRpbmdzIiwiY3VycmVudEZpbGVXYXNSZXF1aXJlZEZyb21TbmFwc2hvdCIsInBhdGgiLCJpc0Fic29sdXRlIiwiX19kaXJuYW1lIiwiam9pbiIsInBhY2thZ2VSb290IiwicmVzb2x2ZSIsImV4dG5hbWUiLCJpbmRleE9mIiwiZ2V0QXRvbUFwcE5hbWUiLCJtYXRjaCIsImdldFZlcnNpb24iLCJjaGFubmVsIiwiY2hhckF0IiwidG9VcHBlckNhc2UiLCJzbGljZSIsImdldEF0b21IZWxwZXJQYXRoIiwicHJvY2VzcyIsInBsYXRmb3JtIiwiYXBwTmFtZSIsInJlc291cmNlc1BhdGgiLCJleGVjUGF0aCIsIkRVR0lURV9QQVRIIiwiZ2V0RHVnaXRlUGF0aCIsInJlcXVpcmUiLCJTSEFSRURfTU9EVUxFX1BBVEhTIiwiTWFwIiwiZ2V0U2hhcmVkTW9kdWxlUGF0aCIsInJlbFBhdGgiLCJtb2R1bGVQYXRoIiwiZ2V0Iiwic2V0IiwiaXNCaW5hcnkiLCJkYXRhIiwiaSIsImNvZGUiLCJjaGFyQ29kZUF0IiwiZGVzY3JpcHRvcnNGcm9tUHJvdG8iLCJwcm90byIsImdldE93blByb3BlcnR5TmFtZXMiLCJhY2MiLCJuYW1lIiwiYXNzaWduIiwiUmVmbGVjdCIsImdldE93blByb3BlcnR5RGVzY3JpcHRvciIsImZpcnN0SW1wbGVtZW50ZXIiLCJ0YXJnZXRzIiwiUHJveHkiLCJfX2ltcGxlbWVudGF0aW9ucyIsInRhcmdldCIsImhhcyIsImZpcnN0VmFsaWRUYXJnZXQiLCJmaW5kIiwidCIsInZhbHVlIiwic29tZSIsImNvbXBvc2l0ZU93blByb3BlcnR5RGVzY3JpcHRvciIsImdldFByb3RvdHlwZU9mIiwicmVkdWNlUmlnaHQiLCJjcmVhdGUiLCJwcm90b3R5cGUiLCJpc1Jvb3QiLCJkaXIiLCJpc1ZhbGlkV29ya2RpciIsIm9zIiwiaG9tZWRpciIsImZpbGVFeGlzdHMiLCJhYnNvbHV0ZUZpbGVQYXRoIiwiZnMiLCJhY2Nlc3MiLCJlIiwiZ2V0VGVtcERpciIsIm9wdGlvbnMiLCJ0ZW1wIiwidHJhY2siLCJQcm9taXNlIiwicmVqZWN0IiwibWtkaXIiLCJ0ZW1wRXJyb3IiLCJmb2xkZXIiLCJzeW1saW5rT2siLCJyZWFscGF0aCIsInJlYWxFcnJvciIsInJwYXRoIiwiaXNGaWxlRXhlY3V0YWJsZSIsInN0YXQiLCJtb2RlIiwiY29uc3RhbnRzIiwiU19JWFVTUiIsImlzRmlsZVN5bWxpbmsiLCJsc3RhdCIsImlzU3ltYm9saWNMaW5rIiwic2hvcnRlblNoYSIsInNoYSIsImNsYXNzTmFtZUZvclN0YXR1cyIsImFkZGVkIiwiZGVsZXRlZCIsIm1vZGlmaWVkIiwidHlwZWNoYW5nZSIsImVxdWl2YWxlbnQiLCJub3JtYWxpemVHaXRIZWxwZXJQYXRoIiwiaW5QYXRoIiwicmVwbGFjZSIsInRvTmF0aXZlUGF0aFNlcCIsInJhd1BhdGgiLCJzcGxpdCIsInNlcCIsInRvR2l0UGF0aFNlcCIsImZpbGVQYXRoRW5kc1dpdGgiLCJmaWxlUGF0aCIsInNlZ21lbnRzIiwiZW5kc1dpdGgiLCJ0b1NlbnRlbmNlIiwiYXJyYXkiLCJsZW4iLCJsZW5ndGgiLCJpdGVtIiwiaWR4IiwicHVzaEF0S2V5IiwibWFwIiwia2V5IiwiZXhpc3RpbmciLCJwdXNoIiwiZ2V0Q29tbWl0TWVzc2FnZVBhdGgiLCJyZXBvc2l0b3J5IiwiZ2V0R2l0RGlyZWN0b3J5UGF0aCIsImdldENvbW1pdE1lc3NhZ2VFZGl0b3JzIiwid29ya3NwYWNlIiwiaXNQcmVzZW50IiwiZ2V0VGV4dEVkaXRvcnMiLCJmaWx0ZXIiLCJlZGl0b3IiLCJnZXRQYXRoIiwiQ2hhbmdlZEZpbGVJdGVtIiwiZ2V0RmlsZVBhdGNoUGFuZUl0ZW1zIiwib25seVN0YWdlZCIsImVtcHR5IiwiZ2V0UGFuZUl0ZW1zIiwiaXNGaWxlUGF0Y2hJdGVtIiwiZ2V0UmVhbEl0ZW0iLCJzdGFnaW5nU3RhdHVzIiwiaXNFbXB0eSIsImRlc3Ryb3lGaWxlUGF0Y2hQYW5lSXRlbXMiLCJpdGVtc1RvRGVzdHJveSIsImZvckVhY2giLCJkZXN0cm95IiwiZGVzdHJveUVtcHR5RmlsZVBhdGNoUGFuZUl0ZW1zIiwiZXh0cmFjdENvQXV0aG9yc0FuZFJhd0NvbW1pdE1lc3NhZ2UiLCJjb21taXRNZXNzYWdlIiwibWVzc2FnZUxpbmVzIiwiY29BdXRob3JzIiwibGluZSIsIl8iLCJlbWFpbCIsIm1lc3NhZ2UiLCJjcmVhdGVJdGVtIiwibm9kZSIsImNvbXBvbmVudEhvbGRlciIsInVyaSIsImV4dHJhIiwiaG9sZGVyIiwiUmVmSG9sZGVyIiwib3ZlcnJpZGUiLCJnZXRFbGVtZW50IiwiZ2V0T3IiLCJnZXRSZWFsSXRlbVByb21pc2UiLCJnZXRQcm9taXNlIiwiZ2V0VVJJIiwiY29tcG9uZW50IiwiZXF1YWxTZXRzIiwibGVmdCIsInJpZ2h0Iiwic2l6ZSIsImVhY2giLCJOQlNQX0NIQVJBQ1RFUiIsImJsYW5rTGFiZWwiLCJyZWFjdGlvblR5cGVUb0Vtb2ppIiwiVEhVTUJTX1VQIiwiVEhVTUJTX0RPV04iLCJMQVVHSCIsIkhPT1JBWSIsIkNPTkZVU0VEIiwiSEVBUlQiLCJST0NLRVQiLCJFWUVTIiwibWFya2VkIiwiZG9tUHVyaWZ5IiwicmVuZGVyTWFya2Rvd24iLCJtZCIsImNyZWF0ZURPTVB1cmlmeSIsInNldE9wdGlvbnMiLCJzaWxlbnQiLCJzYW5pdGl6ZSIsInNhbml0aXplciIsImh0bWwiLCJHSE9TVF9VU0VSIiwibG9naW4iLCJhdmF0YXJVcmwiLCJ1cmwiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0FBQUE7O0FBQ0E7O0FBQ0E7O0FBQ0E7O0FBRUE7Ozs7Ozs7O0FBRU8sTUFBTUEsaUJBQWlCLEdBQUcsT0FBMUI7O0FBQ0EsTUFBTUMsZUFBZSxHQUFHLGtDQUF4Qjs7QUFDQSxNQUFNQyxTQUFTLEdBQUcsRUFBbEI7O0FBQ0EsTUFBTUMsdUJBQXVCLEdBQUcsR0FBaEM7O0FBQ0EsTUFBTUMscUJBQXFCLEdBQUcsRUFBOUI7O0FBQ0EsTUFBTUMsbUJBQW1CLEdBQUcsRUFBNUI7OztBQUVBLFNBQVNDLFFBQVQsQ0FBa0JDLElBQWxCLEVBQXdCLEdBQUdDLE9BQTNCLEVBQW9DO0FBQ3pDLE9BQUssTUFBTUMsTUFBWCxJQUFxQkQsT0FBckIsRUFBOEI7QUFDNUIsUUFBSSxPQUFPRCxJQUFJLENBQUNFLE1BQUQsQ0FBWCxLQUF3QixVQUE1QixFQUF3QztBQUN0QyxZQUFNLElBQUlDLEtBQUosQ0FBVyw2QkFBNEJELE1BQU8sRUFBOUMsQ0FBTjtBQUNEOztBQUNERixJQUFBQSxJQUFJLENBQUNFLE1BQUQsQ0FBSixHQUFlRixJQUFJLENBQUNFLE1BQUQsQ0FBSixDQUFhRSxJQUFiLENBQWtCSixJQUFsQixDQUFmO0FBQ0Q7QUFDRixDLENBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ08sU0FBU0ssWUFBVCxDQUFzQkMsS0FBdEIsRUFBNkJDLFNBQTdCLEVBQXdDQyxPQUFPLEdBQUcsRUFBbEQsRUFBc0Q7QUFDM0QsU0FBT0MsTUFBTSxDQUFDQyxJQUFQLENBQVlILFNBQVosRUFBdUJJLE1BQXZCLENBQThCLENBQUNDLElBQUQsRUFBT0MsUUFBUCxLQUFvQjtBQUN2RCxRQUFJUCxLQUFLLENBQUNPLFFBQUQsQ0FBTCxLQUFvQkMsU0FBeEIsRUFBbUM7QUFDakMsWUFBTUMsWUFBWSxHQUFHUCxPQUFPLENBQUNLLFFBQUQsQ0FBUCxJQUFxQkEsUUFBMUM7QUFDQUQsTUFBQUEsSUFBSSxDQUFDRyxZQUFELENBQUosR0FBcUJULEtBQUssQ0FBQ08sUUFBRCxDQUExQjtBQUNEOztBQUNELFdBQU9ELElBQVA7QUFDRCxHQU5NLEVBTUosRUFOSSxDQUFQO0FBT0QsQyxDQUVEOzs7QUFDTyxTQUFTSSxXQUFULENBQXFCVixLQUFyQixFQUE0QkMsU0FBNUIsRUFBdUM7QUFDNUMsU0FBT0UsTUFBTSxDQUFDQyxJQUFQLENBQVlKLEtBQVosRUFBbUJLLE1BQW5CLENBQTBCLENBQUNDLElBQUQsRUFBT0MsUUFBUCxLQUFvQjtBQUNuRCxRQUFJTixTQUFTLENBQUNNLFFBQUQsQ0FBVCxLQUF3QkMsU0FBNUIsRUFBdUM7QUFDckNGLE1BQUFBLElBQUksQ0FBQ0MsUUFBRCxDQUFKLEdBQWlCUCxLQUFLLENBQUNPLFFBQUQsQ0FBdEI7QUFDRDs7QUFDRCxXQUFPRCxJQUFQO0FBQ0QsR0FMTSxFQUtKLEVBTEksQ0FBUDtBQU1EOztBQUVNLFNBQVNLLGNBQVQsR0FBMEI7QUFDL0IsUUFBTTtBQUFDQyxJQUFBQTtBQUFELE1BQWlCQyxJQUFJLENBQUNDLGVBQUwsRUFBdkI7QUFDQSxRQUFNQyxrQ0FBa0MsR0FBRyxDQUFDQyxpQkFBS0MsVUFBTCxDQUFnQkMsU0FBaEIsQ0FBNUM7O0FBQ0EsTUFBSUgsa0NBQUosRUFBd0M7QUFDdEMsV0FBT0MsaUJBQUtHLElBQUwsQ0FBVVAsWUFBVixFQUF3QixjQUF4QixFQUF3QyxRQUF4QyxDQUFQO0FBQ0QsR0FGRCxNQUVPO0FBQ0wsVUFBTVEsV0FBVyxHQUFHSixpQkFBS0ssT0FBTCxDQUFhSCxTQUFiLEVBQXdCLElBQXhCLENBQXBCOztBQUNBLFFBQUlGLGlCQUFLTSxPQUFMLENBQWFWLFlBQWIsTUFBK0IsT0FBbkMsRUFBNEM7QUFDMUMsVUFBSVEsV0FBVyxDQUFDRyxPQUFaLENBQW9CWCxZQUFwQixNQUFzQyxDQUExQyxFQUE2QztBQUMzQyxlQUFPSSxpQkFBS0csSUFBTCxDQUFXLEdBQUVQLFlBQWEsV0FBMUIsRUFBc0MsY0FBdEMsRUFBc0QsUUFBdEQsQ0FBUDtBQUNEO0FBQ0Y7O0FBQ0QsV0FBT1EsV0FBUDtBQUNEO0FBQ0Y7O0FBRUQsU0FBU0ksY0FBVCxHQUEwQjtBQUN4QixRQUFNQyxLQUFLLEdBQUdaLElBQUksQ0FBQ2EsVUFBTCxHQUFrQkQsS0FBbEIsQ0FBd0IscUJBQXhCLENBQWQ7O0FBQ0EsTUFBSUEsS0FBSixFQUFXO0FBQ1QsVUFBTUUsT0FBTyxHQUFHRixLQUFLLENBQUMsQ0FBRCxDQUFyQjtBQUNBLFdBQVEsUUFBT0UsT0FBTyxDQUFDQyxNQUFSLENBQWUsQ0FBZixFQUFrQkMsV0FBbEIsS0FBa0NGLE9BQU8sQ0FBQ0csS0FBUixDQUFjLENBQWQsQ0FBaUIsU0FBbEU7QUFDRDs7QUFFRCxTQUFPLGFBQVA7QUFDRDs7QUFFTSxTQUFTQyxpQkFBVCxHQUE2QjtBQUNsQyxNQUFJQyxPQUFPLENBQUNDLFFBQVIsS0FBcUIsUUFBekIsRUFBbUM7QUFDakMsVUFBTUMsT0FBTyxHQUFHVixjQUFjLEVBQTlCO0FBQ0EsV0FBT1IsaUJBQUtLLE9BQUwsQ0FBYVcsT0FBTyxDQUFDRyxhQUFyQixFQUFvQyxJQUFwQyxFQUEwQyxZQUExQyxFQUNKLEdBQUVELE9BQVEsTUFETixFQUNhLFVBRGIsRUFDeUIsT0FEekIsRUFDa0NBLE9BRGxDLENBQVA7QUFFRCxHQUpELE1BSU87QUFDTCxXQUFPRixPQUFPLENBQUNJLFFBQWY7QUFDRDtBQUNGOztBQUVELElBQUlDLFdBQUo7O0FBQ08sU0FBU0MsYUFBVCxHQUF5QjtBQUM5QixNQUFJLENBQUNELFdBQUwsRUFBa0I7QUFDaEJBLElBQUFBLFdBQVcsR0FBR0UsT0FBTyxDQUFDbEIsT0FBUixDQUFnQixRQUFoQixDQUFkOztBQUNBLFFBQUksQ0FBQ0wsaUJBQUtDLFVBQUwsQ0FBZ0JvQixXQUFoQixDQUFMLEVBQW1DO0FBQ2pDO0FBQ0EsWUFBTTtBQUFDekIsUUFBQUE7QUFBRCxVQUFpQkMsSUFBSSxDQUFDQyxlQUFMLEVBQXZCOztBQUNBLFVBQUlFLGlCQUFLTSxPQUFMLENBQWFWLFlBQWIsTUFBK0IsT0FBbkMsRUFBNEM7QUFDMUN5QixRQUFBQSxXQUFXLEdBQUdyQixpQkFBS0csSUFBTCxDQUFXLEdBQUVQLFlBQWEsV0FBMUIsRUFBc0MsY0FBdEMsRUFBc0QsUUFBdEQsQ0FBZDtBQUNELE9BRkQsTUFFTztBQUNMeUIsUUFBQUEsV0FBVyxHQUFHckIsaUJBQUtHLElBQUwsQ0FBVVAsWUFBVixFQUF3QixjQUF4QixFQUF3QyxRQUF4QyxDQUFkO0FBQ0Q7QUFDRjtBQUNGOztBQUVELFNBQU95QixXQUFQO0FBQ0Q7O0FBRUQsTUFBTUcsbUJBQW1CLEdBQUcsSUFBSUMsR0FBSixFQUE1Qjs7QUFDTyxTQUFTQyxtQkFBVCxDQUE2QkMsT0FBN0IsRUFBc0M7QUFDM0MsTUFBSUMsVUFBVSxHQUFHSixtQkFBbUIsQ0FBQ0ssR0FBcEIsQ0FBd0JGLE9BQXhCLENBQWpCOztBQUNBLE1BQUksQ0FBQ0MsVUFBTCxFQUFpQjtBQUNmQSxJQUFBQSxVQUFVLEdBQUdMLE9BQU8sQ0FBQ2xCLE9BQVIsQ0FBZ0JMLGlCQUFLRyxJQUFMLENBQVVELFNBQVYsRUFBcUIsUUFBckIsRUFBK0J5QixPQUEvQixDQUFoQixDQUFiOztBQUNBLFFBQUksQ0FBQzNCLGlCQUFLQyxVQUFMLENBQWdCMkIsVUFBaEIsQ0FBTCxFQUFrQztBQUNoQztBQUNBLFlBQU07QUFBQ2hDLFFBQUFBO0FBQUQsVUFBaUJDLElBQUksQ0FBQ0MsZUFBTCxFQUF2QjtBQUNBOEIsTUFBQUEsVUFBVSxHQUFHNUIsaUJBQUtHLElBQUwsQ0FBVVAsWUFBVixFQUF3QmdDLFVBQXhCLENBQWI7QUFDRDs7QUFFREosSUFBQUEsbUJBQW1CLENBQUNNLEdBQXBCLENBQXdCSCxPQUF4QixFQUFpQ0MsVUFBakM7QUFDRDs7QUFFRCxTQUFPQSxVQUFQO0FBQ0Q7O0FBRU0sU0FBU0csUUFBVCxDQUFrQkMsSUFBbEIsRUFBd0I7QUFDN0IsT0FBSyxJQUFJQyxDQUFDLEdBQUcsQ0FBYixFQUFnQkEsQ0FBQyxHQUFHLEVBQXBCLEVBQXdCQSxDQUFDLEVBQXpCLEVBQTZCO0FBQzNCLFVBQU1DLElBQUksR0FBR0YsSUFBSSxDQUFDRyxVQUFMLENBQWdCRixDQUFoQixDQUFiLENBRDJCLENBRTNCO0FBQ0E7O0FBQ0EsUUFBSUMsSUFBSSxLQUFLLEtBQVQsSUFBa0JBLElBQUksR0FBRyxDQUE3QixFQUFnQztBQUM5QixhQUFPLElBQVA7QUFDRDtBQUNGOztBQUVELFNBQU8sS0FBUDtBQUNEOztBQUVELFNBQVNFLG9CQUFULENBQThCQyxLQUE5QixFQUFxQztBQUNuQyxTQUFPbEQsTUFBTSxDQUFDbUQsbUJBQVAsQ0FBMkJELEtBQTNCLEVBQWtDaEQsTUFBbEMsQ0FBeUMsQ0FBQ2tELEdBQUQsRUFBTUMsSUFBTixLQUFlO0FBQzdEckQsSUFBQUEsTUFBTSxDQUFDc0QsTUFBUCxDQUFjRixHQUFkLEVBQW1CO0FBQ2pCLE9BQUNDLElBQUQsR0FBUUUsT0FBTyxDQUFDQyx3QkFBUixDQUFpQ04sS0FBakMsRUFBd0NHLElBQXhDO0FBRFMsS0FBbkI7QUFHQSxXQUFPRCxHQUFQO0FBQ0QsR0FMTSxFQUtKLEVBTEksQ0FBUDtBQU1EO0FBRUQ7Ozs7OztBQUlPLFNBQVNLLGdCQUFULENBQTBCLEdBQUdDLE9BQTdCLEVBQXNDO0FBQzNDLFNBQU8sSUFBSUMsS0FBSixDQUFVO0FBQUNDLElBQUFBLGlCQUFpQixFQUFFRjtBQUFwQixHQUFWLEVBQXdDO0FBQzdDaEIsSUFBQUEsR0FBRyxDQUFDbUIsTUFBRCxFQUFTUixJQUFULEVBQWU7QUFDaEIsVUFBSUEsSUFBSSxLQUFLLGlCQUFiLEVBQWdDO0FBQzlCLGVBQU8sTUFBTUssT0FBYjtBQUNEOztBQUVELFVBQUlILE9BQU8sQ0FBQ08sR0FBUixDQUFZRCxNQUFaLEVBQW9CUixJQUFwQixDQUFKLEVBQStCO0FBQzdCLGVBQU9RLE1BQU0sQ0FBQ1IsSUFBRCxDQUFiO0FBQ0Q7O0FBRUQsWUFBTVUsZ0JBQWdCLEdBQUdMLE9BQU8sQ0FBQ00sSUFBUixDQUFhQyxDQUFDLElBQUlWLE9BQU8sQ0FBQ08sR0FBUixDQUFZRyxDQUFaLEVBQWVaLElBQWYsQ0FBbEIsQ0FBekI7O0FBQ0EsVUFBSVUsZ0JBQUosRUFBc0I7QUFDcEIsZUFBT0EsZ0JBQWdCLENBQUNWLElBQUQsQ0FBdkI7QUFDRCxPQUZELE1BRU87QUFDTCxlQUFPaEQsU0FBUDtBQUNEO0FBQ0YsS0FoQjRDOztBQWtCN0NzQyxJQUFBQSxHQUFHLENBQUNrQixNQUFELEVBQVNSLElBQVQsRUFBZWEsS0FBZixFQUFzQjtBQUN2QixZQUFNSCxnQkFBZ0IsR0FBR0wsT0FBTyxDQUFDTSxJQUFSLENBQWFDLENBQUMsSUFBSVYsT0FBTyxDQUFDTyxHQUFSLENBQVlHLENBQVosRUFBZVosSUFBZixDQUFsQixDQUF6Qjs7QUFDQSxVQUFJVSxnQkFBSixFQUFzQjtBQUNwQjtBQUNBLGVBQU9BLGdCQUFnQixDQUFDVixJQUFELENBQWhCLEdBQXlCYSxLQUFoQztBQUNELE9BSEQsTUFHTztBQUNMO0FBQ0EsZUFBT0wsTUFBTSxDQUFDUixJQUFELENBQU4sR0FBZWEsS0FBdEI7QUFDRDtBQUNGLEtBM0I0Qzs7QUE2QjdDO0FBQ0FKLElBQUFBLEdBQUcsQ0FBQ0QsTUFBRCxFQUFTUixJQUFULEVBQWU7QUFDaEIsVUFBSUEsSUFBSSxLQUFLLGlCQUFiLEVBQWdDO0FBQzlCLGVBQU8sSUFBUDtBQUNEOztBQUVELGFBQU9LLE9BQU8sQ0FBQ1MsSUFBUixDQUFhRixDQUFDLElBQUlWLE9BQU8sQ0FBQ08sR0FBUixDQUFZRyxDQUFaLEVBQWVaLElBQWYsQ0FBbEIsQ0FBUDtBQUNELEtBcEM0Qzs7QUFzQzdDO0FBQ0FHLElBQUFBLHdCQUF3QixDQUFDSyxNQUFELEVBQVNSLElBQVQsRUFBZTtBQUNyQyxZQUFNVSxnQkFBZ0IsR0FBR0wsT0FBTyxDQUFDTSxJQUFSLENBQWFDLENBQUMsSUFBSVYsT0FBTyxDQUFDQyx3QkFBUixDQUFpQ1MsQ0FBakMsRUFBb0NaLElBQXBDLENBQWxCLENBQXpCO0FBQ0EsWUFBTWUsOEJBQThCLEdBQUdiLE9BQU8sQ0FBQ0Msd0JBQVIsQ0FBaUNLLE1BQWpDLEVBQXlDUixJQUF6QyxDQUF2Qzs7QUFDQSxVQUFJVSxnQkFBSixFQUFzQjtBQUNwQixlQUFPUixPQUFPLENBQUNDLHdCQUFSLENBQWlDTyxnQkFBakMsRUFBbURWLElBQW5ELENBQVA7QUFDRCxPQUZELE1BRU8sSUFBSWUsOEJBQUosRUFBb0M7QUFDekMsZUFBT0EsOEJBQVA7QUFDRCxPQUZNLE1BRUE7QUFDTCxlQUFPL0QsU0FBUDtBQUNEO0FBQ0YsS0FqRDRDOztBQW1EN0M7QUFDQWdFLElBQUFBLGNBQWMsQ0FBQ1IsTUFBRCxFQUFTO0FBQ3JCLGFBQU9ILE9BQU8sQ0FBQ1ksV0FBUixDQUFvQixDQUFDbEIsR0FBRCxFQUFNYSxDQUFOLEtBQVk7QUFDckMsZUFBT2pFLE1BQU0sQ0FBQ3VFLE1BQVAsQ0FBY25CLEdBQWQsRUFBbUJILG9CQUFvQixDQUFDakQsTUFBTSxDQUFDcUUsY0FBUCxDQUFzQkosQ0FBdEIsQ0FBRCxDQUF2QyxDQUFQO0FBQ0QsT0FGTSxFQUVKakUsTUFBTSxDQUFDd0UsU0FGSCxDQUFQO0FBR0Q7O0FBeEQ0QyxHQUF4QyxDQUFQO0FBMEREOztBQUVELFNBQVNDLE1BQVQsQ0FBZ0JDLEdBQWhCLEVBQXFCO0FBQ25CLFNBQU83RCxpQkFBS0ssT0FBTCxDQUFhd0QsR0FBYixFQUFrQixJQUFsQixNQUE0QkEsR0FBbkM7QUFDRDs7QUFFTSxTQUFTQyxjQUFULENBQXdCRCxHQUF4QixFQUE2QjtBQUNsQyxTQUFPQSxHQUFHLEtBQUtFLGVBQUdDLE9BQUgsRUFBUixJQUF3QixDQUFDSixNQUFNLENBQUNDLEdBQUQsQ0FBdEM7QUFDRDs7QUFFTSxlQUFlSSxVQUFmLENBQTBCQyxnQkFBMUIsRUFBNEM7QUFDakQsTUFBSTtBQUNGLFVBQU1DLG9CQUFHQyxNQUFILENBQVVGLGdCQUFWLENBQU47QUFDQSxXQUFPLElBQVA7QUFDRCxHQUhELENBR0UsT0FBT0csQ0FBUCxFQUFVO0FBQ1YsUUFBSUEsQ0FBQyxDQUFDbkMsSUFBRixLQUFXLFFBQWYsRUFBeUI7QUFDdkIsYUFBTyxLQUFQO0FBQ0Q7O0FBRUQsVUFBTW1DLENBQU47QUFDRDtBQUNGOztBQUVNLFNBQVNDLFVBQVQsQ0FBb0JDLE9BQU8sR0FBRyxFQUE5QixFQUFrQztBQUN2Q0MsbUJBQUtDLEtBQUw7O0FBRUEsU0FBTyxJQUFJQyxPQUFKLENBQVksQ0FBQ3JFLE9BQUQsRUFBVXNFLE1BQVYsS0FBcUI7QUFDdENILHFCQUFLSSxLQUFMLENBQVdMLE9BQVgsRUFBb0IsQ0FBQ00sU0FBRCxFQUFZQyxNQUFaLEtBQXVCO0FBQ3pDLFVBQUlELFNBQUosRUFBZTtBQUNiRixRQUFBQSxNQUFNLENBQUNFLFNBQUQsQ0FBTjtBQUNBO0FBQ0Q7O0FBRUQsVUFBSU4sT0FBTyxDQUFDUSxTQUFaLEVBQXVCO0FBQ3JCMUUsUUFBQUEsT0FBTyxDQUFDeUUsTUFBRCxDQUFQO0FBQ0QsT0FGRCxNQUVPO0FBQ0xYLDRCQUFHYSxRQUFILENBQVlGLE1BQVosRUFBb0IsQ0FBQ0csU0FBRCxFQUFZQyxLQUFaLEtBQXVCRCxTQUFTLEdBQUdOLE1BQU0sQ0FBQ00sU0FBRCxDQUFULEdBQXVCNUUsT0FBTyxDQUFDNkUsS0FBRCxDQUFsRjtBQUNEO0FBQ0YsS0FYRDtBQVlELEdBYk0sQ0FBUDtBQWNEOztBQUVNLGVBQWVDLGdCQUFmLENBQWdDakIsZ0JBQWhDLEVBQWtEO0FBQ3ZELFFBQU1rQixJQUFJLEdBQUcsTUFBTWpCLG9CQUFHaUIsSUFBSCxDQUFRbEIsZ0JBQVIsQ0FBbkI7QUFDQSxTQUFPa0IsSUFBSSxDQUFDQyxJQUFMLEdBQVlsQixvQkFBR21CLFNBQUgsQ0FBYUMsT0FBaEMsQ0FGdUQsQ0FFZDtBQUMxQzs7QUFFTSxlQUFlQyxhQUFmLENBQTZCdEIsZ0JBQTdCLEVBQStDO0FBQ3BELFFBQU1rQixJQUFJLEdBQUcsTUFBTWpCLG9CQUFHc0IsS0FBSCxDQUFTdkIsZ0JBQVQsQ0FBbkI7QUFDQSxTQUFPa0IsSUFBSSxDQUFDTSxjQUFMLEVBQVA7QUFDRDs7QUFFTSxTQUFTQyxVQUFULENBQW9CQyxHQUFwQixFQUF5QjtBQUM5QixTQUFPQSxHQUFHLENBQUM5RSxLQUFKLENBQVUsQ0FBVixFQUFhLENBQWIsQ0FBUDtBQUNEOztBQUVNLE1BQU0rRSxrQkFBa0IsR0FBRztBQUNoQ0MsRUFBQUEsS0FBSyxFQUFFLE9BRHlCO0FBRWhDQyxFQUFBQSxPQUFPLEVBQUUsU0FGdUI7QUFHaENDLEVBQUFBLFFBQVEsRUFBRSxVQUhzQjtBQUloQ0MsRUFBQUEsVUFBVSxFQUFFLFVBSm9CO0FBS2hDQyxFQUFBQSxVQUFVLEVBQUU7QUFMb0IsQ0FBM0I7QUFRUDs7Ozs7Ozs7Ozs7QUFRTyxTQUFTQyxzQkFBVCxDQUFnQ0MsTUFBaEMsRUFBd0M7QUFDN0MsTUFBSXBGLE9BQU8sQ0FBQ0MsUUFBUixLQUFxQixPQUF6QixFQUFrQztBQUNoQyxXQUFPbUYsTUFBTSxDQUFDQyxPQUFQLENBQWUsS0FBZixFQUFzQixHQUF0QixFQUEyQkEsT0FBM0IsQ0FBbUMsV0FBbkMsRUFBZ0QsS0FBaEQsQ0FBUDtBQUNELEdBRkQsTUFFTztBQUNMLFdBQU9ELE1BQVA7QUFDRDtBQUNGO0FBRUQ7Ozs7OztBQUlPLFNBQVNFLGVBQVQsQ0FBeUJDLE9BQXpCLEVBQWtDO0FBQ3ZDLE1BQUl2RixPQUFPLENBQUNDLFFBQVIsS0FBcUIsT0FBekIsRUFBa0M7QUFDaEMsV0FBT3NGLE9BQVA7QUFDRCxHQUZELE1BRU87QUFDTCxXQUFPQSxPQUFPLENBQUNDLEtBQVIsQ0FBYyxHQUFkLEVBQW1CckcsSUFBbkIsQ0FBd0JILGlCQUFLeUcsR0FBN0IsQ0FBUDtBQUNEO0FBQ0Y7QUFFRDs7Ozs7QUFHTyxTQUFTQyxZQUFULENBQXNCSCxPQUF0QixFQUErQjtBQUNwQyxNQUFJdkYsT0FBTyxDQUFDQyxRQUFSLEtBQXFCLE9BQXpCLEVBQWtDO0FBQ2hDLFdBQU9zRixPQUFQO0FBQ0QsR0FGRCxNQUVPO0FBQ0wsV0FBT0EsT0FBTyxDQUFDQyxLQUFSLENBQWN4RyxpQkFBS3lHLEdBQW5CLEVBQXdCdEcsSUFBeEIsQ0FBNkIsR0FBN0IsQ0FBUDtBQUNEO0FBQ0Y7O0FBRU0sU0FBU3dHLGdCQUFULENBQTBCQyxRQUExQixFQUFvQyxHQUFHQyxRQUF2QyxFQUFpRDtBQUN0RCxTQUFPRCxRQUFRLENBQUNFLFFBQVQsQ0FBa0I5RyxpQkFBS0csSUFBTCxDQUFVLEdBQUcwRyxRQUFiLENBQWxCLENBQVA7QUFDRDtBQUVEOzs7Ozs7Ozs7Ozs7QUFVTyxTQUFTRSxVQUFULENBQW9CQyxLQUFwQixFQUEyQjtBQUNoQyxRQUFNQyxHQUFHLEdBQUdELEtBQUssQ0FBQ0UsTUFBbEI7O0FBQ0EsTUFBSUQsR0FBRyxLQUFLLENBQVosRUFBZTtBQUNiLFdBQVEsR0FBRUQsS0FBSyxDQUFDLENBQUQsQ0FBSSxFQUFuQjtBQUNELEdBRkQsTUFFTyxJQUFJQyxHQUFHLEtBQUssQ0FBWixFQUFlO0FBQ3BCLFdBQVEsR0FBRUQsS0FBSyxDQUFDLENBQUQsQ0FBSSxRQUFPQSxLQUFLLENBQUMsQ0FBRCxDQUFJLEVBQW5DO0FBQ0Q7O0FBRUQsU0FBT0EsS0FBSyxDQUFDM0gsTUFBTixDQUFhLENBQUNrRCxHQUFELEVBQU00RSxJQUFOLEVBQVlDLEdBQVosS0FBb0I7QUFDdEMsUUFBSUEsR0FBRyxLQUFLLENBQVosRUFBZTtBQUNiLGFBQVEsR0FBRUQsSUFBSyxFQUFmO0FBQ0QsS0FGRCxNQUVPLElBQUlDLEdBQUcsS0FBS0gsR0FBRyxHQUFHLENBQWxCLEVBQXFCO0FBQzFCLGFBQVEsR0FBRTFFLEdBQUksU0FBUTRFLElBQUssRUFBM0I7QUFDRCxLQUZNLE1BRUE7QUFDTCxhQUFRLEdBQUU1RSxHQUFJLEtBQUk0RSxJQUFLLEVBQXZCO0FBQ0Q7QUFDRixHQVJNLEVBUUosRUFSSSxDQUFQO0FBU0Q7O0FBRU0sU0FBU0UsU0FBVCxDQUFtQkMsR0FBbkIsRUFBd0JDLEdBQXhCLEVBQTZCbEUsS0FBN0IsRUFBb0M7QUFDekMsTUFBSW1FLFFBQVEsR0FBR0YsR0FBRyxDQUFDekYsR0FBSixDQUFRMEYsR0FBUixDQUFmOztBQUNBLE1BQUksQ0FBQ0MsUUFBTCxFQUFlO0FBQ2JBLElBQUFBLFFBQVEsR0FBRyxFQUFYO0FBQ0FGLElBQUFBLEdBQUcsQ0FBQ3hGLEdBQUosQ0FBUXlGLEdBQVIsRUFBYUMsUUFBYjtBQUNEOztBQUNEQSxFQUFBQSxRQUFRLENBQUNDLElBQVQsQ0FBY3BFLEtBQWQ7QUFDRCxDLENBRUQ7OztBQUVPLFNBQVNxRSxvQkFBVCxDQUE4QkMsVUFBOUIsRUFBMEM7QUFDL0MsU0FBTzNILGlCQUFLRyxJQUFMLENBQVV3SCxVQUFVLENBQUNDLG1CQUFYLEVBQVYsRUFBNEMscUJBQTVDLENBQVA7QUFDRDs7QUFFTSxTQUFTQyx1QkFBVCxDQUFpQ0YsVUFBakMsRUFBNkNHLFNBQTdDLEVBQXdEO0FBQzdELE1BQUksQ0FBQ0gsVUFBVSxDQUFDSSxTQUFYLEVBQUwsRUFBNkI7QUFDM0IsV0FBTyxFQUFQO0FBQ0Q7O0FBQ0QsU0FBT0QsU0FBUyxDQUFDRSxjQUFWLEdBQTJCQyxNQUEzQixDQUFrQ0MsTUFBTSxJQUFJQSxNQUFNLENBQUNDLE9BQVAsT0FBcUJULG9CQUFvQixDQUFDQyxVQUFELENBQXJGLENBQVA7QUFDRDs7QUFFRCxJQUFJUyxlQUFlLEdBQUcsSUFBdEI7O0FBQ08sU0FBU0MscUJBQVQsQ0FBK0I7QUFBQ0MsRUFBQUEsVUFBRDtBQUFhQyxFQUFBQTtBQUFiLElBQXNCLEVBQXJELEVBQXlEVCxTQUF6RCxFQUFvRTtBQUN6RSxNQUFJTSxlQUFlLEtBQUssSUFBeEIsRUFBOEI7QUFDNUJBLElBQUFBLGVBQWUsR0FBRzdHLE9BQU8sQ0FBQywyQkFBRCxDQUFQLFdBQWxCO0FBQ0Q7O0FBRUQsU0FBT3VHLFNBQVMsQ0FBQ1UsWUFBVixHQUF5QlAsTUFBekIsQ0FBZ0NkLElBQUksSUFBSTtBQUM3QyxVQUFNc0IsZUFBZSxHQUFHdEIsSUFBSSxJQUFJQSxJQUFJLENBQUN1QixXQUFiLElBQTRCdkIsSUFBSSxDQUFDdUIsV0FBTCxjQUE4Qk4sZUFBbEY7O0FBQ0EsUUFBSUUsVUFBSixFQUFnQjtBQUNkLGFBQU9HLGVBQWUsSUFBSXRCLElBQUksQ0FBQ3dCLGFBQUwsS0FBdUIsUUFBakQ7QUFDRCxLQUZELE1BRU8sSUFBSUosS0FBSixFQUFXO0FBQ2hCLGFBQU9FLGVBQWUsR0FBR3RCLElBQUksQ0FBQ3lCLE9BQUwsRUFBSCxHQUFvQixLQUExQztBQUNELEtBRk0sTUFFQTtBQUNMLGFBQU9ILGVBQVA7QUFDRDtBQUNGLEdBVE0sQ0FBUDtBQVVEOztBQUVNLFNBQVNJLHlCQUFULENBQW1DO0FBQUNQLEVBQUFBO0FBQUQsSUFBZSxFQUFsRCxFQUFzRFIsU0FBdEQsRUFBaUU7QUFDdEUsUUFBTWdCLGNBQWMsR0FBR1QscUJBQXFCLENBQUM7QUFBQ0MsSUFBQUE7QUFBRCxHQUFELEVBQWVSLFNBQWYsQ0FBNUM7QUFDQWdCLEVBQUFBLGNBQWMsQ0FBQ0MsT0FBZixDQUF1QjVCLElBQUksSUFBSUEsSUFBSSxDQUFDNkIsT0FBTCxFQUEvQjtBQUNEOztBQUVNLFNBQVNDLDhCQUFULENBQXdDbkIsU0FBeEMsRUFBbUQ7QUFDeEQsUUFBTWdCLGNBQWMsR0FBR1QscUJBQXFCLENBQUM7QUFBQ0UsSUFBQUEsS0FBSyxFQUFFO0FBQVIsR0FBRCxFQUFnQlQsU0FBaEIsQ0FBNUM7QUFDQWdCLEVBQUFBLGNBQWMsQ0FBQ0MsT0FBZixDQUF1QjVCLElBQUksSUFBSUEsSUFBSSxDQUFDNkIsT0FBTCxFQUEvQjtBQUNEOztBQUVNLFNBQVNFLG1DQUFULENBQTZDQyxhQUE3QyxFQUE0RDtBQUNqRSxRQUFNQyxZQUFZLEdBQUcsRUFBckI7QUFDQSxRQUFNQyxTQUFTLEdBQUcsRUFBbEI7O0FBRUEsT0FBSyxNQUFNQyxJQUFYLElBQW1CSCxhQUFhLENBQUMzQyxLQUFkLENBQW9CckksaUJBQXBCLENBQW5CLEVBQTJEO0FBQ3pELFVBQU1zQyxLQUFLLEdBQUc2SSxJQUFJLENBQUM3SSxLQUFMLENBQVdyQyxlQUFYLENBQWQ7O0FBQ0EsUUFBSXFDLEtBQUosRUFBVztBQUNUO0FBQ0EsWUFBTSxDQUFDOEksQ0FBRCxFQUFJL0csSUFBSixFQUFVZ0gsS0FBVixJQUFtQi9JLEtBQXpCO0FBQ0E0SSxNQUFBQSxTQUFTLENBQUM1QixJQUFWLENBQWU7QUFBQ2pGLFFBQUFBLElBQUQ7QUFBT2dILFFBQUFBO0FBQVAsT0FBZjtBQUNELEtBSkQsTUFJTztBQUNMSixNQUFBQSxZQUFZLENBQUMzQixJQUFiLENBQWtCNkIsSUFBbEI7QUFDRDtBQUNGOztBQUVELFNBQU87QUFBQ0csSUFBQUEsT0FBTyxFQUFFTCxZQUFZLENBQUNqSixJQUFiLENBQWtCLElBQWxCLENBQVY7QUFBbUNrSixJQUFBQTtBQUFuQyxHQUFQO0FBQ0QsQyxDQUVEOzs7QUFFTyxTQUFTSyxVQUFULENBQW9CQyxJQUFwQixFQUEwQkMsZUFBZSxHQUFHLElBQTVDLEVBQWtEQyxHQUFHLEdBQUcsSUFBeEQsRUFBOERDLEtBQUssR0FBRyxFQUF0RSxFQUEwRTtBQUMvRSxRQUFNQyxNQUFNLEdBQUdILGVBQWUsSUFBSSxJQUFJSSxxQkFBSixFQUFsQzs7QUFFQSxRQUFNQyxRQUFRO0FBQ1pDLElBQUFBLFVBQVUsRUFBRSxNQUFNUCxJQUROO0FBR1pqQixJQUFBQSxXQUFXLEVBQUUsTUFBTXFCLE1BQU0sQ0FBQ0ksS0FBUCxDQUFhLElBQWIsQ0FIUDtBQUtaQyxJQUFBQSxrQkFBa0IsRUFBRSxNQUFNTCxNQUFNLENBQUNNLFVBQVA7QUFMZCxLQU9UUCxLQVBTLENBQWQ7O0FBVUEsTUFBSUQsR0FBSixFQUFTO0FBQ1BJLElBQUFBLFFBQVEsQ0FBQ0ssTUFBVCxHQUFrQixNQUFNVCxHQUF4QjtBQUNEOztBQUVELE1BQUlELGVBQUosRUFBcUI7QUFDbkIsV0FBTyxJQUFJOUcsS0FBSixDQUFVbUgsUUFBVixFQUFvQjtBQUN6QnBJLE1BQUFBLEdBQUcsQ0FBQ21CLE1BQUQsRUFBU1IsSUFBVCxFQUFlO0FBQ2hCLFlBQUlFLE9BQU8sQ0FBQ08sR0FBUixDQUFZRCxNQUFaLEVBQW9CUixJQUFwQixDQUFKLEVBQStCO0FBQzdCLGlCQUFPUSxNQUFNLENBQUNSLElBQUQsQ0FBYjtBQUNELFNBSGUsQ0FLaEI7QUFDQTs7O0FBQ0EsY0FBTTtBQUFDYSxVQUFBQTtBQUFELFlBQVUwRyxNQUFNLENBQUN6QyxHQUFQLENBQVdpRCxTQUFTLEtBQUs7QUFBQ2xILFVBQUFBLEtBQUssRUFBRWtILFNBQVMsQ0FBQy9ILElBQUQ7QUFBakIsU0FBTCxDQUFwQixFQUFvRDJILEtBQXBELENBQTBEO0FBQUM5RyxVQUFBQSxLQUFLLEVBQUU3RDtBQUFSLFNBQTFELENBQWhCO0FBQ0EsZUFBTzZELEtBQVA7QUFDRCxPQVZ3Qjs7QUFZekJ2QixNQUFBQSxHQUFHLENBQUNrQixNQUFELEVBQVNSLElBQVQsRUFBZWEsS0FBZixFQUFzQjtBQUN2QixlQUFPMEcsTUFBTSxDQUFDekMsR0FBUCxDQUFXaUQsU0FBUyxJQUFJO0FBQzdCQSxVQUFBQSxTQUFTLENBQUMvSCxJQUFELENBQVQsR0FBa0JhLEtBQWxCO0FBQ0EsaUJBQU8sSUFBUDtBQUNELFNBSE0sRUFHSjhHLEtBSEksQ0FHRSxJQUhGLENBQVA7QUFJRCxPQWpCd0I7O0FBbUJ6QmxILE1BQUFBLEdBQUcsQ0FBQ0QsTUFBRCxFQUFTUixJQUFULEVBQWU7QUFDaEIsZUFBT3VILE1BQU0sQ0FBQ3pDLEdBQVAsQ0FBV2lELFNBQVMsSUFBSTdILE9BQU8sQ0FBQ08sR0FBUixDQUFZc0gsU0FBWixFQUF1Qi9ILElBQXZCLENBQXhCLEVBQXNEMkgsS0FBdEQsQ0FBNEQsS0FBNUQsS0FBc0V6SCxPQUFPLENBQUNPLEdBQVIsQ0FBWUQsTUFBWixFQUFvQlIsSUFBcEIsQ0FBN0U7QUFDRDs7QUFyQndCLEtBQXBCLENBQVA7QUF1QkQsR0F4QkQsTUF3Qk87QUFDTCxXQUFPeUgsUUFBUDtBQUNEO0FBQ0YsQyxDQUVEOzs7QUFFTyxTQUFTTyxTQUFULENBQW1CQyxJQUFuQixFQUF5QkMsS0FBekIsRUFBZ0M7QUFDckMsTUFBSUQsSUFBSSxDQUFDRSxJQUFMLEtBQWNELEtBQUssQ0FBQ0MsSUFBeEIsRUFBOEI7QUFDNUIsV0FBTyxLQUFQO0FBQ0Q7O0FBRUQsT0FBSyxNQUFNQyxJQUFYLElBQW1CSCxJQUFuQixFQUF5QjtBQUN2QixRQUFJLENBQUNDLEtBQUssQ0FBQ3pILEdBQU4sQ0FBVTJILElBQVYsQ0FBTCxFQUFzQjtBQUNwQixhQUFPLEtBQVA7QUFDRDtBQUNGOztBQUVELFNBQU8sSUFBUDtBQUNELEMsQ0FFRDs7O0FBRU8sTUFBTUMsY0FBYyxHQUFHLFFBQXZCOzs7QUFFQSxTQUFTQyxVQUFULEdBQXNCO0FBQzNCLFNBQU9ELGNBQVA7QUFDRDs7QUFFTSxNQUFNRSxtQkFBbUIsR0FBRztBQUNqQ0MsRUFBQUEsU0FBUyxFQUFFLElBRHNCO0FBRWpDQyxFQUFBQSxXQUFXLEVBQUUsSUFGb0I7QUFHakNDLEVBQUFBLEtBQUssRUFBRSxJQUgwQjtBQUlqQ0MsRUFBQUEsTUFBTSxFQUFFLElBSnlCO0FBS2pDQyxFQUFBQSxRQUFRLEVBQUUsSUFMdUI7QUFNakNDLEVBQUFBLEtBQUssRUFBRSxJQU4wQjtBQU9qQ0MsRUFBQUEsTUFBTSxFQUFFLElBUHlCO0FBUWpDQyxFQUFBQSxJQUFJLEVBQUU7QUFSMkIsQ0FBNUIsQyxDQVdQOzs7QUFFQSxJQUFJQyxNQUFNLEdBQUcsSUFBYjtBQUNBLElBQUlDLFNBQVMsR0FBRyxJQUFoQjs7QUFFTyxTQUFTQyxjQUFULENBQXdCQyxFQUF4QixFQUE0QjtBQUNqQyxNQUFJSCxNQUFNLEtBQUssSUFBZixFQUFxQjtBQUNuQkEsSUFBQUEsTUFBTSxHQUFHakssT0FBTyxDQUFDLFFBQUQsQ0FBaEI7O0FBRUEsUUFBSWtLLFNBQVMsS0FBSyxJQUFsQixFQUF3QjtBQUN0QixZQUFNRyxlQUFlLEdBQUdySyxPQUFPLENBQUMsV0FBRCxDQUEvQjs7QUFDQWtLLE1BQUFBLFNBQVMsR0FBR0csZUFBZSxFQUEzQjtBQUNEOztBQUVESixJQUFBQSxNQUFNLENBQUNLLFVBQVAsQ0FBa0I7QUFDaEJDLE1BQUFBLE1BQU0sRUFBRSxJQURRO0FBRWhCQyxNQUFBQSxRQUFRLEVBQUUsSUFGTTtBQUdoQkMsTUFBQUEsU0FBUyxFQUFFQyxJQUFJLElBQUlSLFNBQVMsQ0FBQ00sUUFBVixDQUFtQkUsSUFBbkI7QUFISCxLQUFsQjtBQUtEOztBQUVELFNBQU9ULE1BQU0sQ0FBQ0csRUFBRCxDQUFiO0FBQ0Q7O0FBRU0sTUFBTU8sVUFBVSxHQUFHO0FBQ3hCQyxFQUFBQSxLQUFLLEVBQUUsT0FEaUI7QUFFeEJDLEVBQUFBLFNBQVMsRUFBRSxvREFGYTtBQUd4QkMsRUFBQUEsR0FBRyxFQUFFO0FBSG1CLENBQW5CIiwic291cmNlUm9vdCI6Ii9idWlsZC9hdG9tL3NyYy9hdG9tLTEuNDAuMS9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCBmcyBmcm9tICdmcy1leHRyYSc7XG5pbXBvcnQgb3MgZnJvbSAnb3MnO1xuaW1wb3J0IHRlbXAgZnJvbSAndGVtcCc7XG5cbmltcG9ydCBSZWZIb2xkZXIgZnJvbSAnLi9tb2RlbHMvcmVmLWhvbGRlcic7XG5cbmV4cG9ydCBjb25zdCBMSU5FX0VORElOR19SRUdFWCA9IC9cXHI/XFxuLztcbmV4cG9ydCBjb25zdCBDT19BVVRIT1JfUkVHRVggPSAvXmNvLWF1dGhvcmVkLWJ5LiAoLis/KSA8KC4rPyk+JC9pO1xuZXhwb3J0IGNvbnN0IFBBR0VfU0laRSA9IDUwO1xuZXhwb3J0IGNvbnN0IFBBR0lOQVRJT05fV0FJVF9USU1FX01TID0gMTAwO1xuZXhwb3J0IGNvbnN0IENIRUNLX1NVSVRFX1BBR0VfU0laRSA9IDEwO1xuZXhwb3J0IGNvbnN0IENIRUNLX1JVTl9QQUdFX1NJWkUgPSAyMDtcblxuZXhwb3J0IGZ1bmN0aW9uIGF1dG9iaW5kKHNlbGYsIC4uLm1ldGhvZHMpIHtcbiAgZm9yIChjb25zdCBtZXRob2Qgb2YgbWV0aG9kcykge1xuICAgIGlmICh0eXBlb2Ygc2VsZlttZXRob2RdICE9PSAnZnVuY3Rpb24nKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWJsZSB0byBhdXRvYmluZCBtZXRob2QgJHttZXRob2R9YCk7XG4gICAgfVxuICAgIHNlbGZbbWV0aG9kXSA9IHNlbGZbbWV0aG9kXS5iaW5kKHNlbGYpO1xuICB9XG59XG5cbi8vIEV4dHJhY3QgYSBzdWJzZXQgb2YgcHJvcHMgY2hvc2VuIGZyb20gYSBwcm9wVHlwZXMgb2JqZWN0IGZyb20gYSBjb21wb25lbnQncyBwcm9wcyB0byBwYXNzIHRvIGEgZGlmZmVyZW50IEFQSS5cbi8vXG4vLyBVc2FnZTpcbi8vXG4vLyBgYGBqc1xuLy8gY29uc3QgYXBpUHJvcHMgPSB7XG4vLyAgIHplcm86IFByb3BUeXBlcy5udW1iZXIuaXNSZXF1aXJlZCxcbi8vICAgb25lOiBQcm9wVHlwZXMuc3RyaW5nLFxuLy8gICB0d286IFByb3BUeXBlcy5vYmplY3QsXG4vLyB9O1xuLy9cbi8vIGNsYXNzIENvbXBvbmVudCBleHRlbmRzIFJlYWN0LkNvbXBvbmVudCB7XG4vLyAgIHN0YXRpYyBwcm9wVHlwZXMgPSB7XG4vLyAgICAgLi4uYXBpUHJvcHMsXG4vLyAgICAgZXh0cmE6IFByb3BUeXBlcy5mdW5jLFxuLy8gICB9XG4vL1xuLy8gICBhY3Rpb24oKSB7XG4vLyAgICAgY29uc3Qgb3B0aW9ucyA9IGV4dHJhY3RQcm9wcyh0aGlzLnByb3BzLCBhcGlQcm9wcyk7XG4vLyAgICAgLy8gb3B0aW9ucyBjb250YWlucyB6ZXJvLCBvbmUsIGFuZCB0d28sIGJ1dCBub3QgZXh0cmFcbi8vICAgfVxuLy8gfVxuLy8gYGBgXG5leHBvcnQgZnVuY3Rpb24gZXh0cmFjdFByb3BzKHByb3BzLCBwcm9wVHlwZXMsIG5hbWVNYXAgPSB7fSkge1xuICByZXR1cm4gT2JqZWN0LmtleXMocHJvcFR5cGVzKS5yZWR1Y2UoKG9wdHMsIHByb3BOYW1lKSA9PiB7XG4gICAgaWYgKHByb3BzW3Byb3BOYW1lXSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBjb25zdCBkZXN0UHJvcE5hbWUgPSBuYW1lTWFwW3Byb3BOYW1lXSB8fCBwcm9wTmFtZTtcbiAgICAgIG9wdHNbZGVzdFByb3BOYW1lXSA9IHByb3BzW3Byb3BOYW1lXTtcbiAgICB9XG4gICAgcmV0dXJuIG9wdHM7XG4gIH0sIHt9KTtcbn1cblxuLy8gVGhlIG9wcG9zaXRlIG9mIGV4dHJhY3RQcm9wcy4gUmV0dXJuIGEgc3Vic2V0IG9mIHByb3BzIHRoYXQgZG8gKm5vdCogYXBwZWFyIGluIGEgY29tcG9uZW50J3MgcHJvcCB0eXBlcy5cbmV4cG9ydCBmdW5jdGlvbiB1bnVzZWRQcm9wcyhwcm9wcywgcHJvcFR5cGVzKSB7XG4gIHJldHVybiBPYmplY3Qua2V5cyhwcm9wcykucmVkdWNlKChvcHRzLCBwcm9wTmFtZSkgPT4ge1xuICAgIGlmIChwcm9wVHlwZXNbcHJvcE5hbWVdID09PSB1bmRlZmluZWQpIHtcbiAgICAgIG9wdHNbcHJvcE5hbWVdID0gcHJvcHNbcHJvcE5hbWVdO1xuICAgIH1cbiAgICByZXR1cm4gb3B0cztcbiAgfSwge30pO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0UGFja2FnZVJvb3QoKSB7XG4gIGNvbnN0IHtyZXNvdXJjZVBhdGh9ID0gYXRvbS5nZXRMb2FkU2V0dGluZ3MoKTtcbiAgY29uc3QgY3VycmVudEZpbGVXYXNSZXF1aXJlZEZyb21TbmFwc2hvdCA9ICFwYXRoLmlzQWJzb2x1dGUoX19kaXJuYW1lKTtcbiAgaWYgKGN1cnJlbnRGaWxlV2FzUmVxdWlyZWRGcm9tU25hcHNob3QpIHtcbiAgICByZXR1cm4gcGF0aC5qb2luKHJlc291cmNlUGF0aCwgJ25vZGVfbW9kdWxlcycsICdnaXRodWInKTtcbiAgfSBlbHNlIHtcbiAgICBjb25zdCBwYWNrYWdlUm9vdCA9IHBhdGgucmVzb2x2ZShfX2Rpcm5hbWUsICcuLicpO1xuICAgIGlmIChwYXRoLmV4dG5hbWUocmVzb3VyY2VQYXRoKSA9PT0gJy5hc2FyJykge1xuICAgICAgaWYgKHBhY2thZ2VSb290LmluZGV4T2YocmVzb3VyY2VQYXRoKSA9PT0gMCkge1xuICAgICAgICByZXR1cm4gcGF0aC5qb2luKGAke3Jlc291cmNlUGF0aH0udW5wYWNrZWRgLCAnbm9kZV9tb2R1bGVzJywgJ2dpdGh1YicpO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcGFja2FnZVJvb3Q7XG4gIH1cbn1cblxuZnVuY3Rpb24gZ2V0QXRvbUFwcE5hbWUoKSB7XG4gIGNvbnN0IG1hdGNoID0gYXRvbS5nZXRWZXJzaW9uKCkubWF0Y2goLy0oW0EtWmEtel0rKShcXGQrfC0pLyk7XG4gIGlmIChtYXRjaCkge1xuICAgIGNvbnN0IGNoYW5uZWwgPSBtYXRjaFsxXTtcbiAgICByZXR1cm4gYEF0b20gJHtjaGFubmVsLmNoYXJBdCgwKS50b1VwcGVyQ2FzZSgpICsgY2hhbm5lbC5zbGljZSgxKX0gSGVscGVyYDtcbiAgfVxuXG4gIHJldHVybiAnQXRvbSBIZWxwZXInO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZ2V0QXRvbUhlbHBlclBhdGgoKSB7XG4gIGlmIChwcm9jZXNzLnBsYXRmb3JtID09PSAnZGFyd2luJykge1xuICAgIGNvbnN0IGFwcE5hbWUgPSBnZXRBdG9tQXBwTmFtZSgpO1xuICAgIHJldHVybiBwYXRoLnJlc29sdmUocHJvY2Vzcy5yZXNvdXJjZXNQYXRoLCAnLi4nLCAnRnJhbWV3b3JrcycsXG4gICAgICBgJHthcHBOYW1lfS5hcHBgLCAnQ29udGVudHMnLCAnTWFjT1MnLCBhcHBOYW1lKTtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcHJvY2Vzcy5leGVjUGF0aDtcbiAgfVxufVxuXG5sZXQgRFVHSVRFX1BBVEg7XG5leHBvcnQgZnVuY3Rpb24gZ2V0RHVnaXRlUGF0aCgpIHtcbiAgaWYgKCFEVUdJVEVfUEFUSCkge1xuICAgIERVR0lURV9QQVRIID0gcmVxdWlyZS5yZXNvbHZlKCdkdWdpdGUnKTtcbiAgICBpZiAoIXBhdGguaXNBYnNvbHV0ZShEVUdJVEVfUEFUSCkpIHtcbiAgICAgIC8vIEFzc3VtZSB3ZSdyZSBzbmFwc2hvdHRlZFxuICAgICAgY29uc3Qge3Jlc291cmNlUGF0aH0gPSBhdG9tLmdldExvYWRTZXR0aW5ncygpO1xuICAgICAgaWYgKHBhdGguZXh0bmFtZShyZXNvdXJjZVBhdGgpID09PSAnLmFzYXInKSB7XG4gICAgICAgIERVR0lURV9QQVRIID0gcGF0aC5qb2luKGAke3Jlc291cmNlUGF0aH0udW5wYWNrZWRgLCAnbm9kZV9tb2R1bGVzJywgJ2R1Z2l0ZScpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgRFVHSVRFX1BBVEggPSBwYXRoLmpvaW4ocmVzb3VyY2VQYXRoLCAnbm9kZV9tb2R1bGVzJywgJ2R1Z2l0ZScpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuXG4gIHJldHVybiBEVUdJVEVfUEFUSDtcbn1cblxuY29uc3QgU0hBUkVEX01PRFVMRV9QQVRIUyA9IG5ldyBNYXAoKTtcbmV4cG9ydCBmdW5jdGlvbiBnZXRTaGFyZWRNb2R1bGVQYXRoKHJlbFBhdGgpIHtcbiAgbGV0IG1vZHVsZVBhdGggPSBTSEFSRURfTU9EVUxFX1BBVEhTLmdldChyZWxQYXRoKTtcbiAgaWYgKCFtb2R1bGVQYXRoKSB7XG4gICAgbW9kdWxlUGF0aCA9IHJlcXVpcmUucmVzb2x2ZShwYXRoLmpvaW4oX19kaXJuYW1lLCAnc2hhcmVkJywgcmVsUGF0aCkpO1xuICAgIGlmICghcGF0aC5pc0Fic29sdXRlKG1vZHVsZVBhdGgpKSB7XG4gICAgICAvLyBBc3N1bWUgd2UncmUgc25hcHNob3R0ZWRcbiAgICAgIGNvbnN0IHtyZXNvdXJjZVBhdGh9ID0gYXRvbS5nZXRMb2FkU2V0dGluZ3MoKTtcbiAgICAgIG1vZHVsZVBhdGggPSBwYXRoLmpvaW4ocmVzb3VyY2VQYXRoLCBtb2R1bGVQYXRoKTtcbiAgICB9XG5cbiAgICBTSEFSRURfTU9EVUxFX1BBVEhTLnNldChyZWxQYXRoLCBtb2R1bGVQYXRoKTtcbiAgfVxuXG4gIHJldHVybiBtb2R1bGVQYXRoO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNCaW5hcnkoZGF0YSkge1xuICBmb3IgKGxldCBpID0gMDsgaSA8IDUwOyBpKyspIHtcbiAgICBjb25zdCBjb2RlID0gZGF0YS5jaGFyQ29kZUF0KGkpO1xuICAgIC8vIENoYXIgY29kZSA2NTUzMyBpcyB0aGUgXCJyZXBsYWNlbWVudCBjaGFyYWN0ZXJcIjtcbiAgICAvLyA4IGFuZCBiZWxvdyBhcmUgY29udHJvbCBjaGFyYWN0ZXJzLlxuICAgIGlmIChjb2RlID09PSA2NTUzMyB8fCBjb2RlIDwgOSkge1xuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIGZhbHNlO1xufVxuXG5mdW5jdGlvbiBkZXNjcmlwdG9yc0Zyb21Qcm90byhwcm90bykge1xuICByZXR1cm4gT2JqZWN0LmdldE93blByb3BlcnR5TmFtZXMocHJvdG8pLnJlZHVjZSgoYWNjLCBuYW1lKSA9PiB7XG4gICAgT2JqZWN0LmFzc2lnbihhY2MsIHtcbiAgICAgIFtuYW1lXTogUmVmbGVjdC5nZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IocHJvdG8sIG5hbWUpLFxuICAgIH0pO1xuICAgIHJldHVybiBhY2M7XG4gIH0sIHt9KTtcbn1cblxuLyoqXG4gKiBUYWtlcyBhbiBhcnJheSBvZiB0YXJnZXRzIGFuZCByZXR1cm5zIGEgcHJveHkuIFRoZSBwcm94eSBpbnRlcmNlcHRzIHByb3BlcnR5IGFjY2Vzc29yIGNhbGxzIGFuZFxuICogcmV0dXJucyB0aGUgdmFsdWUgb2YgdGhhdCBwcm9wZXJ0eSBvbiB0aGUgZmlyc3Qgb2JqZWN0IGluIGB0YXJnZXRzYCB3aGVyZSB0aGUgdGFyZ2V0IGltcGxlbWVudHMgdGhhdCBwcm9wZXJ0eS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGZpcnN0SW1wbGVtZW50ZXIoLi4udGFyZ2V0cykge1xuICByZXR1cm4gbmV3IFByb3h5KHtfX2ltcGxlbWVudGF0aW9uczogdGFyZ2V0c30sIHtcbiAgICBnZXQodGFyZ2V0LCBuYW1lKSB7XG4gICAgICBpZiAobmFtZSA9PT0gJ2dldEltcGxlbWVudGVycycpIHtcbiAgICAgICAgcmV0dXJuICgpID0+IHRhcmdldHM7XG4gICAgICB9XG5cbiAgICAgIGlmIChSZWZsZWN0Lmhhcyh0YXJnZXQsIG5hbWUpKSB7XG4gICAgICAgIHJldHVybiB0YXJnZXRbbmFtZV07XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGZpcnN0VmFsaWRUYXJnZXQgPSB0YXJnZXRzLmZpbmQodCA9PiBSZWZsZWN0Lmhhcyh0LCBuYW1lKSk7XG4gICAgICBpZiAoZmlyc3RWYWxpZFRhcmdldCkge1xuICAgICAgICByZXR1cm4gZmlyc3RWYWxpZFRhcmdldFtuYW1lXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICB9XG4gICAgfSxcblxuICAgIHNldCh0YXJnZXQsIG5hbWUsIHZhbHVlKSB7XG4gICAgICBjb25zdCBmaXJzdFZhbGlkVGFyZ2V0ID0gdGFyZ2V0cy5maW5kKHQgPT4gUmVmbGVjdC5oYXModCwgbmFtZSkpO1xuICAgICAgaWYgKGZpcnN0VmFsaWRUYXJnZXQpIHtcbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLXJldHVybi1hc3NpZ25cbiAgICAgICAgcmV0dXJuIGZpcnN0VmFsaWRUYXJnZXRbbmFtZV0gPSB2YWx1ZTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1yZXR1cm4tYXNzaWduXG4gICAgICAgIHJldHVybiB0YXJnZXRbbmFtZV0gPSB2YWx1ZTtcbiAgICAgIH1cbiAgICB9LFxuXG4gICAgLy8gVXNlZCBieSBzaW5vblxuICAgIGhhcyh0YXJnZXQsIG5hbWUpIHtcbiAgICAgIGlmIChuYW1lID09PSAnZ2V0SW1wbGVtZW50ZXJzJykge1xuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIHRhcmdldHMuc29tZSh0ID0+IFJlZmxlY3QuaGFzKHQsIG5hbWUpKTtcbiAgICB9LFxuXG4gICAgLy8gVXNlZCBieSBzaW5vblxuICAgIGdldE93blByb3BlcnR5RGVzY3JpcHRvcih0YXJnZXQsIG5hbWUpIHtcbiAgICAgIGNvbnN0IGZpcnN0VmFsaWRUYXJnZXQgPSB0YXJnZXRzLmZpbmQodCA9PiBSZWZsZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcih0LCBuYW1lKSk7XG4gICAgICBjb25zdCBjb21wb3NpdGVPd25Qcm9wZXJ0eURlc2NyaXB0b3IgPSBSZWZsZWN0LmdldE93blByb3BlcnR5RGVzY3JpcHRvcih0YXJnZXQsIG5hbWUpO1xuICAgICAgaWYgKGZpcnN0VmFsaWRUYXJnZXQpIHtcbiAgICAgICAgcmV0dXJuIFJlZmxlY3QuZ2V0T3duUHJvcGVydHlEZXNjcmlwdG9yKGZpcnN0VmFsaWRUYXJnZXQsIG5hbWUpO1xuICAgICAgfSBlbHNlIGlmIChjb21wb3NpdGVPd25Qcm9wZXJ0eURlc2NyaXB0b3IpIHtcbiAgICAgICAgcmV0dXJuIGNvbXBvc2l0ZU93blByb3BlcnR5RGVzY3JpcHRvcjtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICB9XG4gICAgfSxcblxuICAgIC8vIFVzZWQgYnkgc2lub25cbiAgICBnZXRQcm90b3R5cGVPZih0YXJnZXQpIHtcbiAgICAgIHJldHVybiB0YXJnZXRzLnJlZHVjZVJpZ2h0KChhY2MsIHQpID0+IHtcbiAgICAgICAgcmV0dXJuIE9iamVjdC5jcmVhdGUoYWNjLCBkZXNjcmlwdG9yc0Zyb21Qcm90byhPYmplY3QuZ2V0UHJvdG90eXBlT2YodCkpKTtcbiAgICAgIH0sIE9iamVjdC5wcm90b3R5cGUpO1xuICAgIH0sXG4gIH0pO1xufVxuXG5mdW5jdGlvbiBpc1Jvb3QoZGlyKSB7XG4gIHJldHVybiBwYXRoLnJlc29sdmUoZGlyLCAnLi4nKSA9PT0gZGlyO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNWYWxpZFdvcmtkaXIoZGlyKSB7XG4gIHJldHVybiBkaXIgIT09IG9zLmhvbWVkaXIoKSAmJiAhaXNSb290KGRpcik7XG59XG5cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBmaWxlRXhpc3RzKGFic29sdXRlRmlsZVBhdGgpIHtcbiAgdHJ5IHtcbiAgICBhd2FpdCBmcy5hY2Nlc3MoYWJzb2x1dGVGaWxlUGF0aCk7XG4gICAgcmV0dXJuIHRydWU7XG4gIH0gY2F0Y2ggKGUpIHtcbiAgICBpZiAoZS5jb2RlID09PSAnRU5PRU5UJykge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cblxuICAgIHRocm93IGU7XG4gIH1cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldFRlbXBEaXIob3B0aW9ucyA9IHt9KSB7XG4gIHRlbXAudHJhY2soKTtcblxuICByZXR1cm4gbmV3IFByb21pc2UoKHJlc29sdmUsIHJlamVjdCkgPT4ge1xuICAgIHRlbXAubWtkaXIob3B0aW9ucywgKHRlbXBFcnJvciwgZm9sZGVyKSA9PiB7XG4gICAgICBpZiAodGVtcEVycm9yKSB7XG4gICAgICAgIHJlamVjdCh0ZW1wRXJyb3IpO1xuICAgICAgICByZXR1cm47XG4gICAgICB9XG5cbiAgICAgIGlmIChvcHRpb25zLnN5bWxpbmtPaykge1xuICAgICAgICByZXNvbHZlKGZvbGRlcik7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBmcy5yZWFscGF0aChmb2xkZXIsIChyZWFsRXJyb3IsIHJwYXRoKSA9PiAocmVhbEVycm9yID8gcmVqZWN0KHJlYWxFcnJvcikgOiByZXNvbHZlKHJwYXRoKSkpO1xuICAgICAgfVxuICAgIH0pO1xuICB9KTtcbn1cblxuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGlzRmlsZUV4ZWN1dGFibGUoYWJzb2x1dGVGaWxlUGF0aCkge1xuICBjb25zdCBzdGF0ID0gYXdhaXQgZnMuc3RhdChhYnNvbHV0ZUZpbGVQYXRoKTtcbiAgcmV0dXJuIHN0YXQubW9kZSAmIGZzLmNvbnN0YW50cy5TX0lYVVNSOyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIG5vLWJpdHdpc2Vcbn1cblxuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGlzRmlsZVN5bWxpbmsoYWJzb2x1dGVGaWxlUGF0aCkge1xuICBjb25zdCBzdGF0ID0gYXdhaXQgZnMubHN0YXQoYWJzb2x1dGVGaWxlUGF0aCk7XG4gIHJldHVybiBzdGF0LmlzU3ltYm9saWNMaW5rKCk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBzaG9ydGVuU2hhKHNoYSkge1xuICByZXR1cm4gc2hhLnNsaWNlKDAsIDgpO1xufVxuXG5leHBvcnQgY29uc3QgY2xhc3NOYW1lRm9yU3RhdHVzID0ge1xuICBhZGRlZDogJ2FkZGVkJyxcbiAgZGVsZXRlZDogJ3JlbW92ZWQnLFxuICBtb2RpZmllZDogJ21vZGlmaWVkJyxcbiAgdHlwZWNoYW5nZTogJ21vZGlmaWVkJyxcbiAgZXF1aXZhbGVudDogJ2lnbm9yZWQnLFxufTtcblxuLypcbiAqIEFwcGx5IGFueSBwbGF0Zm9ybS1zcGVjaWZpYyBtdW5naW5nIHRvIGEgcGF0aCBiZWZvcmUgcHJlc2VudGluZyBpdCBhc1xuICogYSBnaXQgZW52aXJvbm1lbnQgdmFyaWFibGUgb3Igb3B0aW9uLlxuICpcbiAqIENvbnZlcnQgYSBXaW5kb3dzLXN0eWxlIFwiQzpcXGZvb1xcYmFyXFxiYXpcIiBwYXRoIHRvIGEgXCIvYy9mb28vYmFyL2JhelwiIFVOSVgteVxuICogcGF0aCB0aGF0IHRoZSBzaC5leGUgdXNlZCB0byBleGVjdXRlIGdpdCdzIGNyZWRlbnRpYWwgaGVscGVycyB3aWxsXG4gKiB1bmRlcnN0YW5kLlxuICovXG5leHBvcnQgZnVuY3Rpb24gbm9ybWFsaXplR2l0SGVscGVyUGF0aChpblBhdGgpIHtcbiAgaWYgKHByb2Nlc3MucGxhdGZvcm0gPT09ICd3aW4zMicpIHtcbiAgICByZXR1cm4gaW5QYXRoLnJlcGxhY2UoL1xcXFwvZywgJy8nKS5yZXBsYWNlKC9eKFteOl0rKTovLCAnLyQxJyk7XG4gIH0gZWxzZSB7XG4gICAgcmV0dXJuIGluUGF0aDtcbiAgfVxufVxuXG4vKlxuICogT24gV2luZG93cywgZ2l0IGNvbW1hbmRzIHJlcG9ydCBwYXRocyB3aXRoIC8gZGVsaW1pdGVycy4gQ29udmVydCB0aGVtIHRvIFxcLWRlbGltaXRlZCBwYXRoc1xuICogc28gdGhhdCBBdG9tIHVuaWZyb21seSB0cmVhdHMgcGF0aHMgd2l0aCBuYXRpdmUgcGF0aCBzZXBhcmF0b3JzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gdG9OYXRpdmVQYXRoU2VwKHJhd1BhdGgpIHtcbiAgaWYgKHByb2Nlc3MucGxhdGZvcm0gIT09ICd3aW4zMicpIHtcbiAgICByZXR1cm4gcmF3UGF0aDtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcmF3UGF0aC5zcGxpdCgnLycpLmpvaW4ocGF0aC5zZXApO1xuICB9XG59XG5cbi8qXG4gKiBDb252ZXJ0IFdpbmRvd3MgcGF0aHMgYmFjayB0byAvLWRlbGltaXRlZCBwYXRocyB0byBiZSBwcmVzZW50ZWQgdG8gZ2l0LlxuICovXG5leHBvcnQgZnVuY3Rpb24gdG9HaXRQYXRoU2VwKHJhd1BhdGgpIHtcbiAgaWYgKHByb2Nlc3MucGxhdGZvcm0gIT09ICd3aW4zMicpIHtcbiAgICByZXR1cm4gcmF3UGF0aDtcbiAgfSBlbHNlIHtcbiAgICByZXR1cm4gcmF3UGF0aC5zcGxpdChwYXRoLnNlcCkuam9pbignLycpO1xuICB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBmaWxlUGF0aEVuZHNXaXRoKGZpbGVQYXRoLCAuLi5zZWdtZW50cykge1xuICByZXR1cm4gZmlsZVBhdGguZW5kc1dpdGgocGF0aC5qb2luKC4uLnNlZ21lbnRzKSk7XG59XG5cbi8qKlxuICogVHVybnMgYW4gYXJyYXkgb2YgdGhpbmdzIEBrdXljaGFjbyBjYW5ub3QgZWF0XG4gKiBpbnRvIGEgc2VudGVuY2UgY29udGFpbmluZyB0aGluZ3MgQGt1eWNoYWNvIGNhbm5vdCBlYXRcbiAqXG4gKiBbJ3RvYXN0J10gPT4gJ3RvYXN0J1xuICogWyd0b2FzdCcsICdlZ2dzJ10gPT4gJ3RvYXN0IGFuZCBlZ2dzJ1xuICogWyd0b2FzdCcsICdlZ2dzJywgJ2NoZWVzZSddID0+ICd0b2FzdCwgZWdncywgYW5kIGNoZWVzZSdcbiAqXG4gKiBPeGZvcmQgY29tbWEgaW5jbHVkZWQgYmVjYXVzZSB5b3UncmUgd3JvbmcsIHNodXQgdXAuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB0b1NlbnRlbmNlKGFycmF5KSB7XG4gIGNvbnN0IGxlbiA9IGFycmF5Lmxlbmd0aDtcbiAgaWYgKGxlbiA9PT0gMSkge1xuICAgIHJldHVybiBgJHthcnJheVswXX1gO1xuICB9IGVsc2UgaWYgKGxlbiA9PT0gMikge1xuICAgIHJldHVybiBgJHthcnJheVswXX0gYW5kICR7YXJyYXlbMV19YDtcbiAgfVxuXG4gIHJldHVybiBhcnJheS5yZWR1Y2UoKGFjYywgaXRlbSwgaWR4KSA9PiB7XG4gICAgaWYgKGlkeCA9PT0gMCkge1xuICAgICAgcmV0dXJuIGAke2l0ZW19YDtcbiAgICB9IGVsc2UgaWYgKGlkeCA9PT0gbGVuIC0gMSkge1xuICAgICAgcmV0dXJuIGAke2FjY30sIGFuZCAke2l0ZW19YDtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIGAke2FjY30sICR7aXRlbX1gO1xuICAgIH1cbiAgfSwgJycpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gcHVzaEF0S2V5KG1hcCwga2V5LCB2YWx1ZSkge1xuICBsZXQgZXhpc3RpbmcgPSBtYXAuZ2V0KGtleSk7XG4gIGlmICghZXhpc3RpbmcpIHtcbiAgICBleGlzdGluZyA9IFtdO1xuICAgIG1hcC5zZXQoa2V5LCBleGlzdGluZyk7XG4gIH1cbiAgZXhpc3RpbmcucHVzaCh2YWx1ZSk7XG59XG5cbi8vIFJlcG9zaXRvcnkgYW5kIHdvcmtzcGFjZSBoZWxwZXJzXG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRDb21taXRNZXNzYWdlUGF0aChyZXBvc2l0b3J5KSB7XG4gIHJldHVybiBwYXRoLmpvaW4ocmVwb3NpdG9yeS5nZXRHaXREaXJlY3RvcnlQYXRoKCksICdBVE9NX0NPTU1JVF9FRElUTVNHJyk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRDb21taXRNZXNzYWdlRWRpdG9ycyhyZXBvc2l0b3J5LCB3b3Jrc3BhY2UpIHtcbiAgaWYgKCFyZXBvc2l0b3J5LmlzUHJlc2VudCgpKSB7XG4gICAgcmV0dXJuIFtdO1xuICB9XG4gIHJldHVybiB3b3Jrc3BhY2UuZ2V0VGV4dEVkaXRvcnMoKS5maWx0ZXIoZWRpdG9yID0+IGVkaXRvci5nZXRQYXRoKCkgPT09IGdldENvbW1pdE1lc3NhZ2VQYXRoKHJlcG9zaXRvcnkpKTtcbn1cblxubGV0IENoYW5nZWRGaWxlSXRlbSA9IG51bGw7XG5leHBvcnQgZnVuY3Rpb24gZ2V0RmlsZVBhdGNoUGFuZUl0ZW1zKHtvbmx5U3RhZ2VkLCBlbXB0eX0gPSB7fSwgd29ya3NwYWNlKSB7XG4gIGlmIChDaGFuZ2VkRmlsZUl0ZW0gPT09IG51bGwpIHtcbiAgICBDaGFuZ2VkRmlsZUl0ZW0gPSByZXF1aXJlKCcuL2l0ZW1zL2NoYW5nZWQtZmlsZS1pdGVtJykuZGVmYXVsdDtcbiAgfVxuXG4gIHJldHVybiB3b3Jrc3BhY2UuZ2V0UGFuZUl0ZW1zKCkuZmlsdGVyKGl0ZW0gPT4ge1xuICAgIGNvbnN0IGlzRmlsZVBhdGNoSXRlbSA9IGl0ZW0gJiYgaXRlbS5nZXRSZWFsSXRlbSAmJiBpdGVtLmdldFJlYWxJdGVtKCkgaW5zdGFuY2VvZiBDaGFuZ2VkRmlsZUl0ZW07XG4gICAgaWYgKG9ubHlTdGFnZWQpIHtcbiAgICAgIHJldHVybiBpc0ZpbGVQYXRjaEl0ZW0gJiYgaXRlbS5zdGFnaW5nU3RhdHVzID09PSAnc3RhZ2VkJztcbiAgICB9IGVsc2UgaWYgKGVtcHR5KSB7XG4gICAgICByZXR1cm4gaXNGaWxlUGF0Y2hJdGVtID8gaXRlbS5pc0VtcHR5KCkgOiBmYWxzZTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIGlzRmlsZVBhdGNoSXRlbTtcbiAgICB9XG4gIH0pO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZGVzdHJveUZpbGVQYXRjaFBhbmVJdGVtcyh7b25seVN0YWdlZH0gPSB7fSwgd29ya3NwYWNlKSB7XG4gIGNvbnN0IGl0ZW1zVG9EZXN0cm95ID0gZ2V0RmlsZVBhdGNoUGFuZUl0ZW1zKHtvbmx5U3RhZ2VkfSwgd29ya3NwYWNlKTtcbiAgaXRlbXNUb0Rlc3Ryb3kuZm9yRWFjaChpdGVtID0+IGl0ZW0uZGVzdHJveSgpKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGRlc3Ryb3lFbXB0eUZpbGVQYXRjaFBhbmVJdGVtcyh3b3Jrc3BhY2UpIHtcbiAgY29uc3QgaXRlbXNUb0Rlc3Ryb3kgPSBnZXRGaWxlUGF0Y2hQYW5lSXRlbXMoe2VtcHR5OiB0cnVlfSwgd29ya3NwYWNlKTtcbiAgaXRlbXNUb0Rlc3Ryb3kuZm9yRWFjaChpdGVtID0+IGl0ZW0uZGVzdHJveSgpKTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGV4dHJhY3RDb0F1dGhvcnNBbmRSYXdDb21taXRNZXNzYWdlKGNvbW1pdE1lc3NhZ2UpIHtcbiAgY29uc3QgbWVzc2FnZUxpbmVzID0gW107XG4gIGNvbnN0IGNvQXV0aG9ycyA9IFtdO1xuXG4gIGZvciAoY29uc3QgbGluZSBvZiBjb21taXRNZXNzYWdlLnNwbGl0KExJTkVfRU5ESU5HX1JFR0VYKSkge1xuICAgIGNvbnN0IG1hdGNoID0gbGluZS5tYXRjaChDT19BVVRIT1JfUkVHRVgpO1xuICAgIGlmIChtYXRjaCkge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLXVudXNlZC12YXJzXG4gICAgICBjb25zdCBbXywgbmFtZSwgZW1haWxdID0gbWF0Y2g7XG4gICAgICBjb0F1dGhvcnMucHVzaCh7bmFtZSwgZW1haWx9KTtcbiAgICB9IGVsc2Uge1xuICAgICAgbWVzc2FnZUxpbmVzLnB1c2gobGluZSk7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHttZXNzYWdlOiBtZXNzYWdlTGluZXMuam9pbignXFxuJyksIGNvQXV0aG9yc307XG59XG5cbi8vIEF0b20gQVBJIHBhbmUgaXRlbSBtYW5pcHVsYXRpb25cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUl0ZW0obm9kZSwgY29tcG9uZW50SG9sZGVyID0gbnVsbCwgdXJpID0gbnVsbCwgZXh0cmEgPSB7fSkge1xuICBjb25zdCBob2xkZXIgPSBjb21wb25lbnRIb2xkZXIgfHwgbmV3IFJlZkhvbGRlcigpO1xuXG4gIGNvbnN0IG92ZXJyaWRlID0ge1xuICAgIGdldEVsZW1lbnQ6ICgpID0+IG5vZGUsXG5cbiAgICBnZXRSZWFsSXRlbTogKCkgPT4gaG9sZGVyLmdldE9yKG51bGwpLFxuXG4gICAgZ2V0UmVhbEl0ZW1Qcm9taXNlOiAoKSA9PiBob2xkZXIuZ2V0UHJvbWlzZSgpLFxuXG4gICAgLi4uZXh0cmEsXG4gIH07XG5cbiAgaWYgKHVyaSkge1xuICAgIG92ZXJyaWRlLmdldFVSSSA9ICgpID0+IHVyaTtcbiAgfVxuXG4gIGlmIChjb21wb25lbnRIb2xkZXIpIHtcbiAgICByZXR1cm4gbmV3IFByb3h5KG92ZXJyaWRlLCB7XG4gICAgICBnZXQodGFyZ2V0LCBuYW1lKSB7XG4gICAgICAgIGlmIChSZWZsZWN0Lmhhcyh0YXJnZXQsIG5hbWUpKSB7XG4gICAgICAgICAgcmV0dXJuIHRhcmdldFtuYW1lXTtcbiAgICAgICAgfVxuXG4gICAgICAgIC8vIFRoZSB7dmFsdWU6IC4uLn0gd3JhcHBlciBwcmV2ZW50cyAubWFwKCkgZnJvbSBmbGF0dGVuaW5nIGEgcmV0dXJuZWQgUmVmSG9sZGVyLlxuICAgICAgICAvLyBJZiBjb21wb25lbnRbbmFtZV0gaXMgYSBSZWZIb2xkZXIsIHdlIHdhbnQgdG8gcmV0dXJuIHRoYXQgUmVmSG9sZGVyIGFzLWlzLlxuICAgICAgICBjb25zdCB7dmFsdWV9ID0gaG9sZGVyLm1hcChjb21wb25lbnQgPT4gKHt2YWx1ZTogY29tcG9uZW50W25hbWVdfSkpLmdldE9yKHt2YWx1ZTogdW5kZWZpbmVkfSk7XG4gICAgICAgIHJldHVybiB2YWx1ZTtcbiAgICAgIH0sXG5cbiAgICAgIHNldCh0YXJnZXQsIG5hbWUsIHZhbHVlKSB7XG4gICAgICAgIHJldHVybiBob2xkZXIubWFwKGNvbXBvbmVudCA9PiB7XG4gICAgICAgICAgY29tcG9uZW50W25hbWVdID0gdmFsdWU7XG4gICAgICAgICAgcmV0dXJuIHRydWU7XG4gICAgICAgIH0pLmdldE9yKHRydWUpO1xuICAgICAgfSxcblxuICAgICAgaGFzKHRhcmdldCwgbmFtZSkge1xuICAgICAgICByZXR1cm4gaG9sZGVyLm1hcChjb21wb25lbnQgPT4gUmVmbGVjdC5oYXMoY29tcG9uZW50LCBuYW1lKSkuZ2V0T3IoZmFsc2UpIHx8IFJlZmxlY3QuaGFzKHRhcmdldCwgbmFtZSk7XG4gICAgICB9LFxuICAgIH0pO1xuICB9IGVsc2Uge1xuICAgIHJldHVybiBvdmVycmlkZTtcbiAgfVxufVxuXG4vLyBTZXQgZnVuY3Rpb25zXG5cbmV4cG9ydCBmdW5jdGlvbiBlcXVhbFNldHMobGVmdCwgcmlnaHQpIHtcbiAgaWYgKGxlZnQuc2l6ZSAhPT0gcmlnaHQuc2l6ZSkge1xuICAgIHJldHVybiBmYWxzZTtcbiAgfVxuXG4gIGZvciAoY29uc3QgZWFjaCBvZiBsZWZ0KSB7XG4gICAgaWYgKCFyaWdodC5oYXMoZWFjaCkpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gdHJ1ZTtcbn1cblxuLy8gQ29uc3RhbnRzXG5cbmV4cG9ydCBjb25zdCBOQlNQX0NIQVJBQ1RFUiA9ICdcXHUwMGEwJztcblxuZXhwb3J0IGZ1bmN0aW9uIGJsYW5rTGFiZWwoKSB7XG4gIHJldHVybiBOQlNQX0NIQVJBQ1RFUjtcbn1cblxuZXhwb3J0IGNvbnN0IHJlYWN0aW9uVHlwZVRvRW1vamkgPSB7XG4gIFRIVU1CU19VUDogJ/CfkY0nLFxuICBUSFVNQlNfRE9XTjogJ/CfkY4nLFxuICBMQVVHSDogJ/CfmIYnLFxuICBIT09SQVk6ICfwn46JJyxcbiAgQ09ORlVTRUQ6ICfwn5iVJyxcbiAgSEVBUlQ6ICfinaTvuI8nLFxuICBST0NLRVQ6ICfwn5qAJyxcbiAgRVlFUzogJ/CfkYAnLFxufTtcblxuLy8gTWFya2Rvd25cblxubGV0IG1hcmtlZCA9IG51bGw7XG5sZXQgZG9tUHVyaWZ5ID0gbnVsbDtcblxuZXhwb3J0IGZ1bmN0aW9uIHJlbmRlck1hcmtkb3duKG1kKSB7XG4gIGlmIChtYXJrZWQgPT09IG51bGwpIHtcbiAgICBtYXJrZWQgPSByZXF1aXJlKCdtYXJrZWQnKTtcblxuICAgIGlmIChkb21QdXJpZnkgPT09IG51bGwpIHtcbiAgICAgIGNvbnN0IGNyZWF0ZURPTVB1cmlmeSA9IHJlcXVpcmUoJ2RvbXB1cmlmeScpO1xuICAgICAgZG9tUHVyaWZ5ID0gY3JlYXRlRE9NUHVyaWZ5KCk7XG4gICAgfVxuXG4gICAgbWFya2VkLnNldE9wdGlvbnMoe1xuICAgICAgc2lsZW50OiB0cnVlLFxuICAgICAgc2FuaXRpemU6IHRydWUsXG4gICAgICBzYW5pdGl6ZXI6IGh0bWwgPT4gZG9tUHVyaWZ5LnNhbml0aXplKGh0bWwpLFxuICAgIH0pO1xuICB9XG5cbiAgcmV0dXJuIG1hcmtlZChtZCk7XG59XG5cbmV4cG9ydCBjb25zdCBHSE9TVF9VU0VSID0ge1xuICBsb2dpbjogJ2dob3N0JyxcbiAgYXZhdGFyVXJsOiAnaHR0cHM6Ly9hdmF0YXJzMS5naXRodWJ1c2VyY29udGVudC5jb20vdS8xMDEzNz92PTQnLFxuICB1cmw6ICdodHRwczovL2dpdGh1Yi5jb20vZ2hvc3QnLFxufTtcbiJdfQ==