"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = exports.BarePullRequestCheckoutController = exports.checkoutStates = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _reactRelay = require("react-relay");

var _enableableOperation = _interopRequireDefault(require("../models/enableable-operation"));

var _gitShellOutStrategy = require("../git-shell-out-strategy");

var _propTypes2 = require("../prop-types");

var _reporterProxy = require("../reporter-proxy");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class CheckoutState {
  constructor(name) {
    this.name = name;
  }

  when(cases) {
    return cases[this.name] || cases["default"];
  }

}

const checkoutStates = {
  HIDDEN: new CheckoutState('hidden'),
  DISABLED: new CheckoutState('disabled'),
  BUSY: new CheckoutState('busy'),
  CURRENT: new CheckoutState('current')
};
exports.checkoutStates = checkoutStates;

class BarePullRequestCheckoutController extends _react["default"].Component {
  constructor(props) {
    super(props);
    this.state = {
      checkoutInProgress: false
    };
    this.checkoutOp = new _enableableOperation["default"](() => this.checkout()["catch"](e => {
      if (!(e instanceof _gitShellOutStrategy.GitError)) {
        throw e;
      }
    }));
    this.checkoutOp.toggleState(this, 'checkoutInProgress');
  }

  render() {
    return this.props.children(this.nextCheckoutOp());
  }

  nextCheckoutOp() {
    const {
      repository,
      pullRequest
    } = this.props;

    if (this.props.isAbsent) {
      return this.checkoutOp.disable(checkoutStates.HIDDEN, 'No repository found');
    }

    if (this.props.isLoading) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Loading');
    }

    if (!this.props.isPresent) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'No repository found');
    }

    if (this.props.isMerging) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Merge in progress');
    }

    if (this.props.isRebasing) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Rebase in progress');
    }

    if (this.state.checkoutInProgress) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Checking out...');
    } // determine if pullRequest.headRepository is null
    // this can happen if a repository has been deleted.


    if (!pullRequest.headRepository) {
      return this.checkoutOp.disable(checkoutStates.DISABLED, 'Pull request head repository does not exist');
    } // Determine if we already have this PR checked out.


    const headPush = this.props.branches.getHeadBranch().getPush();
    const headRemote = this.props.remotes.withName(headPush.getRemoteName()); // (detect checkout from pull/### refspec)

    const fromPullRefspec = headRemote.getOwner() === repository.owner.login && headRemote.getRepo() === repository.name && headPush.getShortRemoteRef() === `pull/${pullRequest.number}/head`; // (detect checkout from head repository)

    const fromHeadRepo = headRemote.getOwner() === pullRequest.headRepository.owner.login && headRemote.getRepo() === pullRequest.headRepository.name && headPush.getShortRemoteRef() === pullRequest.headRefName;

    if (fromPullRefspec || fromHeadRepo) {
      return this.checkoutOp.disable(checkoutStates.CURRENT, 'Current');
    }

    return this.checkoutOp.enable();
  }

  async checkout() {
    const {
      pullRequest
    } = this.props;
    const {
      headRepository
    } = pullRequest;
    const fullHeadRef = `refs/heads/${pullRequest.headRefName}`;
    let sourceRemoteName, localRefName; // Discover or create a remote pointing to the repo containing the pull request's head ref.
    // If the local repository already has the head repository specified as a remote, that remote will be used, so
    // that any related configuration is picked up for the fetch. Otherwise, the head repository fetch URL is used
    // directly.

    const headRemotes = this.props.remotes.matchingGitHubRepository(headRepository.owner.login, headRepository.name);

    if (headRemotes.length > 0) {
      sourceRemoteName = headRemotes[0].getName();
    } else {
      const url = {
        https: headRepository.url + '.git',
        ssh: headRepository.sshUrl
      }[this.props.remotes.mostUsedProtocol(['https', 'ssh'])]; // This will throw if a remote with this name already exists (and points somewhere else, or we would have found
      // it above). ¯\_(ツ)_/¯

      const remote = await this.props.localRepository.addRemote(headRepository.owner.login, url);
      sourceRemoteName = remote.getName();
    } // Identify an existing local ref that already corresponds to the pull request, if one exists. Otherwise, generate
    // a new local ref name.


    const pullTargets = this.props.branches.getPullTargets(sourceRemoteName, fullHeadRef);

    if (pullTargets.length > 0) {
      localRefName = pullTargets[0].getName(); // Check out the existing local ref.

      await this.props.localRepository.checkout(localRefName);

      try {
        await this.props.localRepository.pull(fullHeadRef, {
          remoteName: sourceRemoteName,
          ffOnly: true
        });
      } finally {
        (0, _reporterProxy.incrementCounter)('checkout-pr');
      }

      return;
    }

    await this.props.localRepository.fetch(fullHeadRef, {
      remoteName: sourceRemoteName
    }); // Check out the local ref and set it up to track the head ref.

    await this.props.localRepository.checkout(`pr-${pullRequest.number}/${headRepository.owner.login}/${pullRequest.headRefName}`, {
      createNew: true,
      track: true,
      startPoint: `refs/remotes/${sourceRemoteName}/${pullRequest.headRefName}`
    });
    (0, _reporterProxy.incrementCounter)('checkout-pr');
  }

}

exports.BarePullRequestCheckoutController = BarePullRequestCheckoutController;

_defineProperty(BarePullRequestCheckoutController, "propTypes", {
  // GraphQL response
  repository: _propTypes["default"].shape({
    name: _propTypes["default"].string.isRequired,
    owner: _propTypes["default"].shape({
      login: _propTypes["default"].string.isRequired
    }).isRequired
  }).isRequired,
  pullRequest: _propTypes["default"].shape({
    number: _propTypes["default"].number.isRequired,
    headRefName: _propTypes["default"].string.isRequired,
    headRepository: _propTypes["default"].shape({
      name: _propTypes["default"].string.isRequired,
      url: _propTypes["default"].string.isRequired,
      sshUrl: _propTypes["default"].string.isRequired,
      owner: _propTypes["default"].shape({
        login: _propTypes["default"].string.isRequired
      })
    })
  }).isRequired,
  // Repository model and attributes
  localRepository: _propTypes["default"].object.isRequired,
  isAbsent: _propTypes["default"].bool.isRequired,
  isLoading: _propTypes["default"].bool.isRequired,
  isPresent: _propTypes["default"].bool.isRequired,
  isMerging: _propTypes["default"].bool.isRequired,
  isRebasing: _propTypes["default"].bool.isRequired,
  branches: _propTypes2.BranchSetPropType.isRequired,
  remotes: _propTypes2.RemoteSetPropType.isRequired,
  children: _propTypes["default"].func.isRequired
});

var _default = (0, _reactRelay.createFragmentContainer)(BarePullRequestCheckoutController, {
  repository: function () {
    const node = require("./__generated__/prCheckoutController_repository.graphql");

    if (node.hash && node.hash !== "b2212745240c03ff8fc7cb13dfc63183") {
      console.error("The definition of 'prCheckoutController_repository' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prCheckoutController_repository.graphql");
  },
  pullRequest: function () {
    const node = require("./__generated__/prCheckoutController_pullRequest.graphql");

    if (node.hash && node.hash !== "66e001f389a2c4f74c1369cf69b31268") {
      console.error("The definition of 'prCheckoutController_pullRequest' appears to have changed. Run `relay-compiler` to update the generated files to receive the expected data.");
    }

    return require("./__generated__/prCheckoutController_pullRequest.graphql");
  }
});

exports["default"] = _default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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