"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.expectRelayQuery = expectRelayQuery;
exports.clearRelayExpectations = clearRelayExpectations;
exports["default"] = void 0;

var _util = _interopRequireDefault(require("util"));

var _relayRuntime = require("relay-runtime");

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

const LODASH_ISEQUAL = 'lodash.isequal';
let isEqual = null;
const relayEnvironmentPerURL = new Map();
const tokenPerURL = new Map();
const fetchPerURL = new Map();
const responsesByQuery = new Map();

function logRatelimitApi(headers) {
  const remaining = headers.get('x-ratelimit-remaining');
  const total = headers.get('x-ratelimit-limit');
  const resets = headers.get('x-ratelimit-reset');

  const resetsIn = _moment["default"].unix(parseInt(resets, 10)).from(); // eslint-disable-next-line no-console


  console.debug(`GitHub API Rate Limit: ${remaining}/${total} — resets ${resetsIn}`);
}

function expectRelayQuery(operationPattern, response) {
  let resolve, reject;
  const handler = typeof response === 'function' ? response : () => ({
    data: response
  });
  const promise = new Promise((resolve0, reject0) => {
    resolve = resolve0;
    reject = reject0;
  });
  const existing = responsesByQuery.get(operationPattern.name) || [];
  existing.push({
    promise,
    handler,
    variables: operationPattern.variables || {},
    trace: operationPattern.trace
  });
  responsesByQuery.set(operationPattern.name, existing);

  const disable = () => responsesByQuery["delete"](operationPattern.name);

  return {
    promise,
    resolve,
    reject,
    disable
  };
}

function clearRelayExpectations() {
  responsesByQuery.clear();
}

function createFetchQuery(url) {
  if (atom.inSpecMode()) {
    return function specFetchQuery(operation, variables, _cacheConfig, _uploadables) {
      const expectations = responsesByQuery.get(operation.name) || [];
      const match = expectations.find(expectation => {
        if (isEqual === null) {
          // Lazily require lodash.isequal so we can keep it as a dev dependency.
          // Require indirectly to trick electron-link into not following this.
          isEqual = require(LODASH_ISEQUAL);
        }

        return isEqual(expectation.variables, variables);
      });

      if (!match) {
        // eslint-disable-next-line no-console
        console.log(`GraphQL query ${operation.name} was:\n  ${operation.text.replace(/\n/g, '\n  ')}\n` + _util["default"].inspect(variables));
        const e = new Error(`Unexpected GraphQL query: ${operation.name}`);
        e.rawStack = e.stack;
        throw e;
      }

      const responsePromise = match.promise.then(() => {
        return match.handler(operation);
      });

      if (match.trace) {
        // eslint-disable-next-line no-console
        console.log(`[Relay] query "${operation.name}":\n${operation.text}`);
        responsePromise.then(result => {
          // eslint-disable-next-line no-console
          console.log(`[Relay] response "${operation.name}":`, result);
        }, err => {
          // eslint-disable-next-line no-console
          console.error(`[Relay] error "${operation.name}":\n${err.stack || err}`);
          throw err;
        });
      }

      return responsePromise;
    };
  }

  return async function fetchQuery(operation, variables, _cacheConfig, _uploadables) {
    const currentToken = tokenPerURL.get(url);
    let response;

    try {
      response = await fetch(url, {
        method: 'POST',
        headers: {
          'content-type': 'application/json',
          'Authorization': `bearer ${currentToken}`,
          'Accept': 'application/vnd.github.antiope-preview+json'
        },
        body: JSON.stringify({
          query: operation.text,
          variables
        })
      });
    } catch (e) {
      // A network error was encountered. Mark it so that QueryErrorView and ErrorView can distinguish these, because
      // the errors from "fetch" are TypeErrors without much information.
      e.network = true;
      e.rawStack = e.stack;
      throw e;
    }

    try {
      atom && atom.inDevMode() && logRatelimitApi(response.headers);
    } catch (_e) {
      /* do nothing */
    }

    if (response.status !== 200) {
      const e = new Error(`GraphQL API endpoint at ${url} returned ${response.status}`);
      e.response = response;
      e.responseText = await response.text();
      e.rawStack = e.stack;
      throw e;
    }

    const payload = await response.json();

    if (payload && payload.errors && payload.errors.length > 0) {
      const e = new Error(`GraphQL API endpoint at ${url} returned an error for query ${operation.name}.`);
      e.response = response;
      e.errors = payload.errors;
      e.rawStack = e.stack;
      throw e;
    }

    return payload;
  };
}

class RelayNetworkLayerManager {
  static getEnvironmentForHost(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    let {
      environment,
      network
    } = relayEnvironmentPerURL.get(url) || {};
    tokenPerURL.set(url, token);

    if (!environment) {
      const source = new _relayRuntime.RecordSource();
      const store = new _relayRuntime.Store(source);
      network = _relayRuntime.Network.create(this.getFetchQuery(endpoint, token));
      environment = new _relayRuntime.Environment({
        network,
        store
      });
      relayEnvironmentPerURL.set(url, {
        environment,
        network
      });
    }

    return environment;
  }

  static getFetchQuery(endpoint, token) {
    const url = endpoint.getGraphQLRoot();
    tokenPerURL.set(url, token);
    let fetch = fetchPerURL.get(url);

    if (!fetch) {
      fetch = createFetchQuery(url);
      fetchPerURL.set(fetch);
    }

    return fetch;
  }

}

exports["default"] = RelayNetworkLayerManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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