"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildStatusFromStatusContext = buildStatusFromStatusContext;
exports.buildStatusFromCheckResult = buildStatusFromCheckResult;
exports.combineBuildStatuses = combineBuildStatuses;
// Commit or pull request build status, unified from those derived from the Checks API and the Status API.
const DEFAULT = {
  icon: 'unverified',
  classSuffix: 'pending'
};
const PENDING = {
  icon: 'primitive-dot',
  classSuffix: 'pending'
};
const SUCCESS = {
  icon: 'check',
  classSuffix: 'success'
};
const FAILURE = {
  icon: 'x',
  classSuffix: 'failure'
};
const ERROR = {
  icon: 'alert',
  classSuffix: 'failure'
};
const ACTION_REQUIRED = {
  icon: 'bell',
  classSuffix: 'failure'
};
const NEUTRAL = {
  icon: 'dash',
  classSuffix: 'neutral'
};
const STATUS_CONTEXT_MAP = {
  EXPECTED: PENDING,
  PENDING,
  SUCCESS,
  ERROR,
  FAILURE
};

function buildStatusFromStatusContext({
  state
}) {
  return STATUS_CONTEXT_MAP[state] || DEFAULT;
}

const PENDING_CHECK_STATUSES = new Set(['QUEUED', 'IN_PROGRESS', 'REQUESTED']);
const COMPLETED_CHECK_CONCLUSION_MAP = {
  SUCCESS,
  FAILURE,
  TIMED_OUT: ERROR,
  CANCELLED: ERROR,
  ACTION_REQUIRED,
  NEUTRAL
};

function buildStatusFromCheckResult({
  status,
  conclusion
}) {
  if (PENDING_CHECK_STATUSES.has(status)) {
    return PENDING;
  } else if (status === 'COMPLETED') {
    return COMPLETED_CHECK_CONCLUSION_MAP[conclusion] || DEFAULT;
  } else {
    return DEFAULT;
  }
}

const STATUS_PRIORITY = [DEFAULT, NEUTRAL, SUCCESS, PENDING, FAILURE, ERROR, ACTION_REQUIRED];

function combineBuildStatuses(...statuses) {
  let highestPriority = 0;
  let highestPriorityStatus = NEUTRAL;

  for (const status of statuses) {
    const priority = STATUS_PRIORITY.indexOf(status);

    if (priority > highestPriority) {
      highestPriority = priority;
      highestPriorityStatus = status;
    }
  }

  return highestPriorityStatus;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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