"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DashboardStateManager = void 0;

var _i18n = require("@kbn/i18n");

var _lodash = _interopRequireDefault(require("lodash"));

var _state_monitor_factory = require("ui/state_management/state_monitor_factory");

var _dashboard_view_mode = require("./dashboard_view_mode");

var _filter_utils = require("./lib/filter_utils");

var _panel_utils = require("./panel/panel_utils");

var _store = require("../store");

var _actions = require("./actions");

var _panel = require("./panel");

var _lib = require("./lib");

var _selectors = require("../selectors");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _toConsumableArray(arr) { return _arrayWithoutHoles(arr) || _iterableToArray(arr) || _nonIterableSpread(); }

function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance"); }

function _iterableToArray(iter) { if (Symbol.iterator in Object(iter) || Object.prototype.toString.call(iter) === "[object Arguments]") return Array.from(iter); }

function _arrayWithoutHoles(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = new Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } }

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError("Cannot call a class as a function"); } }

function _defineProperties(target, props) { for (var i = 0; i < props.length; i++) { var descriptor = props[i]; descriptor.enumerable = descriptor.enumerable || false; descriptor.configurable = true; if ("value" in descriptor) descriptor.writable = true; Object.defineProperty(target, descriptor.key, descriptor); } }

function _createClass(Constructor, protoProps, staticProps) { if (protoProps) _defineProperties(Constructor.prototype, protoProps); if (staticProps) _defineProperties(Constructor, staticProps); return Constructor; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Dashboard state manager handles connecting angular and redux state between the angular and react portions of the
 * app. There are two "sources of truth" that need to stay in sync - AppState (aka the `_a` portion of the url) and
 * the Store. They aren't complete duplicates of each other as AppState has state that the Store doesn't, and vice
 * versa. They should be as decoupled as possible so updating the store won't affect bwc of urls.
 */
var DashboardStateManager =
/*#__PURE__*/
function () {
  /**
   *
   * @param savedDashboard
   * @param AppState The AppState class to use when instantiating a new AppState instance.
   * @param hideWriteControls true if write controls should be hidden.
   * @param addFilter a function that can be used to add a filter bar filter
   */
  function DashboardStateManager(_ref) {
    var _this = this;

    var savedDashboard = _ref.savedDashboard,
        AppStateClass = _ref.AppStateClass,
        hideWriteControls = _ref.hideWriteControls,
        addFilter = _ref.addFilter;

    _classCallCheck(this, DashboardStateManager);

    _defineProperty(this, "savedDashboard", void 0);

    _defineProperty(this, "appState", void 0);

    _defineProperty(this, "lastSavedDashboardFilters", void 0);

    _defineProperty(this, "stateDefaults", void 0);

    _defineProperty(this, "hideWriteControls", void 0);

    _defineProperty(this, "isDirty", void 0);

    _defineProperty(this, "changeListeners", void 0);

    _defineProperty(this, "stateMonitor", void 0);

    _defineProperty(this, "panelIndexPatternMapping", {});

    _defineProperty(this, "addFilter", void 0);

    _defineProperty(this, "unsubscribe", void 0);

    _defineProperty(this, "addNewPanel", function (id, type) {
      var maxPanelIndex = _panel_utils.PanelUtils.getMaxPanelIndex(_this.getPanels());

      var newPanel = (0, _panel.createPanelState)(id, type, maxPanelIndex.toString(), _this.getPanels());

      _this.getPanels().push(newPanel);

      _this.saveState();
    });

    this.savedDashboard = savedDashboard;
    this.hideWriteControls = hideWriteControls;
    this.addFilter = addFilter;
    this.stateDefaults = (0, _lib.getAppStateDefaults)(this.savedDashboard, this.hideWriteControls);
    this.appState = new AppStateClass(this.stateDefaults); // Initializing appState does two things - first it translates the defaults into AppState, second it updates
    // appState based on the URL (the url trumps the defaults). This means if we update the state format at all and
    // want to handle BWC, we must not only migrate the data stored with saved Dashboard, but also any old state in the
    // url.

    (0, _lib.migrateAppState)(this.appState);
    this.isDirty = false; // We can't compare the filters stored on this.appState to this.savedDashboard because in order to apply
    // the filters to the visualizations, we need to save it on the dashboard. We keep track of the original
    // filter state in order to let the user know if their filters changed and provide this specific information
    // in the 'lose changes' warning message.

    this.lastSavedDashboardFilters = this.getFilterState(); // A mapping of panel index to the index pattern it uses.

    this.panelIndexPatternMapping = {};

    _panel_utils.PanelUtils.initPanelIndexes(this.getPanels());
    /**
     * Creates a state monitor and saves it to this.stateMonitor. Used to track unsaved changes made to appState.
     */


    this.stateMonitor = _state_monitor_factory.stateMonitorFactory.create(this.appState, this.stateDefaults);
    this.stateMonitor.ignoreProps('viewMode'); // Filters need to be compared manually because they sometimes have a $$hashkey stored on the object.

    this.stateMonitor.ignoreProps('filters'); // Query needs to be compared manually because saved legacy queries get migrated in app state automatically

    this.stateMonitor.ignoreProps('query');
    this.stateMonitor.onChange(function (status) {
      _this.isDirty = status.dirty;
    });

    _store.store.dispatch((0, _actions.closeContextMenu)()); // Always start out with all panels minimized when a dashboard is first loaded.


    _store.store.dispatch((0, _actions.minimizePanel)());

    this.pushAppStateChangesToStore();
    this.changeListeners = [];
    this.unsubscribe = _store.store.subscribe(function () {
      return _this.handleStoreChanges();
    });
    this.stateMonitor.onChange(function (status) {
      _this.changeListeners.forEach(function (listener) {
        return listener(status);
      });

      _this.pushAppStateChangesToStore();
    });
  }

  _createClass(DashboardStateManager, [{
    key: "registerChangeListener",
    value: function registerChangeListener(callback) {
      this.changeListeners.push(callback);
    }
  }, {
    key: "areStoreAndAppStatePanelsEqual",
    value: function areStoreAndAppStatePanelsEqual() {
      var state = _store.store.getState();

      var storePanels = (0, _selectors.getPanels)(_store.store.getState());
      var appStatePanels = this.getPanels();

      if (Object.values(storePanels).length !== appStatePanels.length) {
        return false;
      }

      return appStatePanels.every(function (appStatePanel) {
        var storePanel = (0, _selectors.getPanel)(state, appStatePanel.panelIndex);
        return _lodash.default.isEqual(appStatePanel, storePanel);
      });
    }
    /**
     * Time is part of global state so we need to deal with it outside of pushAppStateChangesToStore.
     */

  }, {
    key: "handleTimeChange",
    value: function handleTimeChange(newTimeRange) {
      var from = _filter_utils.FilterUtils.convertTimeToUTCString(newTimeRange.from);

      var to = _filter_utils.FilterUtils.convertTimeToUTCString(newTimeRange.to);

      _store.store.dispatch((0, _actions.updateTimeRange)({
        from: from ? from.toString() : '',
        to: to ? to.toString() : ''
      }));
    }
  }, {
    key: "handleRefreshConfigChange",
    value: function handleRefreshConfigChange(refreshInterval) {
      _store.store.dispatch((0, _actions.updateRefreshConfig)(refreshInterval));
    }
    /**
     * Changes made to app state outside of direct calls to this class will need to be propagated to the store.
     * @private
     */

  }, {
    key: "pushAppStateChangesToStore",
    value: function pushAppStateChangesToStore() {
      // We need these checks, or you can get into a loop where a change is triggered by the store, which updates
      // AppState, which then dispatches the change here, which will end up triggering setState warnings.
      if (!this.areStoreAndAppStatePanelsEqual()) {
        // Translate appState panels data into the data expected by redux, copying the panel objects as we do so
        // because the panels inside appState can be mutated, while redux state should never be mutated directly.
        var panelsMap = this.getPanels().reduce(function (acc, panel) {
          acc[panel.panelIndex] = _lodash.default.cloneDeep(panel);
          return acc;
        }, {});

        _store.store.dispatch((0, _actions.setPanels)(panelsMap));
      }

      var state = _store.store.getState();

      if ((0, _selectors.getTitle)(state) !== this.getTitle()) {
        _store.store.dispatch((0, _actions.updateTitle)(this.getTitle()));
      }

      if ((0, _selectors.getDescription)(state) !== this.getDescription()) {
        _store.store.dispatch((0, _actions.updateDescription)(this.getDescription()));
      }

      if ((0, _selectors.getViewMode)(state) !== this.getViewMode()) {
        _store.store.dispatch((0, _actions.updateViewMode)(this.getViewMode()));
      }

      if ((0, _selectors.getUseMargins)(state) !== this.getUseMargins()) {
        _store.store.dispatch((0, _actions.updateUseMargins)(this.getUseMargins()));
      }

      if ((0, _selectors.getHidePanelTitles)(state) !== this.getHidePanelTitles()) {
        _store.store.dispatch((0, _actions.updateHidePanelTitles)(this.getHidePanelTitles()));
      }

      if ((0, _selectors.getFullScreenMode)(state) !== this.getFullScreenMode()) {
        _store.store.dispatch((0, _actions.updateIsFullScreenMode)(this.getFullScreenMode()));
      }

      if ((0, _selectors.getTitle)(state) !== this.getTitle()) {
        _store.store.dispatch((0, _actions.updateTitle)(this.getTitle()));
      }

      if ((0, _selectors.getDescription)(state) !== this.getDescription()) {
        _store.store.dispatch((0, _actions.updateDescription)(this.getDescription()));
      }

      if ((0, _selectors.getQuery)(state) !== this.getQuery()) {
        _store.store.dispatch((0, _actions.updateQuery)(this.getQuery()));
      }

      this._pushFiltersToStore();
    }
  }, {
    key: "_pushFiltersToStore",
    value: function _pushFiltersToStore() {
      var state = _store.store.getState();

      var dashboardFilters = this.savedDashboard.getFilters();

      if (!_lodash.default.isEqual(_filter_utils.FilterUtils.cleanFiltersForComparison(dashboardFilters), _filter_utils.FilterUtils.cleanFiltersForComparison((0, _selectors.getFilters)(state)))) {
        _store.store.dispatch((0, _actions.updateFilters)(dashboardFilters));
      }
    }
  }, {
    key: "requestReload",
    value: function requestReload() {
      _store.store.dispatch((0, _actions.requestReload)());
    }
  }, {
    key: "handleStoreChanges",
    value: function handleStoreChanges() {
      var _this2 = this;

      var dirty = false;

      if (!this.areStoreAndAppStatePanelsEqual()) {
        var panels = (0, _selectors.getPanels)(_store.store.getState());
        this.appState.panels = [];
        this.panelIndexPatternMapping = {};
        Object.values(panels).map(function (panel) {
          _this2.appState.panels.push(_lodash.default.cloneDeep(panel));
        });
        dirty = true;
      }

      _lodash.default.forEach((0, _selectors.getEmbeddables)(_store.store.getState()), function (embeddable, panelId) {
        if (panelId && embeddable.initialized && !_this2.panelIndexPatternMapping.hasOwnProperty(panelId)) {
          var embeddableMetadata = (0, _selectors.getEmbeddableMetadata)(_store.store.getState(), panelId);

          if (embeddableMetadata && embeddableMetadata.indexPatterns) {
            _this2.panelIndexPatternMapping[panelId] = _lodash.default.compact(embeddableMetadata.indexPatterns);
            dirty = true;
          }
        }
      });

      var stagedFilters = (0, _selectors.getStagedFilters)(_store.store.getState());
      stagedFilters.forEach(function (filter) {
        _this2.addFilter(filter, _this2.getAppState());
      });

      if (stagedFilters.length > 0) {
        this.saveState();

        _store.store.dispatch((0, _actions.clearStagedFilters)());
      }

      var fullScreen = (0, _selectors.getFullScreenMode)(_store.store.getState());

      if (fullScreen !== this.getFullScreenMode()) {
        this.setFullScreenMode(fullScreen);
      }

      this.changeListeners.forEach(function (listener) {
        return listener({
          dirty: dirty
        });
      });
      this.saveState();
    }
  }, {
    key: "getFullScreenMode",
    value: function getFullScreenMode() {
      return this.appState.fullScreenMode;
    }
  }, {
    key: "setFullScreenMode",
    value: function setFullScreenMode(fullScreenMode) {
      this.appState.fullScreenMode = fullScreenMode;
      this.saveState();
    }
  }, {
    key: "getPanelIndexPatterns",
    value: function getPanelIndexPatterns() {
      var indexPatterns = _lodash.default.flatten(Object.values(this.panelIndexPatternMapping));

      return _lodash.default.uniq(indexPatterns, 'id');
    }
    /**
     * Resets the state back to the last saved version of the dashboard.
     */

  }, {
    key: "resetState",
    value: function resetState() {
      // In order to show the correct warning, we have to store the unsaved
      // title on the dashboard object. We should fix this at some point, but this is how all the other object
      // save panels work at the moment.
      this.savedDashboard.title = this.savedDashboard.lastSavedTitle; // appState.reset uses the internal defaults to reset the state, but some of the default settings (e.g. the panels
      // array) point to the same object that is stored on appState and is getting modified.
      // The right way to fix this might be to ensure the defaults object stored on state is a deep
      // clone, but given how much code uses the state object, I determined that to be too risky of a change for
      // now.  TODO: revisit this!

      this.stateDefaults = (0, _lib.getAppStateDefaults)(this.savedDashboard, this.hideWriteControls); // The original query won't be restored by the above because the query on this.savedDashboard is applied
      // in place in order for it to affect the visualizations.

      this.stateDefaults.query = this.lastSavedDashboardFilters.query; // Need to make a copy to ensure they are not overwritten.

      this.stateDefaults.filters = _toConsumableArray(this.getLastSavedFilterBars());
      this.isDirty = false;
      this.appState.setDefaults(this.stateDefaults);
      this.appState.reset();
      this.stateMonitor.setInitialState(this.appState.toJSON());
    }
    /**
     * Returns an object which contains the current filter state of this.savedDashboard.
     * @returns {{timeTo: String, timeFrom: String, filterBars: Array, query: Object}}
     */

  }, {
    key: "getFilterState",
    value: function getFilterState() {
      return {
        timeTo: this.savedDashboard.timeTo,
        timeFrom: this.savedDashboard.timeFrom,
        filterBars: this.savedDashboard.getFilters(),
        query: this.savedDashboard.getQuery()
      };
    }
  }, {
    key: "getTitle",
    value: function getTitle() {
      return this.appState.title;
    }
  }, {
    key: "getDescription",
    value: function getDescription() {
      return this.appState.description;
    }
  }, {
    key: "setDescription",
    value: function setDescription(description) {
      this.appState.description = description;
      this.saveState();
    }
  }, {
    key: "setTitle",
    value: function setTitle(title) {
      this.appState.title = title;
      this.savedDashboard.title = title;
      this.saveState();
    }
  }, {
    key: "getAppState",
    value: function getAppState() {
      return this.appState;
    }
  }, {
    key: "getQuery",
    value: function getQuery() {
      return this.appState.query;
    }
  }, {
    key: "getUseMargins",
    value: function getUseMargins() {
      // Existing dashboards that don't define this should default to false.
      return this.appState.options.useMargins === undefined ? false : this.appState.options.useMargins;
    }
  }, {
    key: "setUseMargins",
    value: function setUseMargins(useMargins) {
      this.appState.options.useMargins = useMargins;
      this.saveState();
    }
  }, {
    key: "getHidePanelTitles",
    value: function getHidePanelTitles() {
      return this.appState.options.hidePanelTitles;
    }
  }, {
    key: "setHidePanelTitles",
    value: function setHidePanelTitles(hidePanelTitles) {
      this.appState.options.hidePanelTitles = hidePanelTitles;
      this.saveState();
    }
  }, {
    key: "getTimeRestore",
    value: function getTimeRestore() {
      return this.appState.timeRestore;
    }
  }, {
    key: "setTimeRestore",
    value: function setTimeRestore(timeRestore) {
      this.appState.timeRestore = timeRestore;
      this.saveState();
    }
    /**
     * @returns {boolean}
     */

  }, {
    key: "getIsTimeSavedWithDashboard",
    value: function getIsTimeSavedWithDashboard() {
      return this.savedDashboard.timeRestore;
    }
  }, {
    key: "getLastSavedFilterBars",
    value: function getLastSavedFilterBars() {
      return this.lastSavedDashboardFilters.filterBars;
    }
  }, {
    key: "getLastSavedQuery",
    value: function getLastSavedQuery() {
      return this.lastSavedDashboardFilters.query;
    }
    /**
     * @returns {boolean} True if the query changed since the last time the dashboard was saved, or if it's a
     * new dashboard, if the query differs from the default.
     */

  }, {
    key: "getQueryChanged",
    value: function getQueryChanged() {
      var currentQuery = this.appState.query;
      var lastSavedQuery = this.getLastSavedQuery();

      var isLegacyStringQuery = _lodash.default.isString(lastSavedQuery) && _lodash.default.isPlainObject(currentQuery) && _lodash.default.has(currentQuery, 'query');

      if (isLegacyStringQuery) {
        return lastSavedQuery !== currentQuery.query;
      }

      return !_lodash.default.isEqual(currentQuery, lastSavedQuery);
    }
    /**
     * @returns {boolean} True if the filter bar state has changed since the last time the dashboard was saved,
     * or if it's a new dashboard, if the query differs from the default.
     */

  }, {
    key: "getFilterBarChanged",
    value: function getFilterBarChanged() {
      return !_lodash.default.isEqual(_filter_utils.FilterUtils.cleanFiltersForComparison(this.appState.filters), _filter_utils.FilterUtils.cleanFiltersForComparison(this.getLastSavedFilterBars()));
    }
    /**
     * @param timeFilter
     * @returns {boolean} True if the time state has changed since the time saved with the dashboard.
     */

  }, {
    key: "getTimeChanged",
    value: function getTimeChanged(timeFilter) {
      return !_filter_utils.FilterUtils.areTimesEqual(this.lastSavedDashboardFilters.timeFrom, timeFilter.getTime().from) || !_filter_utils.FilterUtils.areTimesEqual(this.lastSavedDashboardFilters.timeTo, timeFilter.getTime().to);
    }
    /**
     *
     * @returns {DashboardViewMode}
     */

  }, {
    key: "getViewMode",
    value: function getViewMode() {
      return this.hideWriteControls ? _dashboard_view_mode.DashboardViewMode.VIEW : this.appState.viewMode;
    }
    /**
     * @returns {boolean}
     */

  }, {
    key: "getIsViewMode",
    value: function getIsViewMode() {
      return this.getViewMode() === _dashboard_view_mode.DashboardViewMode.VIEW;
    }
    /**
     * @returns {boolean}
     */

  }, {
    key: "getIsEditMode",
    value: function getIsEditMode() {
      return this.getViewMode() === _dashboard_view_mode.DashboardViewMode.EDIT;
    }
    /**
     *
     * @returns {boolean} True if the dashboard has changed since the last save (or, is new).
     */

  }, {
    key: "getIsDirty",
    value: function getIsDirty(timeFilter) {
      // Filter bar comparison is done manually (see cleanFiltersForComparison for the reason) and time picker
      // changes are not tracked by the state monitor.
      var hasTimeFilterChanged = timeFilter ? this.getFiltersChanged(timeFilter) : false;
      return this.getIsEditMode() && (this.isDirty || hasTimeFilterChanged);
    }
  }, {
    key: "getPanels",
    value: function getPanels() {
      return this.appState.panels;
    }
  }, {
    key: "updatePanel",
    value: function updatePanel(panelIndex, panelAttributes) {
      var foundPanel = this.getPanels().find(function (panel) {
        return panel.panelIndex === panelIndex;
      });
      Object.assign(foundPanel, panelAttributes);
      this.saveState();
      return foundPanel;
    }
    /**
     * Creates and initializes a basic panel, adding it to the state.
     * @param {number} id
     * @param {string} type
     */

  }, {
    key: "removePanel",
    value: function removePanel(panelIndex) {
      var _this3 = this;

      _lodash.default.remove(this.getPanels(), function (panel) {
        if (panel.panelIndex === panelIndex) {
          delete _this3.panelIndexPatternMapping[panelIndex];
          return true;
        } else {
          return false;
        }
      });

      this.saveState();
    }
    /**
     * @param timeFilter
     * @returns {Array.<string>} An array of user friendly strings indicating the filter types that have changed.
     */

  }, {
    key: "getChangedFilterTypes",
    value: function getChangedFilterTypes(timeFilter) {
      var changedFilters = [];

      if (this.getFilterBarChanged()) {
        changedFilters.push('filter');
      }

      if (this.getQueryChanged()) {
        changedFilters.push('query');
      }

      if (this.savedDashboard.timeRestore && this.getTimeChanged(timeFilter)) {
        changedFilters.push('time range');
      }

      return changedFilters;
    }
    /**
     * @return True if filters (query, filter bar filters, and time picker if time is stored
     * with the dashboard) have changed since the last saved state (or if the dashboard hasn't been saved,
     * the default state).
     */

  }, {
    key: "getFiltersChanged",
    value: function getFiltersChanged(timeFilter) {
      return this.getChangedFilterTypes(timeFilter).length > 0;
    }
    /**
     * Updates timeFilter to match the time saved with the dashboard.
     */

  }, {
    key: "syncTimefilterWithDashboard",
    value: function syncTimefilterWithDashboard(timeFilter) {
      if (!this.getIsTimeSavedWithDashboard()) {
        throw new Error(_i18n.i18n.translate('kbn.dashboard.stateManager.timeNotSavedWithDashboardErrorMessage', {
          defaultMessage: 'The time is not saved with this dashboard so should not be synced.'
        }));
      }

      if (this.savedDashboard.timeFrom && this.savedDashboard.timeTo) {
        timeFilter.setTime({
          from: this.savedDashboard.timeFrom,
          to: this.savedDashboard.timeTo
        });
      }

      if (this.savedDashboard.refreshInterval) {
        timeFilter.setRefreshInterval(this.savedDashboard.refreshInterval);
      }
    }
    /**
     * Saves the current application state to the URL.
     */

  }, {
    key: "saveState",
    value: function saveState() {
      this.appState.save();
    }
    /**
     * Applies the current filter state to the dashboard.
     * @param filter {Array.<Object>} An array of filter bar filters.
     */

  }, {
    key: "applyFilters",
    value: function applyFilters(query, filters) {
      this.appState.query = query;
      this.savedDashboard.searchSource.setField('query', query);
      this.savedDashboard.searchSource.setField('filter', filters);
      this.saveState(); // pinned filters go on global state, therefore are not propagated to store via app state and have to be pushed manually.

      this._pushFiltersToStore();
    }
    /**
     * @param newMode {DashboardViewMode}
     */

  }, {
    key: "switchViewMode",
    value: function switchViewMode(newMode) {
      this.appState.viewMode = newMode;
      this.saveState();
    }
    /**
     * Destroys and cleans up this object when it's no longer used.
     */

  }, {
    key: "destroy",
    value: function destroy() {
      if (this.stateMonitor) {
        this.stateMonitor.destroy();
      }

      this.savedDashboard.destroy();
      this.unsubscribe();
    }
  }]);

  return DashboardStateManager;
}();

exports.DashboardStateManager = DashboardStateManager;