import * as A from './Array';
import { getMeetSemigroup, getJoinSemigroup } from './Semigroup';
import { some, none } from './Option';
import { compose } from './function';
export const URI = 'NonEmptyArray2v';
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    const SA = A.getShow(S);
    return {
        show: arr => `make(${S.show(arr[0])}, ${SA.show(arr.slice(1))})`
    };
};
/**
 * @since 1.15.0
 */
export function make(head, tail) {
    return [head, ...tail];
}
/**
 * @since 1.15.0
 */
export function head(nea) {
    return nea[0];
}
/**
 * @since 1.15.0
 */
export function tail(nea) {
    return nea.slice(1);
}
/**
 * @since 1.17.3
 */
export const reverse = A.reverse;
/**
 * @since 1.15.0
 */
export function min(ord) {
    const S = getMeetSemigroup(ord);
    return nea => nea.reduce(S.concat);
}
/**
 * @since 1.15.0
 */
export function max(ord) {
    const S = getJoinSemigroup(ord);
    return nea => nea.reduce(S.concat);
}
/**
 * Builds a `NonEmptyArray` from an `Array` returning `none` if `as` is an empty array
 *
 * @since 1.15.0
 */
export function fromArray(as) {
    return as.length > 0 ? some(as) : none;
}
/**
 * Builds a `NonEmptyArray` from a provably (compile time) non empty `Array`.
 *
 * @since 1.15.0
 */
export function fromNonEmptyArray(as) {
    return as;
}
/**
 * Builds a `Semigroup` instance for `NonEmptyArray`
 *
 * @since 1.15.0
 */
export const getSemigroup = () => {
    return {
        concat: (x, y) => x.concat(y)
    };
};
/**
 * @example
 * import { fromNonEmptyArray, getSetoid, make } from 'fp-ts/lib/NonEmptyArray2v'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * const S = getSetoid(setoidNumber)
 * assert.strictEqual(S.equals(make(1, [2]), fromNonEmptyArray([1, 2])), true)
 * assert.strictEqual(S.equals(make(1, [2]), fromNonEmptyArray([1, 3])), false)
 *
 * @since 1.15.0
 */
export function getSetoid(S) {
    return A.getSetoid(S);
}
/**
 * Group equal, consecutive elements of an array into non empty arrays.
 *
 * @example
 * import { make, group } from 'fp-ts/lib/NonEmptyArray2v'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(group(ordNumber)([1, 2, 1, 1]), [
 *   make(1, []),
 *   make(2, []),
 *   make(1, [1])
 * ])
 *
 * @since 1.15.0
 */
export const group = (S) => (as) => {
    const len = as.length;
    if (len === 0) {
        return A.empty;
    }
    const r = [];
    let head = as[0];
    let nea = fromNonEmptyArray([head]);
    for (let i = 1; i < len; i++) {
        const x = as[i];
        if (S.equals(x, head)) {
            nea.push(x);
        }
        else {
            r.push(nea);
            head = x;
            nea = fromNonEmptyArray([head]);
        }
    }
    r.push(nea);
    return r;
};
/**
 * Sort and then group the elements of an array into non empty arrays.
 *
 * @example
 * import { make, groupSort } from 'fp-ts/lib/NonEmptyArray2v'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(groupSort(ordNumber)([1, 2, 1, 1]), [make(1, [1, 1]), make(2, [])])
 *
 * @since 1.15.0
 */
export const groupSort = (O) => {
    return compose(group(O), A.sort(O));
};
/**
 * Splits an array into sub-non-empty-arrays stored in an object, based on the result of calling a `string`-returning
 * function on each element, and grouping the results according to values returned
 *
 * @example
 * import { make, groupBy } from 'fp-ts/lib/NonEmptyArray2v'
 *
 * assert.deepStrictEqual(groupBy(['foo', 'bar', 'foobar'], a => String(a.length)), {
 *   '3': make('foo', ['bar']),
 *   '6': make('foobar', [])
 * })
 *
 * @since 1.15.0
 */
export const groupBy = (as, f) => {
    const r = {};
    for (const a of as) {
        const k = f(a);
        if (r.hasOwnProperty(k)) {
            r[k].push(a);
        }
        else {
            r[k] = make(a, []);
        }
    }
    return r;
};
/**
 * @since 1.15.0
 */
export function last(nea) {
    return nea[nea.length - 1];
}
/**
 * @since 1.15.0
 */
export function sort(O) {
    return A.sort(O);
}
export function findFirst(nea, predicate) {
    return A.findFirst(nea, predicate);
}
export function findLast(nea, predicate) {
    return A.findLast(nea, predicate);
}
/**
 * @since 1.15.0
 */
export function findIndex(nea, predicate) {
    return A.findIndex(nea, predicate);
}
/**
 * @since 1.15.0
 */
export function findLastIndex(nea, predicate) {
    return A.findLastIndex(nea, predicate);
}
/**
 * @since 1.15.0
 */
export function insertAt(i, a, nea) {
    return A.insertAt(i, a, nea);
}
/**
 * @since 1.15.0
 */
export function updateAt(i, a, nea) {
    return A.updateAt(i, a, nea);
}
/**
 * @since 1.17.0
 */
export function modifyAt(nea, i, f) {
    return A.modifyAt(nea, i, f);
}
/**
 * @since 1.17.0
 */
export const copy = (nea) => {
    return A.copy(nea);
};
export function filter(nea, predicate) {
    return filterWithIndex(nea, (_, a) => predicate(a));
}
/**
 * @since 1.15.0
 */
export function filterWithIndex(nea, predicate) {
    return fromArray(nea.filter((a, i) => predicate(i, a)));
}
const mapWithIndex = (fa, f) => {
    return fa.map((a, i) => f(i, a));
};
/**
 * Append an element to the end of an array, creating a new non empty array
 *
 * @example
 * import { snoc } from 'fp-ts/lib/NonEmptyArray2v'
 *
 * assert.deepStrictEqual(snoc([1, 2, 3], 4), [1, 2, 3, 4])
 *
 * @since 1.16.0
 */
export const snoc = A.snoc;
/**
 * Append an element to the front of an array, creating a new non empty array
 *
 * @example
 * import { cons } from 'fp-ts/lib/NonEmptyArray2v'
 *
 * assert.deepStrictEqual(cons(1, [2, 3, 4]), [1, 2, 3, 4])
 *
 * @since 1.16.0
 */
export const cons = A.cons;
/**
 * @since 1.15.0
 */
export const nonEmptyArray = {
    URI,
    map: A.array.map,
    mapWithIndex,
    of: A.array.of,
    ap: A.array.ap,
    chain: A.array.chain,
    extend: A.array.extend,
    extract: head,
    reduce: A.array.reduce,
    foldMap: A.array.foldMap,
    foldr: A.array.foldr,
    traverse: A.array.traverse,
    sequence: A.array.sequence,
    reduceWithIndex: A.array.reduceWithIndex,
    foldMapWithIndex: A.array.foldMapWithIndex,
    foldrWithIndex: A.array.foldrWithIndex,
    traverseWithIndex: A.array.traverseWithIndex
};
