"use strict";
const MatrixAction = require("../../lib/models/MatrixAction");

const FakeIntent = {
    getProfileInfo: (userId) => {
        return new Promise((resolve, reject) => {
            if (userId === "@jc.denton:unatco.gov") {
                resolve({displayname: "TheJCDenton"});
            }
            else if (userId === "@paul.denton:unatco.gov") {
                resolve({displayname: "ThePaulDenton"});
            }
            else {
                reject("This user was not found");
            }
        });
    }
}

describe("MatrixAction", function() {

    it("should not highlight mentions to text without mentions", () => {
        let action = new MatrixAction("message", "Some text");
        return action.formatMentions({
            "Some Person": "@foobar:localhost"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("Some text");
            expect(action.htmlText).toBeUndefined();
        });
    });

    it("should highlight a user", () => {
        let action = new MatrixAction(
            "message",
            "JCDenton, it's a bomb!",
            "JCDenton, it's a bomb!",
            null
        );
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton, it's a bomb!");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">"+
                "TheJCDenton</a>, it's a bomb!"
            );
        });
    });
    it("should highlight a user, regardless of case", () => {
        let action = new MatrixAction(
            "message",
            "JCDenton, it's a bomb!",
            "JCDenton, it's a bomb!",
            null
        );
        return action.formatMentions({
            "jcdenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton, it's a bomb!");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">"+
                "TheJCDenton</a>, it's a bomb!"
            );
        });

    });
    it("should highlight a user, with plain text", () => {
        let action = new MatrixAction("message", "JCDenton, it's a bomb!");
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton, it's a bomb!");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">"+
                "TheJCDenton</a>, it's a bomb!"
            );
        });
    });
    it("should highlight a user, with weird characters", () => {
        let action = new MatrixAction("message", "`||JCDenton[m], it's a bomb!");
        return action.formatMentions({
            "`||JCDenton[m]": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton, it's a bomb!");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">"+
                "TheJCDenton</a>, it's a bomb!"
            );
        });
    });
    it("should highlight multiple users", () => {
        let action = new MatrixAction(
            "message",
            "JCDenton is sent to assassinate PaulDenton",
            "JCDenton is sent to assassinate PaulDenton",
            null
        );
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov",
            "PaulDenton": "@paul.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton is sent to assassinate ThePaulDenton");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">TheJCDenton</a> is sent" +
                " to assassinate <a href=\"https://matrix.to/#/@paul.denton:unatco.gov\">" +
                "ThePaulDenton</a>"
            );
        });
    });
    it("should highlight multiple mentions of the same user", () => {
        let action = new MatrixAction(
            "message",
            "JCDenton, meet JCDenton",
            "JCDenton, meet JCDenton",
            null
        );
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("TheJCDenton, meet TheJCDenton");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">TheJCDenton</a>," +
                " meet <a href=\"https://matrix.to/#/@jc.denton:unatco.gov\">TheJCDenton</a>"
            );
        });
    });
    it("should not highlight mentions in a URL with www.", () => {
        let action = new MatrixAction(
            "message",
            "Go to http://www.JCDenton.com",
            "Go to <a href='http://www.JCDenton.com'>my website</a>",
            null
        );
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("Go to http://www.JCDenton.com");
            expect(action.htmlText).toEqual(
                "Go to <a href='http://www.JCDenton.com'>my website</a>"
            );
        });
    });
    it("should not highlight mentions in a URL with http://", () => {
        let action = new MatrixAction(
            "message",
            "Go to http://JCDenton.com",
            "Go to <a href='http://JCDenton.com'>my website</a>",
            null
        );
        return action.formatMentions({
            "JCDenton": "@jc.denton:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("Go to http://JCDenton.com");
            expect(action.htmlText).toEqual(
                "Go to <a href='http://JCDenton.com'>my website</a>"
            );
        });
    });
    it("should fallback to userIds", () => {
        let action = new MatrixAction(
            "message",
            "AnnaNavarre: The machine would not make a mistake!",
            "AnnaNavarre: The machine would not make a mistake!",
            null
        );
        return action.formatMentions({
            "AnnaNavarre": "@anna.navarre:unatco.gov"
        }, FakeIntent).then(() => {
            expect(action.text).toEqual("anna.navarre: The machine would not make a mistake!");
            expect(action.htmlText).toEqual(
                "<a href=\"https://matrix.to/#/@anna.navarre:unatco.gov\">"+
                "anna.navarre</a>: The machine would not make a mistake!"
            );
        });
    });
});
