import { getDictionaryMonoid } from './Monoid';
import { none, some as optionSome } from './Option';
import { fromEquals } from './Setoid';
import { showString } from './Show';
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    return {
        show: r => {
            const elements = collect(r, (k, a) => `${showString.show(k)}: ${S.show(a)}`).join(', ');
            return elements === '' ? '{}' : `{ ${elements} }`;
        }
    };
};
/**
 * Calculate the number of key/value pairs in a record
 *
 * @since 1.10.0
 */
export const size = (d) => {
    return Object.keys(d).length;
};
/**
 * Test whether a record is empty
 *
 * @since 1.10.0
 */
export const isEmpty = (d) => {
    return Object.keys(d).length === 0;
};
export function collect(d, f) {
    const out = [];
    const keys = Object.keys(d).sort();
    for (const key of keys) {
        out.push(f(key, d[key]));
    }
    return out;
}
export function toArray(d) {
    return collect(d, (k, a) => [k, a]);
}
export function toUnfoldable(unfoldable) {
    return d => {
        const arr = toArray(d);
        const len = arr.length;
        return unfoldable.unfoldr(0, b => (b < len ? optionSome([arr[b], b + 1]) : none));
    };
}
export function insert(k, a, d) {
    if (d[k] === a) {
        return d;
    }
    const r = Object.assign({}, d);
    r[k] = a;
    return r;
}
export function remove(k, d) {
    if (!d.hasOwnProperty(k)) {
        return d;
    }
    const r = Object.assign({}, d);
    delete r[k];
    return r;
}
/**
 * Delete a key and value from a map, returning the value as well as the subsequent map
 *
 * @since 1.10.0
 */
export const pop = (k, d) => {
    const a = lookup(k, d);
    return a.isNone() ? none : optionSome([a.value, remove(k, d)]);
};
/**
 * Test whether one record contains all of the keys and values contained in another record
 *
 * @since 1.14.0
 */
export const isSubrecord = (S) => (d1, d2) => {
    for (let k in d1) {
        if (!d2.hasOwnProperty(k) || !S.equals(d1[k], d2[k])) {
            return false;
        }
    }
    return true;
};
/**
 * Use `isSubrecord` instead
 * @since 1.10.0
 * @deprecated
 */
export const isSubdictionary = isSubrecord;
export function getSetoid(S) {
    const isSubrecordS = isSubrecord(S);
    return fromEquals((x, y) => isSubrecordS(x, y) && isSubrecordS(y, x));
}
export function getMonoid(S) {
    // tslint:disable-next-line: deprecation
    return getDictionaryMonoid(S);
}
/**
 * Lookup the value for a key in a record
 * @since 1.10.0
 */
export const lookup = (key, fa) => {
    return fa.hasOwnProperty(key) ? optionSome(fa[key]) : none;
};
export function filter(fa, p) {
    return filterWithKey(fa, (_, a) => p(a));
}
/**
 * @since 1.10.0
 */
export const empty = {};
export function mapWithKey(fa, f) {
    const r = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        r[key] = f(key, fa[key]);
    }
    return r;
}
export function map(fa, f) {
    return mapWithKey(fa, (_, a) => f(a));
}
/**
 * @since 1.10.0
 */
export const reduce = (fa, b, f) => {
    return reduceWithKey(fa, b, (_, b, a) => f(b, a));
};
/**
 * @since 1.10.0
 */
export const foldMap = (M) => {
    const foldMapWithKeyM = foldMapWithKey(M);
    return (fa, f) => foldMapWithKeyM(fa, (_, a) => f(a));
};
/**
 * @since 1.10.0
 */
export const foldr = (fa, b, f) => {
    return foldrWithKey(fa, b, (_, a, b) => f(a, b));
};
export function reduceWithKey(fa, b, f) {
    let out = b;
    const keys = Object.keys(fa).sort();
    const len = keys.length;
    for (let i = 0; i < len; i++) {
        const k = keys[i];
        out = f(k, out, fa[k]);
    }
    return out;
}
/**
 * @since 1.12.0
 */
export const foldMapWithKey = (M) => (fa, f) => {
    let out = M.empty;
    const keys = Object.keys(fa).sort();
    const len = keys.length;
    for (let i = 0; i < len; i++) {
        const k = keys[i];
        out = M.concat(out, f(k, fa[k]));
    }
    return out;
};
export function foldrWithKey(fa, b, f) {
    let out = b;
    const keys = Object.keys(fa).sort();
    const len = keys.length;
    for (let i = len - 1; i >= 0; i--) {
        const k = keys[i];
        out = f(k, fa[k], out);
    }
    return out;
}
/**
 * Create a record with one key/value pair
 *
 * @since 1.10.0
 */
export const singleton = (k, a) => {
    return { [k]: a };
};
export function traverseWithKey(F) {
    return (ta, f) => {
        const keys = Object.keys(ta);
        if (keys.length === 0) {
            return F.of(empty);
        }
        let fr = F.of({});
        for (const key of keys) {
            fr = F.ap(F.map(fr, r => (b) => {
                r[key] = b;
                return r;
            }), f(key, ta[key]));
        }
        return fr;
    };
}
export function traverse(F) {
    const traverseWithKeyF = traverseWithKey(F);
    return (ta, f) => traverseWithKeyF(ta, (_, a) => f(a));
}
export function sequence(F) {
    const traverseWithKeyF = traverseWithKey(F);
    return ta => traverseWithKeyF(ta, (_, a) => a);
}
/**
 * @since 1.10.0
 */
export const compact = (fa) => {
    const r = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        const optionA = fa[key];
        if (optionA.isSome()) {
            r[key] = optionA.value;
        }
    }
    return r;
};
/**
 * @since 1.10.0
 */
export const partitionMap = (fa, f) => {
    return partitionMapWithKey(fa, (_, a) => f(a));
};
/**
 * @since 1.10.0
 */
export const partition = (fa, p) => {
    return partitionWithKey(fa, (_, a) => p(a));
};
/**
 * @since 1.10.0
 */
export const separate = (fa) => {
    const left = {};
    const right = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        const e = fa[key];
        if (e.isLeft()) {
            left[key] = e.value;
        }
        else {
            right[key] = e.value;
        }
    }
    return {
        left,
        right
    };
};
export function wither(F) {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), compact);
}
export function wilt(F) {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), separate);
}
/**
 * @since 1.10.0
 */
export const filterMap = (fa, f) => {
    return filterMapWithKey(fa, (_, a) => f(a));
};
export function partitionMapWithKey(fa, f) {
    const left = {};
    const right = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        const e = f(key, fa[key]);
        if (e.isLeft()) {
            left[key] = e.value;
        }
        else {
            right[key] = e.value;
        }
    }
    return {
        left,
        right
    };
}
export function partitionWithKey(fa, p) {
    const left = {};
    const right = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        const a = fa[key];
        if (p(key, a)) {
            right[key] = a;
        }
        else {
            left[key] = a;
        }
    }
    return {
        left,
        right
    };
}
export function filterMapWithKey(fa, f) {
    const r = {};
    const keys = Object.keys(fa);
    for (const key of keys) {
        const optionB = f(key, fa[key]);
        if (optionB.isSome()) {
            r[key] = optionB.value;
        }
    }
    return r;
}
export function filterWithKey(fa, p) {
    const r = {};
    let changed = false;
    for (const key in fa) {
        if (fa.hasOwnProperty(key)) {
            const a = fa[key];
            if (p(key, a)) {
                r[key] = a;
            }
            else {
                changed = true;
            }
        }
    }
    return changed ? r : fa;
}
export function fromFoldable(
// tslint:disable-next-line: deprecation
F) {
    return (ta, f) => {
        return F.reduce(ta, {}, (b, [k, a]) => {
            b[k] = b.hasOwnProperty(k) ? f(b[k], a) : a;
            return b;
        });
    };
}
export function fromFoldableMap(M, 
// tslint:disable-next-line: deprecation
F) {
    return (ta, f) => {
        return F.reduce(ta, {}, (r, a) => {
            const [k, b] = f(a);
            r[k] = r.hasOwnProperty(k) ? M.concat(r[k], b) : b;
            return r;
        });
    };
}
/**
 * @since 1.14.0
 */
export function every(fa, predicate) {
    for (const k in fa) {
        if (!predicate(fa[k])) {
            return false;
        }
    }
    return true;
}
/**
 * @since 1.14.0
 */
export function some(fa, predicate) {
    for (const k in fa) {
        if (predicate(fa[k])) {
            return true;
        }
    }
    return false;
}
/**
 * @since 1.14.0
 */
export function elem(S) {
    return (a, fa) => some(fa, x => S.equals(x, a));
}
export function partitionMapWithIndex(fa, f) {
    return partitionMapWithKey(fa, f);
}
export function partitionWithIndex(fa, p) {
    return partitionWithKey(fa, p);
}
export function filterMapWithIndex(fa, f) {
    return filterMapWithKey(fa, f);
}
export function filterWithIndex(fa, p) {
    return filterWithKey(fa, p);
}
