/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation;

import com.carrotsearch.hppc.ObjectLookupContainer;
import com.carrotsearch.hppc.cursors.ObjectObjectCursor;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.client.Client;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.routing.RerouteService;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.set.Sets;

public class DiskThresholdMonitor {
    private static final Logger logger = LogManager.getLogger(DiskThresholdMonitor.class);
    private final DiskThresholdSettings diskThresholdSettings;
    private final Client client;
    private final Set<String> nodeHasPassedWatermark = Sets.newConcurrentHashSet();
    private final Supplier<ClusterState> clusterStateSupplier;
    private final LongSupplier currentTimeMillisSupplier;
    private final RerouteService rerouteService;
    private final AtomicLong lastRunTimeMillis = new AtomicLong(Long.MIN_VALUE);
    private final AtomicBoolean checkInProgress = new AtomicBoolean();

    public DiskThresholdMonitor(Settings settings, Supplier<ClusterState> clusterStateSupplier, ClusterSettings clusterSettings, Client client, LongSupplier currentTimeMillisSupplier, RerouteService rerouteService) {
        this.clusterStateSupplier = clusterStateSupplier;
        this.currentTimeMillisSupplier = currentTimeMillisSupplier;
        this.rerouteService = rerouteService;
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
        this.client = client;
    }

    private void warnAboutDiskIfNeeded(DiskUsage usage) {
        if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdFloodStage().getBytes()) {
            logger.warn("flood stage disk watermark [{}] exceeded on {}, all indices on this node will be marked read-only", (Object)this.diskThresholdSettings.getFreeBytesThresholdFloodStage(), (Object)usage);
        } else if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes()) {
            logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", (Object)this.diskThresholdSettings.getFreeBytesThresholdHigh(), (Object)usage);
        } else if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes()) {
            logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", (Object)this.diskThresholdSettings.getFreeBytesThresholdLow(), (Object)usage);
        }
        if (usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdFloodStage()) {
            logger.warn("flood stage disk watermark [{}] exceeded on {}, all indices on this node will be marked read-only", (Object)Strings.format1Decimals(100.0 - this.diskThresholdSettings.getFreeDiskThresholdFloodStage(), "%"), (Object)usage);
        } else if (usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
            logger.warn("high disk watermark [{}] exceeded on {}, shards will be relocated away from this node", (Object)Strings.format1Decimals(100.0 - this.diskThresholdSettings.getFreeDiskThresholdHigh(), "%"), (Object)usage);
        } else if (usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
            logger.info("low disk watermark [{}] exceeded on {}, replicas will not be assigned to this node", (Object)Strings.format1Decimals(100.0 - this.diskThresholdSettings.getFreeDiskThresholdLow(), "%"), (Object)usage);
        }
    }

    private void checkFinished() {
        boolean checkFinished = this.checkInProgress.compareAndSet(true, false);
        assert (checkFinished);
    }

    public void onNewInfo(ClusterInfo info) {
        if (!this.checkInProgress.compareAndSet(false, true)) {
            logger.info("skipping monitor as a check is already in progress");
            return;
        }
        ImmutableOpenMap<String, DiskUsage> usages = info.getNodeLeastAvailableDiskUsages();
        if (usages == null) {
            this.checkFinished();
            return;
        }
        boolean reroute = false;
        String explanation = "";
        long currentTimeMillis = this.currentTimeMillisSupplier.getAsLong();
        ObjectLookupContainer<String> nodes = usages.keys();
        for (String node : this.nodeHasPassedWatermark) {
            if (nodes.contains((Object)node)) continue;
            this.nodeHasPassedWatermark.remove(node);
        }
        ClusterState state = this.clusterStateSupplier.get();
        HashSet<String> indicesToMarkReadOnly = new HashSet<String>();
        for (ObjectObjectCursor<String, DiskUsage> objectObjectCursor : usages) {
            String node = (String)objectObjectCursor.key;
            DiskUsage usage = (DiskUsage)objectObjectCursor.value;
            this.warnAboutDiskIfNeeded(usage);
            if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdFloodStage().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdFloodStage()) {
                RoutingNode routingNode = state.getRoutingNodes().node(node);
                if (routingNode == null) continue;
                for (ShardRouting routing : routingNode) {
                    indicesToMarkReadOnly.add(routing.index().getName());
                }
                continue;
            }
            if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdHigh().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdHigh()) {
                if (this.lastRunTimeMillis.get() < currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                    reroute = true;
                    explanation = "high disk watermark exceeded on one or more nodes";
                } else {
                    logger.debug("high disk watermark exceeded on {} but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
                }
                this.nodeHasPassedWatermark.add(node);
                continue;
            }
            if (usage.getFreeBytes() < this.diskThresholdSettings.getFreeBytesThresholdLow().getBytes() || usage.getFreeDiskAsPercentage() < this.diskThresholdSettings.getFreeDiskThresholdLow()) {
                this.nodeHasPassedWatermark.add(node);
                continue;
            }
            if (!this.nodeHasPassedWatermark.contains(node)) continue;
            if (this.lastRunTimeMillis.get() < currentTimeMillis - this.diskThresholdSettings.getRerouteInterval().millis()) {
                reroute = true;
                explanation = "one or more nodes has gone under the high or low watermark";
                this.nodeHasPassedWatermark.remove(node);
                continue;
            }
            logger.debug("{} has gone below a disk threshold, but an automatic reroute has occurred in the last [{}], skipping reroute", (Object)node, (Object)this.diskThresholdSettings.getRerouteInterval());
        }
        GroupedActionListener listener = new GroupedActionListener(ActionListener.wrap(this::checkFinished), 2);
        if (reroute) {
            logger.info("rerouting shards: [{}]", (Object)explanation);
            this.rerouteService.reroute("disk threshold monitor", ActionListener.wrap(r -> {
                this.setLastRunTimeMillis();
                listener.onResponse(r);
            }, e -> {
                logger.debug("reroute failed", (Throwable)e);
                this.setLastRunTimeMillis();
                listener.onFailure((Exception)e);
            }));
        } else {
            listener.onResponse(null);
        }
        indicesToMarkReadOnly.removeIf(index -> state.getBlocks().indexBlocked(ClusterBlockLevel.WRITE, (String)index));
        if (!indicesToMarkReadOnly.isEmpty()) {
            this.markIndicesReadOnly(indicesToMarkReadOnly, ActionListener.wrap(r -> {
                this.setLastRunTimeMillis();
                listener.onResponse(r);
            }, e -> {
                logger.debug("marking indices readonly failed", (Throwable)e);
                this.setLastRunTimeMillis();
                listener.onFailure((Exception)e);
            }));
        } else {
            listener.onResponse(null);
        }
    }

    private void setLastRunTimeMillis() {
        this.lastRunTimeMillis.getAndUpdate(l -> Math.max(l, this.currentTimeMillisSupplier.getAsLong()));
    }

    protected void markIndicesReadOnly(Set<String> indicesToMarkReadOnly, ActionListener<Void> listener) {
        this.client.admin().indices().prepareUpdateSettings(indicesToMarkReadOnly.toArray(Strings.EMPTY_ARRAY)).setSettings(Settings.builder().put("index.blocks.read_only_allow_delete", true).build()).execute(ActionListener.map(listener, r -> null));
    }
}

