import { phantom, toString } from './function';
import { fromEquals } from './Setoid';
export const URI = 'Validation';
export class Failure {
    constructor(value) {
        this.value = value;
        this._tag = 'Failure';
    }
    map(f) {
        return this;
    }
    bimap(f, g) {
        return new Failure(f(this.value));
    }
    reduce(b, f) {
        return b;
    }
    fold(failure, success) {
        return failure(this.value);
    }
    /** Returns the value from this `Success` or the given argument if this is a `Failure` */
    getOrElse(a) {
        return a;
    }
    /** Returns the value from this `Success` or the result of given argument if this is a `Failure` */
    getOrElseL(f) {
        return f(this.value);
    }
    mapFailure(f) {
        return new Failure(f(this.value));
    }
    swap() {
        return new Success(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `failure(${toString(this.value)})`;
    }
    /** Returns `true` if the validation is an instance of `Failure`, `false` otherwise */
    isFailure() {
        return true;
    }
    /** Returns `true` if the validation is an instance of `Success`, `false` otherwise */
    isSuccess() {
        return false;
    }
}
export class Success {
    constructor(value) {
        this.value = value;
        this._tag = 'Success';
    }
    map(f) {
        return new Success(f(this.value));
    }
    bimap(f, g) {
        return new Success(g(this.value));
    }
    reduce(b, f) {
        return f(b, this.value);
    }
    fold(failure, success) {
        return success(this.value);
    }
    getOrElse(a) {
        return this.value;
    }
    getOrElseL(f) {
        return this.value;
    }
    mapFailure(f) {
        return this;
    }
    swap() {
        return new Failure(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `success(${toString(this.value)})`;
    }
    isFailure() {
        return false;
    }
    isSuccess() {
        return true;
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (SL, SA) => {
    return {
        show: e => e.fold(l => `failure(${SL.show(l)})`, a => `success(${SA.show(a)})`)
    };
};
/**
 * @since 1.0.0
 */
export const getSetoid = (SL, SA) => {
    return fromEquals((x, y) => x.isFailure() ? y.isFailure() && SL.equals(x.value, y.value) : y.isSuccess() && SA.equals(x.value, y.value));
};
const map = (fa, f) => {
    return fa.map(f);
};
/**
 * @since 1.0.0
 */
export const success = (a) => {
    return new Success(a);
};
const of = success;
/**
 * @example
 * import { Validation, success, failure, getApplicative } from 'fp-ts/lib/Validation'
 * import { getArraySemigroup } from 'fp-ts/lib/Semigroup'
 *
 * interface Person {
 *   name: string
 *   age: number
 * }
 *
 * const person = (name: string) => (age: number): Person => ({ name, age })
 *
 * const validateName = (name: string): Validation<string[], string> =>
 *   name.length === 0 ? failure(['invalid name']) : success(name)
 *
 * const validateAge = (age: number): Validation<string[], number> =>
 *   age > 0 && age % 1 === 0 ? success(age) : failure(['invalid age'])
 *
 * const A = getApplicative(getArraySemigroup<string>())
 *
 * const validatePerson = (name: string, age: number): Validation<string[], Person> =>
 *   A.ap(A.map(validateName(name), person), validateAge(age))
 *
 * assert.deepStrictEqual(validatePerson('Nicolas Bourbaki', 45), success({ "name": "Nicolas Bourbaki", "age": 45 }))
 * assert.deepStrictEqual(validatePerson('Nicolas Bourbaki', -1), failure(["invalid age"]))
 * assert.deepStrictEqual(validatePerson('', 0), failure(["invalid name", "invalid age"]))
 *
 * @since 1.0.0
 */
export const getApplicative = (S) => {
    const ap = (fab, fa) => {
        return fab.isFailure()
            ? fa.isFailure()
                ? failure(S.concat(fab.value, fa.value))
                : failure(fab.value)
            : fa.isFailure()
                ? failure(fa.value)
                : success(fab.value(fa.value));
    };
    return {
        URI,
        _L: phantom,
        map,
        of,
        ap
    };
};
/**
 * **Note**: This function is here just to avoid switching to / from `Either`
 *
 * @since 1.0.0
 */
export const getMonad = (S) => {
    const chain = (fa, f) => {
        return fa.isFailure() ? failure(fa.value) : f(fa.value);
    };
    return Object.assign({}, getApplicative(S), { chain });
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.isFailure() ? M.empty : f(fa.value);
};
const foldr = (fa, b, f) => {
    return fa.isFailure() ? b : f(fa.value, b);
};
const traverse = (F) => (ta, f) => {
    return ta.isFailure() ? F.of(failure(ta.value)) : F.map(f(ta.value), of);
};
const sequence = (F) => (ta) => {
    return ta.isFailure() ? F.of(failure(ta.value)) : F.map(ta.value, of);
};
const bimap = (fla, f, g) => {
    return fla.bimap(f, g);
};
/**
 * @since 1.0.0
 */
export const failure = (l) => {
    return new Failure(l);
};
export function fromPredicate(predicate, f) {
    return a => (predicate(a) ? success(a) : failure(f(a)));
}
/**
 * @since 1.0.0
 */
export const fromEither = (e) => {
    return e.isLeft() ? failure(e.value) : success(e.value);
};
/**
 * Constructs a new `Validation` from a function that might throw
 *
 * @example
 * import { Validation, failure, success, tryCatch } from 'fp-ts/lib/Validation'
 *
 * const unsafeHead = <A>(as: Array<A>): A => {
 *   if (as.length > 0) {
 *     return as[0]
 *   } else {
 *     throw new Error('empty array')
 *   }
 * }
 *
 * const head = <A>(as: Array<A>): Validation<Error, A> => {
 *   return tryCatch(() => unsafeHead(as), e => (e instanceof Error ? e : new Error('unknown error')))
 * }
 *
 * assert.deepStrictEqual(head([]), failure(new Error('empty array')))
 * assert.deepStrictEqual(head([1, 2, 3]), success(1))
 *
 * @since 1.16.0
 */
export const tryCatch = (f, onError) => {
    try {
        return success(f());
    }
    catch (e) {
        return failure(onError(e));
    }
};
/**
 * @since 1.0.0
 */
export const getSemigroup = (SL, SA) => {
    const concat = (fx, fy) => {
        return fx.isFailure()
            ? fy.isFailure()
                ? failure(SL.concat(fx.value, fy.value))
                : failure(fx.value)
            : fy.isFailure()
                ? failure(fy.value)
                : success(SA.concat(fx.value, fy.value));
    };
    return {
        concat
    };
};
/**
 * @since 1.0.0
 */
export const getMonoid = (SL, SA) => {
    return Object.assign({}, getSemigroup(SL, SA), { empty: success(SA.empty) });
};
/**
 * @since 1.0.0
 */
export const getAlt = (S) => {
    const alt = (fx, fy) => {
        return fx.isFailure() ? (fy.isFailure() ? failure(S.concat(fx.value, fy.value)) : fy) : fx;
    };
    return {
        URI,
        _L: phantom,
        map,
        alt
    };
};
/**
 * Returns `true` if the validation is an instance of `Failure`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isFailure = (fa) => {
    return fa.isFailure();
};
/**
 * Returns `true` if the validation is an instance of `Success`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isSuccess = (fa) => {
    return fa.isSuccess();
};
/**
 * Builds `Compactable` instance for `Validation` given `Monoid` for the failure side
 *
 * @since 1.7.0
 */
export function getCompactable(ML) {
    const compact = (fa) => {
        if (fa.isFailure()) {
            return fa;
        }
        if (fa.value.isNone()) {
            return failure(ML.empty);
        }
        return success(fa.value.value);
    };
    const separate = (fa) => {
        if (fa.isFailure()) {
            return {
                left: fa,
                right: fa
            };
        }
        if (fa.value.isLeft()) {
            return {
                left: success(fa.value.value),
                right: failure(ML.empty)
            };
        }
        return {
            left: failure(ML.empty),
            right: success(fa.value.value)
        };
    };
    return {
        URI,
        _L: phantom,
        compact,
        separate
    };
}
/**
 * Builds `Filterable` instance for `Validation` given `Monoid` for the left side
 *
 * @since 1.7.0
 */
export function getFilterable(ML) {
    const C = getCompactable(ML);
    const partitionMap = (fa, f) => {
        if (fa.isFailure()) {
            return {
                left: fa,
                right: fa
            };
        }
        const e = f(fa.value);
        if (e.isLeft()) {
            return {
                left: success(e.value),
                right: failure(ML.empty)
            };
        }
        return {
            left: failure(ML.empty),
            right: success(e.value)
        };
    };
    const partition = (fa, p) => {
        if (fa.isFailure()) {
            return {
                left: fa,
                right: fa
            };
        }
        if (p(fa.value)) {
            return {
                left: failure(ML.empty),
                right: success(fa.value)
            };
        }
        return {
            left: success(fa.value),
            right: failure(ML.empty)
        };
    };
    const filterMap = (fa, f) => {
        if (fa.isFailure()) {
            return fa;
        }
        const optionB = f(fa.value);
        if (optionB.isSome()) {
            return success(optionB.value);
        }
        return failure(ML.empty);
    };
    const filter = (fa, p) => {
        if (fa.isFailure()) {
            return fa;
        }
        const a = fa.value;
        if (p(a)) {
            return success(a);
        }
        return failure(ML.empty);
    };
    return Object.assign({}, C, { map,
        partitionMap,
        filterMap,
        partition,
        filter });
}
/**
 * Builds `Witherable` instance for `Validation` given `Monoid` for the left side
 *
 * @since 1.7.0
 */
export function getWitherable(ML) {
    const filterableValidation = getFilterable(ML);
    const wither = (F) => {
        const traverseF = traverse(F);
        return (wa, f) => F.map(traverseF(wa, f), filterableValidation.compact);
    };
    const wilt = (F) => {
        const traverseF = traverse(F);
        return (wa, f) => F.map(traverseF(wa, f), filterableValidation.separate);
    };
    return Object.assign({}, filterableValidation, { traverse,
        reduce,
        wither,
        wilt });
}
const throwError = failure;
/**
 * @since 1.16.0
 */
export const getMonadThrow = (S) => {
    return Object.assign({}, getMonad(S), { throwError,
        fromEither, fromOption: (o, e) => (o.isNone() ? throwError(e) : of(o.value)) });
};
/**
 * @since 1.0.0
 */
export const validation = {
    URI,
    map,
    bimap,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence
};
