import { array, findFirst as arrayFindFirst, findIndex as arrayFindIndex, findLast as arrayFindLast, findLastIndex as arrayFindLastIndex, insertAt as arrayInsertAt, last, lookup, sort, updateAt as arrayUpdateAt } from './Array';
import { compose, toString } from './function';
import { none, some } from './Option';
import { fold, getJoinSemigroup, getMeetSemigroup } from './Semigroup';
import { fromEquals, getArraySetoid } from './Setoid';
export const URI = 'NonEmptyArray';
/**
 * @since 1.0.0
 */
export class NonEmptyArray {
    constructor(head, tail) {
        this.head = head;
        this.tail = tail;
    }
    /**
     * Converts this `NonEmptyArray` to a plain `Array`
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).toArray(), [1, 2, 3])
     */
    toArray() {
        return [this.head, ...this.tail];
    }
    /**
     * Converts this `NonEmptyArray` to a plain `Array` using the given map function
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.deepStrictEqual(new NonEmptyArray('a', ['bb', 'ccc']).toArrayMap(s => s.length), [1, 2, 3])
     *
     * @since 1.14.0
     */
    toArrayMap(f) {
        return [f(this.head), ...this.tail.map(a => f(a))];
    }
    /**
     * Concatenates this `NonEmptyArray` and passed `Array`
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.deepStrictEqual(new NonEmptyArray<number>(1, []).concatArray([2]), new NonEmptyArray(1, [2]))
     */
    concatArray(as) {
        return new NonEmptyArray(this.head, [...this.tail, ...as]);
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const double = (n: number): number => n * 2
     * assert.deepStrictEqual(new NonEmptyArray(1, [2]).map(double), new NonEmptyArray(2, [4]))
     */
    map(f) {
        return new NonEmptyArray(f(this.head), this.tail.map(f));
    }
    mapWithIndex(f) {
        return new NonEmptyArray(f(0, this.head), array.mapWithIndex(this.tail, (i, a) => f(i + 1, a)));
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const x = new NonEmptyArray(1, [2])
     * const double = (n: number): number => n * 2
     * assert.deepStrictEqual(x.ap(new NonEmptyArray(double, [double])).toArray(), [2, 4, 2, 4])
     */
    ap(fab) {
        return fab.chain(f => this.map(f)); // <= derived
    }
    /**
     * Flipped version of `ap`
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const x = new NonEmptyArray(1, [2])
     * const double = (n: number) => n * 2
     * assert.deepStrictEqual(new NonEmptyArray(double, [double]).ap_(x).toArray(), [2, 4, 2, 4])
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const x = new NonEmptyArray(1, [2])
     * const f = (a: number) => new NonEmptyArray(a, [4])
     * assert.deepStrictEqual(x.chain(f).toArray(), [1, 4, 2, 4])
     */
    chain(f) {
        return f(this.head).concatArray(array.chain(this.tail, a => f(a).toArray()));
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const x = new NonEmptyArray(1, [2])
     * const y = new NonEmptyArray(3, [4])
     * assert.deepStrictEqual(x.concat(y).toArray(), [1, 2, 3, 4])
     */
    concat(y) {
        return this.concatArray(y.toArray());
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * const x = new NonEmptyArray('a', ['b'])
     * assert.strictEqual(x.reduce('', (b, a) => b + a), 'ab')
     */
    reduce(b, f) {
        return array.reduce(this.toArray(), b, f);
    }
    /**
     * @since 1.12.0
     */
    reduceWithIndex(b, f) {
        return array.reduceWithIndex(this.toArray(), b, f);
    }
    /**
     * @since 1.12.0
     */
    foldr(b, f) {
        return this.foldrWithIndex(b, (_, a, b) => f(a, b));
    }
    /**
     * @since 1.12.0
     */
    foldrWithIndex(b, f) {
        return f(0, this.head, this.tail.reduceRight((acc, a, i) => f(i + 1, a, acc), b));
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { fold, monoidSum } from 'fp-ts/lib/Monoid'
     *
     * const sum = (as: NonEmptyArray<number>) => fold(monoidSum)(as.toArray())
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3, 4]).extend(sum), new NonEmptyArray(10, [9, 7, 4]))
     */
    extend(f) {
        return unsafeFromArray(array.extend(this.toArray(), as => f(unsafeFromArray(as))));
    }
    /**
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.strictEqual(new NonEmptyArray(1, [2, 3]).extract(), 1)
     */
    extract() {
        return this.head;
    }
    /**
     * Same as `toString`
     */
    inspect() {
        return this.toString();
    }
    /**
     * Return stringified representation of this `NonEmptyArray`
     */
    toString() {
        return `new NonEmptyArray(${toString(this.head)}, ${toString(this.tail)})`;
    }
    /**
     * Gets minimum of this `NonEmptyArray` using specified `Ord` instance
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { ordNumber } from 'fp-ts/lib/Ord'
     *
     * assert.strictEqual(new NonEmptyArray(1, [2, 3]).min(ordNumber), 1)
     *
     * @since 1.3.0
     */
    min(ord) {
        return fold(getMeetSemigroup(ord))(this.head)(this.tail);
    }
    /**
     * Gets maximum of this `NonEmptyArray` using specified `Ord` instance
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { ordNumber } from 'fp-ts/lib/Ord'
     *
     * assert.strictEqual(new NonEmptyArray(1, [2, 3]).max(ordNumber), 3)
     *
     * @since 1.3.0
     */
    max(ord) {
        return fold(getJoinSemigroup(ord))(this.head)(this.tail);
    }
    /**
     * Gets last element of this `NonEmptyArray`
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.strictEqual(new NonEmptyArray(1, [2, 3]).last(), 3)
     * assert.strictEqual(new NonEmptyArray(1, []).last(), 1)
     *
     * @since 1.6.0
     */
    last() {
        return last(this.tail).getOrElse(this.head);
    }
    /**
     * Sorts this `NonEmptyArray` using specified `Ord` instance
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { ordNumber } from 'fp-ts/lib/Ord'
     *
     * assert.deepStrictEqual(new NonEmptyArray(3, [2, 1]).sort(ordNumber), new NonEmptyArray(1, [2, 3]))
     *
     * @since 1.6.0
     */
    sort(ord) {
        return unsafeFromArray(sort(ord)(this.toArray()));
    }
    /**
     * Reverts this `NonEmptyArray`
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).reverse(), new NonEmptyArray(3, [2, 1]))
     *
     * @since 1.6.0
     */
    reverse() {
        return unsafeFromArray(this.toArray().reverse());
    }
    /**
     * @since 1.10.0
     */
    length() {
        return 1 + this.tail.length;
    }
    /**
     * This function provides a safe way to read a value at a particular index from an NonEmptyArray
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).lookup(1), some(2))
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).lookup(3), none)
     *
     * @since 1.14.0
     */
    lookup(i) {
        return i === 0 ? some(this.head) : lookup(i - 1, this.tail);
    }
    /**
     * Use `lookup` instead
     * @since 1.11.0
     * @deprecated
     */
    index(i) {
        return this.lookup(i);
    }
    findFirst(predicate) {
        return predicate(this.head) ? some(this.head) : arrayFindFirst(this.tail, predicate);
    }
    findLast(predicate) {
        const a = arrayFindLast(this.tail, predicate);
        return a.isSome() ? a : predicate(this.head) ? some(this.head) : none;
    }
    /**
     * Find the first index for which a predicate holds
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).findIndex(x => x === 2), some(1))
     * assert.deepStrictEqual(new NonEmptyArray<number>(1, []).findIndex(x => x === 2), none)
     *
     * @since 1.11.0
     */
    findIndex(predicate) {
        if (predicate(this.head)) {
            return some(0);
        }
        else {
            const i = arrayFindIndex(this.tail, predicate);
            return i.isSome() ? some(i.value + 1) : none;
        }
    }
    /**
     * Returns the index of the last element of the list which matches the predicate
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * interface X {
     *   a: number
     *   b: number
     * }
     * const xs: NonEmptyArray<X> = new NonEmptyArray({ a: 1, b: 0 }, [{ a: 1, b: 1 }])
     * assert.deepStrictEqual(xs.findLastIndex(x => x.a === 1), some(1))
     * assert.deepStrictEqual(xs.findLastIndex(x => x.a === 4), none)
     *
     * @since 1.11.0
     */
    findLastIndex(predicate) {
        const i = arrayFindLastIndex(this.tail, predicate);
        return i.isSome() ? some(i.value + 1) : predicate(this.head) ? some(0) : none;
    }
    /**
     * Insert an element at the specified index, creating a new NonEmptyArray, or returning `None` if the index is out of bounds
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { some } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3, 4]).insertAt(2, 5), some(new NonEmptyArray(1, [2, 5, 3, 4])))
     *
     * @since 1.11.0
     */
    insertAt(i, a) {
        if (i === 0) {
            return some(new NonEmptyArray(a, this.toArray()));
        }
        else {
            const t = arrayInsertAt(i - 1, a, this.tail);
            return t.isSome() ? some(new NonEmptyArray(this.head, t.value)) : none;
        }
    }
    /**
     * Change the element at the specified index, creating a new NonEmptyArray, or returning `None` if the index is out of bounds
     *
     * @example
     * import { NonEmptyArray } from 'fp-ts/lib/NonEmptyArray'
     * import { some, none } from 'fp-ts/lib/Option'
     *
     * assert.deepStrictEqual(new NonEmptyArray(1, [2, 3]).updateAt(1, 1), some(new NonEmptyArray(1, [1, 3])))
     * assert.deepStrictEqual(new NonEmptyArray(1, []).updateAt(1, 1), none)
     *
     * @since 1.11.0
     */
    updateAt(i, a) {
        if (i === 0) {
            return this.head === a ? some(this) : some(new NonEmptyArray(a, this.tail));
        }
        else {
            const t = arrayUpdateAt(i - 1, a, this.tail);
            return t.isSome() ? (t.value === this.tail ? some(this) : some(new NonEmptyArray(this.head, t.value))) : none;
        }
    }
    filter(predicate) {
        return this.filterWithIndex((_, a) => predicate(a));
    }
    /**
     * @since 1.12.0
     */
    filterWithIndex(predicate) {
        const t = array.filterWithIndex(this.tail, (i, a) => predicate(i + 1, a));
        return predicate(0, this.head) ? some(new NonEmptyArray(this.head, t)) : fromArray(t);
    }
    /**
     * @since 1.14.0
     */
    some(predicate) {
        return predicate(this.head) || this.tail.some(a => predicate(a));
    }
    /**
     * @since 1.14.0
     */
    every(predicate) {
        return predicate(this.head) && this.tail.every(a => predicate(a));
    }
}
const unsafeFromArray = (as) => {
    return new NonEmptyArray(as[0], as.slice(1));
};
/**
 * Builds a `NonEmptyArray` from an `Array` returning `none` if `as` is an empty array
 *
 * @since 1.0.0
 */
export const fromArray = (as) => {
    return as.length > 0 ? some(unsafeFromArray(as)) : none;
};
const map = (fa, f) => {
    return fa.map(f);
};
const mapWithIndex = (fa, f) => {
    return fa.mapWithIndex(f);
};
const of = (a) => {
    return new NonEmptyArray(a, []);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const concat = (fx, fy) => {
    return fx.concat(fy);
};
/**
 * Builds a `Semigroup` instance for `NonEmptyArray`
 *
 * @since 1.0.0
 */
export const getSemigroup = () => {
    return { concat };
};
/**
 * @example
 * import { NonEmptyArray, getSetoid } from 'fp-ts/lib/NonEmptyArray'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * const S = getSetoid(setoidNumber)
 * assert.strictEqual(S.equals(new NonEmptyArray(1, []), new NonEmptyArray(1, [])), true)
 * assert.strictEqual(S.equals(new NonEmptyArray(1, []), new NonEmptyArray(1, [2])), false)
 *
 * @since 1.14.0
 */
export const getSetoid = (S) => {
    const setoidTail = getArraySetoid(S);
    return fromEquals((x, y) => S.equals(x.head, y.head) && setoidTail.equals(x.tail, y.tail));
};
/**
 * Group equal, consecutive elements of an array into non empty arrays.
 *
 * @example
 * import { NonEmptyArray, group } from 'fp-ts/lib/NonEmptyArray'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(group(ordNumber)([1, 2, 1, 1]), [
 *   new NonEmptyArray(1, []),
 *   new NonEmptyArray(2, []),
 *   new NonEmptyArray(1, [1])
 * ])
 *
 * @since 1.7.0
 */
export const group = (S) => (as) => {
    const r = [];
    const len = as.length;
    if (len === 0) {
        return r;
    }
    let head = as[0];
    let tail = [];
    for (let i = 1; i < len; i++) {
        const x = as[i];
        if (S.equals(x, head)) {
            tail.push(x);
        }
        else {
            r.push(new NonEmptyArray(head, tail));
            head = x;
            tail = [];
        }
    }
    r.push(new NonEmptyArray(head, tail));
    return r;
};
/**
 * Sort and then group the elements of an array into non empty arrays.
 *
 * @example
 * import { NonEmptyArray, groupSort } from 'fp-ts/lib/NonEmptyArray'
 * import { ordNumber } from 'fp-ts/lib/Ord'
 *
 * assert.deepStrictEqual(groupSort(ordNumber)([1, 2, 1, 1]), [new NonEmptyArray(1, [1, 1]), new NonEmptyArray(2, [])])
 *
 * @since 1.7.0
 */
export const groupSort = (O) => {
    return compose(group(O), sort(O));
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.tail.reduce((acc, a) => M.concat(acc, f(a)), f(fa.head));
};
const foldr = (fa, b, f) => {
    return fa.foldr(b, f);
};
const reduceWithIndex = (fa, b, f) => {
    return fa.reduceWithIndex(b, f);
};
const foldMapWithIndex = (M) => (fa, f) => {
    return fa.tail.reduce((acc, a, i) => M.concat(acc, f(i + 1, a)), f(0, fa.head));
};
const foldrWithIndex = (fa, b, f) => {
    return fa.foldrWithIndex(b, f);
};
const extend = (fa, f) => {
    return fa.extend(f);
};
const extract = (fa) => {
    return fa.extract();
};
function traverse(F) {
    const traverseWithIndexF = traverseWithIndex(F);
    return (ta, f) => traverseWithIndexF(ta, (_, a) => f(a));
}
function sequence(F) {
    const sequenceF = array.sequence(F);
    return (ta) => F.ap(F.map(ta.head, a => (as) => new NonEmptyArray(a, as)), sequenceF(ta.tail));
}
/**
 * Splits an array into sub-non-empty-arrays stored in an object, based on the result of calling a `string`-returning
 * function on each element, and grouping the results according to values returned
 *
 * @example
 * import { NonEmptyArray, groupBy } from 'fp-ts/lib/NonEmptyArray'
 *
 * assert.deepStrictEqual(groupBy(['foo', 'bar', 'foobar'], a => String(a.length)), {
 *   '3': new NonEmptyArray('foo', ['bar']),
 *   '6': new NonEmptyArray('foobar', [])
 * })
 *
 * @since 1.10.0
 */
export const groupBy = (as, f) => {
    const r = {};
    for (const a of as) {
        const k = f(a);
        if (r.hasOwnProperty(k)) {
            r[k].tail.push(a);
        }
        else {
            r[k] = new NonEmptyArray(a, []);
        }
    }
    return r;
};
const traverseWithIndex = (F) => {
    const traverseWithIndexF = array.traverseWithIndex(F);
    return (ta, f) => {
        const fb = f(0, ta.head);
        const fbs = traverseWithIndexF(ta.tail, (i, a) => f(i + 1, a));
        return F.ap(F.map(fb, b => (bs) => new NonEmptyArray(b, bs)), fbs);
    };
};
/**
 * @since 1.0.0
 */
export const nonEmptyArray = {
    URI,
    extend,
    extract,
    map,
    mapWithIndex,
    of,
    ap,
    chain,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    reduceWithIndex,
    foldMapWithIndex,
    foldrWithIndex,
    traverseWithIndex
};
