"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adoptToHapiOnPostAuthFormat = adoptToHapiOnPostAuthFormat;

var _boom = _interopRequireDefault(require("boom"));

var _router = require("../router");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var ResultType;

(function (ResultType) {
  ResultType["next"] = "next";
  ResultType["redirected"] = "redirected";
  ResultType["rejected"] = "rejected";
})(ResultType || (ResultType = {}));

const postAuthResult = {
  next() {
    return {
      type: ResultType.next
    };
  },

  redirected(url) {
    return {
      type: ResultType.redirected,
      url
    };
  },

  rejected(error, options = {}) {
    return {
      type: ResultType.rejected,
      error,
      statusCode: options.statusCode
    };
  },

  isValid(candidate) {
    return candidate && (candidate.type === ResultType.next || candidate.type === ResultType.rejected || candidate.type === ResultType.redirected);
  },

  isNext(result) {
    return result.type === ResultType.next;
  },

  isRedirected(result) {
    return result.type === ResultType.redirected;
  },

  isRejected(result) {
    return result.type === ResultType.rejected;
  }

};
/**
 * @public
 * A tool set defining an outcome of OnPostAuth interceptor for incoming request.
 */

const toolkit = {
  next: postAuthResult.next,
  redirected: postAuthResult.redirected,
  rejected: postAuthResult.rejected
};
/**
 * @public
 * Adopt custom request interceptor to Hapi lifecycle system.
 * @param fn - an extension point allowing to perform custom logic for
 * incoming HTTP requests.
 */

function adoptToHapiOnPostAuthFormat(fn) {
  return async function interceptRequest(request, h) {
    try {
      const result = await fn(_router.KibanaRequest.from(request), toolkit);

      if (!postAuthResult.isValid(result)) {
        throw new Error(`Unexpected result from OnPostAuth. Expected OnPostAuthResult, but given: ${result}.`);
      }

      if (postAuthResult.isNext(result)) {
        return h.continue;
      }

      if (postAuthResult.isRedirected(result)) {
        return h.redirect(result.url).takeover();
      }

      const {
        error,
        statusCode
      } = result;
      return _boom.default.boomify(error, {
        statusCode
      });
    } catch (error) {
      return _boom.default.internal(error.message, {
        statusCode: 500
      });
    }
  };
}