import { array, cons, drop, empty, isEmpty, isOutOfBound, snoc, take, getShow as getArrayShow } from './Array';
import { decrement, increment, toString } from './function';
import { none, some } from './Option';
export const URI = 'Zipper';
/**
 * @since 1.9.0
 */
export class Zipper {
    constructor(lefts, focus, rights) {
        this.lefts = lefts;
        this.focus = focus;
        this.rights = rights;
        this.length = lefts.length + 1 + rights.length;
    }
    /**
     * Update the focus in this zipper.
     * @since 1.9.0
     */
    update(a) {
        return new Zipper(this.lefts, a, this.rights);
    }
    /**
     * Apply `f` to the focus and update with the result.
     * @since 1.9.0
     */
    modify(f) {
        return this.update(f(this.focus));
    }
    /**
     * @since 1.9.0
     */
    toArray() {
        return snoc(this.lefts, this.focus).concat(this.rights);
    }
    /**
     * @since 1.9.0
     */
    isOutOfBound(index) {
        return index < 0 || index >= this.length;
    }
    /**
     * Moves focus in the zipper, or `None` if there is no such element.
     * @since 1.9.0
     */
    move(f) {
        const newIndex = f(this.lefts.length);
        if (this.isOutOfBound(newIndex)) {
            return none;
        }
        else {
            return fromArray(this.toArray(), newIndex);
        }
    }
    /**
     * @since 1.9.0
     */
    up() {
        return this.move(decrement);
    }
    /**
     * @since 1.9.0
     */
    down() {
        return this.move(increment);
    }
    /**
     * Moves focus to the start of the zipper.
     * @since 1.9.0
     */
    start() {
        if (isEmpty(this.lefts)) {
            return this;
        }
        else {
            return new Zipper(empty, this.lefts[0], snoc(drop(1, this.lefts), this.focus).concat(this.rights));
        }
    }
    /**
     * Moves focus to the end of the zipper.
     * @since 1.9.0
     */
    end() {
        const len = this.rights.length;
        if (len === 0) {
            return this;
        }
        else {
            return new Zipper(snoc(this.lefts, this.focus).concat(take(len - 1, this.rights)), this.rights[len - 1], empty);
        }
    }
    /**
     * Inserts an element to the left of focus and focuses on the new element.
     * @since 1.9.0
     */
    insertLeft(a) {
        return new Zipper(this.lefts, a, cons(this.focus, this.rights));
    }
    /**
     * Inserts an element to the right of focus and focuses on the new element.
     * @since 1.9.0
     */
    insertRight(a) {
        return new Zipper(snoc(this.lefts, this.focus), a, this.rights);
    }
    /**
     * Deletes the element at focus and moves the focus to the left. If there is no element on the left,
     * focus is moved to the right.
     * @since 1.9.0
     */
    deleteLeft() {
        const len = this.lefts.length;
        return fromArray(this.lefts.concat(this.rights), len > 0 ? len - 1 : 0);
    }
    /**
     * Deletes the element at focus and moves the focus to the right. If there is no element on the right,
     * focus is moved to the left.
     * @since 1.9.0
     */
    deleteRight() {
        const lenl = this.lefts.length;
        const lenr = this.rights.length;
        return fromArray(this.lefts.concat(this.rights), lenr > 0 ? lenl : lenl - 1);
    }
    /**
     * @since 1.9.0
     */
    map(f) {
        return new Zipper(this.lefts.map(f), f(this.focus), this.rights.map(f));
    }
    /**
     * @since 1.9.0
     */
    ap(fab) {
        return new Zipper(array.ap(fab.lefts, this.lefts), fab.focus(this.focus), array.ap(fab.rights, this.rights));
    }
    /**
     * @since 1.9.0
     */
    reduce(b, f) {
        return this.rights.reduce(f, f(this.lefts.reduce(f, b), this.focus));
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Zipper(${toString(this.lefts)}, ${toString(this.focus)}, ${toString(this.rights)})`;
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    const SA = getArrayShow(S);
    return {
        show: z => `new Zipper(${SA.show(z.lefts)}, ${S.show(z.focus)}, ${SA.show(z.rights)})`
    };
};
/**
 * @since 1.9.0
 */
export const fromArray = (as, focusIndex = 0) => {
    if (isEmpty(as) || isOutOfBound(focusIndex, as)) {
        return none;
    }
    else {
        return some(new Zipper(take(focusIndex, as), as[focusIndex], drop(focusIndex + 1, as)));
    }
};
/**
 * @since 1.9.0
 */
export const fromNonEmptyArray = (nea) => {
    return new Zipper(empty, nea.head, nea.tail);
};
/**
 * @since 1.17.0
 */
export const fromNonEmptyArray2v = (nea) => {
    return new Zipper(empty, nea[0], nea.slice(1));
};
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new Zipper(empty, a, empty);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    const lefts = fa.lefts.reduce((acc, a) => M.concat(acc, f(a)), M.empty);
    const rights = fa.rights.reduce((acc, a) => M.concat(acc, f(a)), M.empty);
    return M.concat(M.concat(lefts, f(fa.focus)), rights);
};
const foldr = (fa, b, f) => {
    const rights = fa.rights.reduceRight((acc, a) => f(a, acc), b);
    const focus = f(fa.focus, rights);
    return fa.lefts.reduceRight((acc, a) => f(a, acc), focus);
};
function traverse(F) {
    const traverseF = array.traverse(F);
    return (ta, f) => F.ap(F.ap(F.map(traverseF(ta.lefts, f), lefts => (focus) => (rights) => new Zipper(lefts, focus, rights)), f(ta.focus)), traverseF(ta.rights, f));
}
function sequence(F) {
    const sequenceF = array.sequence(F);
    return (ta) => F.ap(F.ap(F.map(sequenceF(ta.lefts), lefts => (focus) => (rights) => new Zipper(lefts, focus, rights)), ta.focus), sequenceF(ta.rights));
}
const extract = (fa) => {
    return fa.focus;
};
const extend = (fa, f) => {
    const lefts = fa.lefts.map((a, i) => f(new Zipper(take(i, fa.lefts), a, snoc(drop(i + 1, fa.lefts), fa.focus).concat(fa.rights))));
    const rights = fa.rights.map((a, i) => f(new Zipper(snoc(fa.lefts, fa.focus).concat(take(i, fa.rights)), a, drop(i + 1, fa.rights))));
    return new Zipper(lefts, f(fa), rights);
};
/**
 * @since 1.9.0
 */
export const getSemigroup = (S) => {
    return {
        concat: (x, y) => new Zipper(x.lefts.concat(y.lefts), S.concat(x.focus, y.focus), x.rights.concat(y.rights))
    };
};
/**
 * @since 1.9.0
 */
export const getMonoid = (M) => {
    return Object.assign({}, getSemigroup(M), { empty: new Zipper(empty, M.empty, empty) });
};
/**
 * @since 1.9.0
 */
export const zipper = {
    URI,
    map,
    of,
    ap,
    extend,
    extract,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence
};
