/**
 * @file Represents a value of one of two possible types (a disjoint union).
 *
 * An instance of `Either` is either an instance of `Left` or `Right`.
 *
 * A common use of `Either` is as an alternative to `Option` for dealing with possible missing values. In this usage,
 * `None` is replaced with a `Left` which can contain useful information. `Right` takes the place of `Some`. Convention
 * dictates that `Left` is used for failure and `Right` is used for success.
 *
 * For example, you could use `Either<string, number>` to detect whether a received input is a `string` or a `number`.
 *
 * ```ts
 * const parse = (errorMessage: string) => (input: string): Either<string, number> => {
 *   const n = parseInt(input, 10)
 *   return isNaN(n) ? left(errorMessage) : right(n)
 * }
 * ```
 *
 * `Either` is right-biased, which means that `Right` is assumed to be the default case to operate on. If it is `Left`,
 * operations like `map`, `chain`, ... return the `Left` value unchanged:
 *
 * ```ts
 * right(12).map(double) // right(24)
 * left(23).map(double)  // left(23)
 * ```
 */
import { tailRec } from './ChainRec';
import { phantom, toString, identity } from './function';
import { fromEquals } from './Setoid';
export const URI = 'Either';
/**
 * Left side of `Either`
 */
export class Left {
    constructor(value) {
        this.value = value;
        this._tag = 'Left';
    }
    /** The given function is applied if this is a `Right` */
    map(f) {
        return this;
    }
    ap(fab) {
        return (fab.isLeft() ? fab : this);
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /** Binds the given function across `Right` */
    chain(f) {
        return this;
    }
    bimap(f, g) {
        return new Left(f(this.value));
    }
    alt(fy) {
        return fy;
    }
    /**
     * Lazy version of `alt`
     *
     * @example
     * import { right } from 'fp-ts/lib/Either'
     *
     * assert.deepStrictEqual(right(1).orElse(() => right(2)), right(1))
     *
     * @since 1.6.0
     */
    orElse(fy) {
        return fy(this.value);
    }
    extend(f) {
        return this;
    }
    reduce(b, f) {
        return b;
    }
    /** Applies a function to each case in the data structure */
    fold(onLeft, onRight) {
        return onLeft(this.value);
    }
    /** Returns the value from this `Right` or the given argument if this is a `Left` */
    getOrElse(a) {
        return a;
    }
    /** Returns the value from this `Right` or the result of given argument if this is a `Left` */
    getOrElseL(f) {
        return f(this.value);
    }
    /** Maps the left side of the disjunction */
    mapLeft(f) {
        return new Left(f(this.value));
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `left(${toString(this.value)})`;
    }
    /** Returns `true` if the either is an instance of `Left`, `false` otherwise */
    isLeft() {
        return true;
    }
    /** Returns `true` if the either is an instance of `Right`, `false` otherwise */
    isRight() {
        return false;
    }
    /** Swaps the disjunction values */
    swap() {
        return new Right(this.value);
    }
    filterOrElse(_, zero) {
        return this;
    }
    filterOrElseL(_, zero) {
        return this;
    }
    /**
     * Use `filterOrElse` instead
     * @since 1.6.0
     * @deprecated
     */
    refineOrElse(p, zero) {
        return this;
    }
    /**
     * Lazy version of `refineOrElse`
     * Use `filterOrElseL` instead
     * @since 1.6.0
     * @deprecated
     */
    refineOrElseL(p, zero) {
        return this;
    }
}
/**
 * Right side of `Either`
 */
export class Right {
    constructor(value) {
        this.value = value;
        this._tag = 'Right';
    }
    map(f) {
        return new Right(f(this.value));
    }
    ap(fab) {
        return fab.isRight() ? this.map(fab.value) : left(fab.value);
    }
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return f(this.value);
    }
    bimap(f, g) {
        return new Right(g(this.value));
    }
    alt(fy) {
        return this;
    }
    orElse(fy) {
        return this;
    }
    extend(f) {
        return new Right(f(this));
    }
    reduce(b, f) {
        return f(b, this.value);
    }
    fold(onLeft, onRight) {
        return onRight(this.value);
    }
    getOrElse(a) {
        return this.value;
    }
    getOrElseL(f) {
        return this.value;
    }
    mapLeft(f) {
        return new Right(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `right(${toString(this.value)})`;
    }
    isLeft() {
        return false;
    }
    isRight() {
        return true;
    }
    swap() {
        return new Left(this.value);
    }
    filterOrElse(p, zero) {
        return p(this.value) ? this : left(zero);
    }
    filterOrElseL(p, zero) {
        return p(this.value) ? this : left(zero(this.value));
    }
    refineOrElse(p, zero) {
        return p(this.value) ? this : left(zero);
    }
    refineOrElseL(p, zero) {
        return p(this.value) ? this : left(zero(this.value));
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (SL, SA) => {
    return {
        show: e => e.fold(l => `left(${SL.show(l)})`, a => `right(${SA.show(a)})`)
    };
};
/**
 * @since 1.0.0
 */
export const getSetoid = (SL, SA) => {
    return fromEquals((x, y) => x.isLeft() ? y.isLeft() && SL.equals(x.value, y.value) : y.isRight() && SA.equals(x.value, y.value));
};
/**
 * Semigroup returning the left-most non-`Left` value. If both operands are `Right`s then the inner values are
 * appended using the provided `Semigroup`
 *
 * @example
 * import { getSemigroup, left, right } from 'fp-ts/lib/Either'
 * import { semigroupSum } from 'fp-ts/lib/Semigroup'
 *
 * const S = getSemigroup<string, number>(semigroupSum)
 * assert.deepStrictEqual(S.concat(left('a'), left('b')), left('a'))
 * assert.deepStrictEqual(S.concat(left('a'), right(2)), right(2))
 * assert.deepStrictEqual(S.concat(right(1), left('b')), right(1))
 * assert.deepStrictEqual(S.concat(right(1), right(2)), right(3))
 *
 *
 * @since 1.7.0
 */
export const getSemigroup = (S) => {
    return {
        concat: (x, y) => (y.isLeft() ? x : x.isLeft() ? y : right(S.concat(x.value, y.value)))
    };
};
/**
 * `Apply` semigroup
 *
 * @example
 * import { getApplySemigroup, left, right } from 'fp-ts/lib/Either'
 * import { semigroupSum } from 'fp-ts/lib/Semigroup'
 *
 * const S = getApplySemigroup<string, number>(semigroupSum)
 * assert.deepStrictEqual(S.concat(left('a'), left('b')), left('a'))
 * assert.deepStrictEqual(S.concat(left('a'), right(2)), left('a'))
 * assert.deepStrictEqual(S.concat(right(1), left('b')), left('b'))
 * assert.deepStrictEqual(S.concat(right(1), right(2)), right(3))
 *
 *
 * @since 1.7.0
 */
export const getApplySemigroup = (S) => {
    return {
        concat: (x, y) => (x.isLeft() ? x : y.isLeft() ? y : right(S.concat(x.value, y.value)))
    };
};
/**
 * @since 1.7.0
 */
export const getApplyMonoid = (M) => {
    return Object.assign({}, getApplySemigroup(M), { empty: right(M.empty) });
};
const map = (fa, f) => {
    return fa.map(f);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
const bimap = (fla, f, g) => {
    return fla.bimap(f, g);
};
const alt = (fx, fy) => {
    return fx.alt(fy);
};
const extend = (ea, f) => {
    return ea.extend(f);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.isLeft() ? M.empty : f(fa.value);
};
const foldr = (fa, b, f) => {
    return fa.isLeft() ? b : f(fa.value, b);
};
const traverse = (F) => (ta, f) => {
    return ta.isLeft() ? F.of(left(ta.value)) : F.map(f(ta.value), of);
};
const sequence = (F) => (ta) => {
    return ta.isLeft() ? F.of(left(ta.value)) : F.map(ta.value, right);
};
const chainRec = (a, f) => {
    return tailRec(e => {
        if (e.isLeft()) {
            return right(left(e.value));
        }
        else {
            const r = e.value;
            return r.isLeft() ? left(f(r.value)) : right(right(r.value));
        }
    }, f(a));
};
/**
 * Constructs a new `Either` holding a `Left` value. This usually represents a failure, due to the right-bias of this
 * structure
 *
 * @since 1.0.0
 */
export const left = (l) => {
    return new Left(l);
};
/**
 * Constructs a new `Either` holding a `Right` value. This usually represents a successful value due to the right bias
 * of this structure
 *
 * @since 1.0.0
 */
export const right = (a) => {
    return new Right(a);
};
const of = right;
export function fromPredicate(predicate, onFalse) {
    return a => (predicate(a) ? right(a) : left(onFalse(a)));
}
/**
 * Use `fromPredicate` instead
 *
 * @since 1.6.0
 * @deprecated
 */
export const fromRefinement = (refinement, onFalse) => (a) => {
    return refinement(a) ? right(a) : left(onFalse(a));
};
/**
 * Takes a default and a `Option` value, if the value is a `Some`, turn it into a `Right`, if the value is a `None` use
 * the provided default as a `Left`
 *
 * @since 1.0.0
 */
export const fromOption = (defaultValue) => (fa) => {
    return fa.isNone() ? left(defaultValue) : right(fa.value);
};
/**
 * Lazy version of `fromOption`
 *
 * @since 1.3.0
 */
export const fromOptionL = (defaultValue) => (fa) => {
    return fa.isNone() ? left(defaultValue()) : right(fa.value);
};
/**
 * Takes a default and a nullable value, if the value is not nully, turn it into a `Right`, if the value is nully use
 * the provided default as a `Left`
 *
 * @since 1.0.0
 */
export const fromNullable = (defaultValue) => (a) => {
    return a == null ? left(defaultValue) : right(a);
};
/**
 * Default value for the optional `onerror` argument of `tryCatch`
 *
 * @since 1.0.0
 */
export const toError = (e) => {
    if (e instanceof Error) {
        return e;
    }
    else {
        return new Error(String(e));
    }
};
/**
 * Use `tryCatch2v` instead
 *
 * @since 1.0.0
 * @deprecated
 */
export const tryCatch = (f, onerror = toError) => {
    return tryCatch2v(f, onerror);
};
/**
 * Constructs a new `Either` from a function that might throw
 *
 * @example
 * import { Either, left, right, tryCatch2v } from 'fp-ts/lib/Either'
 *
 * const unsafeHead = <A>(as: Array<A>): A => {
 *   if (as.length > 0) {
 *     return as[0]
 *   } else {
 *     throw new Error('empty array')
 *   }
 * }
 *
 * const head = <A>(as: Array<A>): Either<Error, A> => {
 *   return tryCatch2v(() => unsafeHead(as), e => (e instanceof Error ? e : new Error('unknown error')))
 * }
 *
 * assert.deepStrictEqual(head([]), left(new Error('empty array')))
 * assert.deepStrictEqual(head([1, 2, 3]), right(1))
 *
 * @since 1.11.0
 */
export const tryCatch2v = (f, onerror) => {
    try {
        return right(f());
    }
    catch (e) {
        return left(onerror(e));
    }
};
/**
 * @since 1.0.0
 */
export const fromValidation = (fa) => {
    return fa.isFailure() ? left(fa.value) : right(fa.value);
};
/**
 * Returns `true` if the either is an instance of `Left`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isLeft = (fa) => {
    return fa.isLeft();
};
/**
 * Returns `true` if the either is an instance of `Right`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isRight = (fa) => {
    return fa.isRight();
};
/**
 * Builds `Compactable` instance for `Either` given `Monoid` for the left side
 *
 * @since 1.7.0
 */
export function getCompactable(ML) {
    const compact = (fa) => {
        if (fa.isLeft()) {
            return fa;
        }
        if (fa.value.isNone()) {
            return left(ML.empty);
        }
        return right(fa.value.value);
    };
    const separate = (fa) => {
        if (fa.isLeft()) {
            return {
                left: fa,
                right: fa
            };
        }
        if (fa.value.isLeft()) {
            return {
                left: right(fa.value.value),
                right: left(ML.empty)
            };
        }
        return {
            left: left(ML.empty),
            right: right(fa.value.value)
        };
    };
    return {
        URI,
        _L: phantom,
        compact,
        separate
    };
}
/**
 * Builds `Filterable` instance for `Either` given `Monoid` for the left side
 *
 * @since 1.7.0
 */
export function getFilterable(ML) {
    const C = getCompactable(ML);
    const partitionMap = (fa, f) => {
        if (fa.isLeft()) {
            return {
                left: fa,
                right: fa
            };
        }
        const e = f(fa.value);
        if (e.isLeft()) {
            return {
                left: right(e.value),
                right: left(ML.empty)
            };
        }
        return {
            left: left(ML.empty),
            right: right(e.value)
        };
    };
    const partition = (fa, p) => {
        if (fa.isLeft()) {
            return {
                left: fa,
                right: fa
            };
        }
        if (p(fa.value)) {
            return {
                left: left(ML.empty),
                right: right(fa.value)
            };
        }
        return {
            left: right(fa.value),
            right: left(ML.empty)
        };
    };
    const filterMap = (fa, f) => {
        if (fa.isLeft()) {
            return fa;
        }
        const optionB = f(fa.value);
        if (optionB.isSome()) {
            return right(optionB.value);
        }
        return left(ML.empty);
    };
    const filter = (fa, p) => fa.filterOrElse(p, ML.empty);
    return Object.assign({}, C, { map,
        partitionMap,
        filterMap,
        partition,
        filter });
}
/**
 * Builds `Witherable` instance for `Either` given `Monoid` for the left side
 *
 * @since 1.7.0
 */
export function getWitherable(ML) {
    const filterableEither = getFilterable(ML);
    const wither = (F) => {
        const traverseF = traverse(F);
        return (wa, f) => F.map(traverseF(wa, f), filterableEither.compact);
    };
    const wilt = (F) => {
        const traverseF = traverse(F);
        return (wa, f) => F.map(traverseF(wa, f), filterableEither.separate);
    };
    return Object.assign({}, filterableEither, { traverse,
        reduce,
        wither,
        wilt });
}
/**
 * Converts a JavaScript Object Notation (JSON) string into an object.
 *
 * @example
 * import { parseJSON, toError } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(parseJSON('{"a":1}', toError).value, { a: 1 })
 * assert.deepStrictEqual(parseJSON('{"a":}', toError).value, new SyntaxError('Unexpected token } in JSON at position 5'))
 *
 * @since 1.16.0
 */
export const parseJSON = (s, onError) => {
    return tryCatch2v(() => JSON.parse(s), onError);
};
/**
 * Converts a JavaScript value to a JavaScript Object Notation (JSON) string.
 *
 * @example
 * import { stringifyJSON, toError } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(stringifyJSON({ a: 1 }, toError).value, '{"a":1}')
 * const circular: any = { ref: null }
 * circular.ref = circular
 * assert.deepStrictEqual(stringifyJSON(circular, toError).value, new TypeError('Converting circular structure to JSON'))
 *
 * @since 1.16.0
 */
export const stringifyJSON = (u, onError) => {
    return tryCatch2v(() => JSON.stringify(u), onError);
};
const throwError = left;
const fromEither = identity;
/**
 * @since 1.0.0
 */
export const either = {
    URI,
    map,
    of,
    ap,
    chain,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    bimap,
    alt,
    extend,
    chainRec,
    throwError,
    fromEither,
    fromOption: (o, e) => (o.isNone() ? throwError(e) : of(o.value))
};
