/*
 * Decompiled with CFR 0.152.
 */
package com.android.internal.os;

import android.os.StrictMode;
import android.os.SystemClock;
import android.util.IntArray;
import android.util.Slog;
import android.util.SparseArray;
import com.android.internal.annotations.VisibleForTesting;
import com.android.internal.os.KernelCpuProcStringReader;
import com.android.internal.os.PowerProfile;
import com.android.internal.util.Preconditions;
import java.io.BufferedReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.CharBuffer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

public abstract class KernelCpuUidTimeReader<T> {
    protected static final boolean DEBUG = false;
    private static final long DEFAULT_MIN_TIME_BETWEEN_READ = 1000L;
    final String mTag = this.getClass().getSimpleName();
    final SparseArray<T> mLastTimes = new SparseArray();
    final KernelCpuProcStringReader mReader;
    final boolean mThrottle;
    private long mMinTimeBetweenRead = 1000L;
    private long mLastReadTimeMs = 0L;

    KernelCpuUidTimeReader(KernelCpuProcStringReader reader, boolean throttle) {
        this.mReader = reader;
        this.mThrottle = throttle;
    }

    public void readDelta(Callback<T> cb) {
        if (!this.mThrottle) {
            this.readDeltaImpl(cb);
            return;
        }
        long currTimeMs = SystemClock.elapsedRealtime();
        if (currTimeMs < this.mLastReadTimeMs + this.mMinTimeBetweenRead) {
            return;
        }
        this.readDeltaImpl(cb);
        this.mLastReadTimeMs = currTimeMs;
    }

    public void readAbsolute(Callback<T> cb) {
        if (!this.mThrottle) {
            this.readAbsoluteImpl(cb);
            return;
        }
        long currTimeMs = SystemClock.elapsedRealtime();
        if (currTimeMs < this.mLastReadTimeMs + this.mMinTimeBetweenRead) {
            return;
        }
        this.readAbsoluteImpl(cb);
        this.mLastReadTimeMs = currTimeMs;
    }

    abstract void readDeltaImpl(Callback<T> var1);

    abstract void readAbsoluteImpl(Callback<T> var1);

    public void removeUid(int uid) {
        this.mLastTimes.delete(uid);
    }

    public void removeUidsInRange(int startUid, int endUid) {
        if (endUid < startUid) {
            Slog.e(this.mTag, "start UID " + startUid + " > end UID " + endUid);
            return;
        }
        this.mLastTimes.put(startUid, null);
        this.mLastTimes.put(endUid, null);
        int firstIndex = this.mLastTimes.indexOfKey(startUid);
        int lastIndex = this.mLastTimes.indexOfKey(endUid);
        this.mLastTimes.removeAtRange(firstIndex, lastIndex - firstIndex + 1);
    }

    public void setThrottle(long minTimeBetweenRead) {
        if (this.mThrottle && minTimeBetweenRead >= 0L) {
            this.mMinTimeBetweenRead = minTimeBetweenRead;
        }
    }

    public static class KernelCpuUidClusterTimeReader
    extends KernelCpuUidTimeReader<long[]> {
        private int mNumClusters;
        private int mNumCores;
        private int[] mCoresOnClusters;
        private long[] mBuffer;
        private long[] mCurTime;
        private long[] mDeltaTime;

        public KernelCpuUidClusterTimeReader(boolean throttle) {
            super(KernelCpuProcStringReader.getClusterTimeReaderInstance(), throttle);
        }

        @VisibleForTesting
        public KernelCpuUidClusterTimeReader(KernelCpuProcStringReader reader, boolean throttle) {
            super(reader, throttle);
        }

        @Override
        void readDeltaImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    int uid = (int)this.mBuffer[0];
                    long[] lastTimes = (long[])this.mLastTimes.get(uid);
                    if (lastTimes == null) {
                        lastTimes = new long[this.mNumClusters];
                        this.mLastTimes.put(uid, lastTimes);
                    }
                    this.sumClusterTime();
                    boolean valid = true;
                    boolean notify = false;
                    for (int i = 0; i < this.mNumClusters; ++i) {
                        this.mDeltaTime[i] = this.mCurTime[i] - lastTimes[i];
                        if (this.mDeltaTime[i] < 0L) {
                            Slog.e(this.mTag, "Negative delta from cluster time proc: " + this.mDeltaTime[i]);
                            valid = false;
                        }
                        notify |= this.mDeltaTime[i] > 0L;
                    }
                    if (!notify || !valid) continue;
                    System.arraycopy(this.mCurTime, 0, lastTimes, 0, this.mNumClusters);
                    if (cb == null) continue;
                    cb.onUidCpuTime(uid, this.mDeltaTime);
                }
            }
        }

        @Override
        void readAbsoluteImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    this.sumClusterTime();
                    cb.onUidCpuTime((int)this.mBuffer[0], this.mCurTime);
                }
            }
        }

        private void sumClusterTime() {
            int core = 1;
            for (int i = 0; i < this.mNumClusters; ++i) {
                double sum = 0.0;
                for (int j = 1; j <= this.mCoresOnClusters[i]; ++j) {
                    sum += (double)this.mBuffer[core++] * 10.0 / (double)j;
                }
                this.mCurTime[i] = (long)sum;
            }
        }

        private boolean checkPrecondition(KernelCpuProcStringReader.ProcFileIterator iter) {
            if (iter == null || !iter.hasNextLine()) {
                return false;
            }
            CharBuffer line = iter.nextLine();
            if (this.mNumClusters > 0) {
                return true;
            }
            String[] lineArray = line.toString().split(" ");
            if (lineArray.length % 2 != 0) {
                Slog.wtf(this.mTag, "Malformed uid_concurrent_policy_time line: " + line);
                return false;
            }
            int[] clusters = new int[lineArray.length / 2];
            int cores = 0;
            for (int i = 0; i < clusters.length; ++i) {
                if (!lineArray[i * 2].startsWith("policy")) {
                    Slog.wtf(this.mTag, "Malformed uid_concurrent_policy_time line: " + line);
                    return false;
                }
                clusters[i] = Integer.parseInt(lineArray[i * 2 + 1], 10);
                cores += clusters[i];
            }
            this.mNumClusters = clusters.length;
            this.mNumCores = cores;
            this.mCoresOnClusters = clusters;
            this.mBuffer = new long[cores + 1];
            this.mCurTime = new long[this.mNumClusters];
            this.mDeltaTime = new long[this.mNumClusters];
            return true;
        }
    }

    public static class KernelCpuUidActiveTimeReader
    extends KernelCpuUidTimeReader<Long> {
        private int mCores = 0;
        private long[] mBuffer;

        public KernelCpuUidActiveTimeReader(boolean throttle) {
            super(KernelCpuProcStringReader.getActiveTimeReaderInstance(), throttle);
        }

        @VisibleForTesting
        public KernelCpuUidActiveTimeReader(KernelCpuProcStringReader reader, boolean throttle) {
            super(reader, throttle);
        }

        @Override
        void readDeltaImpl(Callback<Long> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    int uid = (int)this.mBuffer[0];
                    long cpuActiveTime = KernelCpuUidActiveTimeReader.sumActiveTime(this.mBuffer);
                    if (cpuActiveTime <= 0L) continue;
                    long delta = cpuActiveTime - this.mLastTimes.get(uid, 0L);
                    if (delta > 0L) {
                        this.mLastTimes.put(uid, cpuActiveTime);
                        if (cb == null) continue;
                        cb.onUidCpuTime(uid, delta);
                        continue;
                    }
                    if (delta >= 0L) continue;
                    Slog.e(this.mTag, "Negative delta from active time proc: " + delta);
                }
            }
        }

        @Override
        void readAbsoluteImpl(Callback<Long> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    long cpuActiveTime = KernelCpuUidActiveTimeReader.sumActiveTime(this.mBuffer);
                    if (cpuActiveTime <= 0L) continue;
                    cb.onUidCpuTime((int)this.mBuffer[0], cpuActiveTime);
                }
            }
        }

        private static long sumActiveTime(long[] times) {
            double sum = 0.0;
            for (int i = 1; i < times.length; ++i) {
                sum += (double)times[i] * 10.0 / (double)i;
            }
            return (long)sum;
        }

        private boolean checkPrecondition(KernelCpuProcStringReader.ProcFileIterator iter) {
            if (iter == null || !iter.hasNextLine()) {
                return false;
            }
            CharBuffer line = iter.nextLine();
            if (this.mCores > 0) {
                return true;
            }
            String str = line.toString();
            if (!str.startsWith("cpus:")) {
                Slog.wtf(this.mTag, "Malformed uid_concurrent_active_time line: " + line);
                return false;
            }
            int cores = Integer.parseInt(str.substring(5).trim(), 10);
            if (cores <= 0) {
                Slog.wtf(this.mTag, "Malformed uid_concurrent_active_time line: " + line);
                return false;
            }
            this.mCores = cores;
            this.mBuffer = new long[this.mCores + 1];
            return true;
        }
    }

    public static class KernelCpuUidFreqTimeReader
    extends KernelCpuUidTimeReader<long[]> {
        private static final String UID_TIMES_PROC_FILE = "/proc/uid_time_in_state";
        private static final int MAX_ERROR_COUNT = 5;
        private final Path mProcFilePath;
        private long[] mBuffer;
        private long[] mCurTimes;
        private long[] mDeltaTimes;
        private long[] mCpuFreqs;
        private int mFreqCount = 0;
        private int mErrors = 0;
        private boolean mPerClusterTimesAvailable;
        private boolean mAllUidTimesAvailable = true;

        public KernelCpuUidFreqTimeReader(boolean throttle) {
            this(UID_TIMES_PROC_FILE, KernelCpuProcStringReader.getFreqTimeReaderInstance(), throttle);
        }

        @VisibleForTesting
        public KernelCpuUidFreqTimeReader(String procFile, KernelCpuProcStringReader reader, boolean throttle) {
            super(reader, throttle);
            this.mProcFilePath = Paths.get(procFile, new String[0]);
        }

        public boolean perClusterTimesAvailable() {
            return this.mPerClusterTimesAvailable;
        }

        public boolean allUidTimesAvailable() {
            return this.mAllUidTimesAvailable;
        }

        public SparseArray<long[]> getAllUidCpuFreqTimeMs() {
            return this.mLastTimes;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public long[] readFreqs(PowerProfile powerProfile) {
            Throwable throwable;
            Preconditions.checkNotNull(powerProfile);
            if (this.mCpuFreqs != null) {
                return this.mCpuFreqs;
            }
            if (!this.mAllUidTimesAvailable) {
                return null;
            }
            int oldMask = StrictMode.allowThreadDiskReadsMask();
            try {
                throwable = null;
                try (BufferedReader reader = Files.newBufferedReader(this.mProcFilePath);){
                    if (this.readFreqs(reader.readLine()) == null) {
                        long[] lArray = null;
                        return lArray;
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
            }
            catch (IOException e) {
                if (++this.mErrors >= 5) {
                    this.mAllUidTimesAvailable = false;
                }
                Slog.e(this.mTag, "Failed to read /proc/uid_time_in_state: " + e);
                throwable = null;
                return throwable;
            }
            finally {
                StrictMode.setThreadPolicyMask(oldMask);
            }
            IntArray numClusterFreqs = this.extractClusterInfoFromProcFileFreqs();
            int numClusters = powerProfile.getNumCpuClusters();
            if (numClusterFreqs.size() == numClusters) {
                this.mPerClusterTimesAvailable = true;
                for (int i = 0; i < numClusters; ++i) {
                    if (numClusterFreqs.get(i) == powerProfile.getNumSpeedStepsInCpuCluster(i)) continue;
                    this.mPerClusterTimesAvailable = false;
                    break;
                }
            } else {
                this.mPerClusterTimesAvailable = false;
            }
            Slog.i(this.mTag, "mPerClusterTimesAvailable=" + this.mPerClusterTimesAvailable);
            return this.mCpuFreqs;
        }

        private long[] readFreqs(String line) {
            if (line == null) {
                return null;
            }
            String[] lineArray = line.split(" ");
            if (lineArray.length <= 1) {
                Slog.wtf(this.mTag, "Malformed freq line: " + line);
                return null;
            }
            this.mFreqCount = lineArray.length - 1;
            this.mCpuFreqs = new long[this.mFreqCount];
            this.mCurTimes = new long[this.mFreqCount];
            this.mDeltaTimes = new long[this.mFreqCount];
            this.mBuffer = new long[this.mFreqCount + 1];
            for (int i = 0; i < this.mFreqCount; ++i) {
                this.mCpuFreqs[i] = Long.parseLong(lineArray[i + 1], 10);
            }
            return this.mCpuFreqs;
        }

        @Override
        void readDeltaImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    int uid = (int)this.mBuffer[0];
                    long[] lastTimes = (long[])this.mLastTimes.get(uid);
                    if (lastTimes == null) {
                        lastTimes = new long[this.mFreqCount];
                        this.mLastTimes.put(uid, lastTimes);
                    }
                    this.copyToCurTimes();
                    boolean notify = false;
                    boolean valid = true;
                    for (int i = 0; i < this.mFreqCount; ++i) {
                        this.mDeltaTimes[i] = this.mCurTimes[i] - lastTimes[i];
                        if (this.mDeltaTimes[i] < 0L) {
                            Slog.e(this.mTag, "Negative delta from freq time proc: " + this.mDeltaTimes[i]);
                            valid = false;
                        }
                        notify |= this.mDeltaTimes[i] > 0L;
                    }
                    if (!notify || !valid) continue;
                    System.arraycopy(this.mCurTimes, 0, lastTimes, 0, this.mFreqCount);
                    if (cb == null) continue;
                    cb.onUidCpuTime(uid, this.mDeltaTimes);
                }
            }
        }

        @Override
        void readAbsoluteImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (!this.checkPrecondition(iter)) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) != this.mBuffer.length) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    this.copyToCurTimes();
                    cb.onUidCpuTime((int)this.mBuffer[0], this.mCurTimes);
                }
            }
        }

        private void copyToCurTimes() {
            for (int i = 0; i < this.mFreqCount; ++i) {
                this.mCurTimes[i] = this.mBuffer[i + 1] * 10L;
            }
        }

        private boolean checkPrecondition(KernelCpuProcStringReader.ProcFileIterator iter) {
            if (iter == null || !iter.hasNextLine()) {
                return false;
            }
            CharBuffer line = iter.nextLine();
            if (this.mCpuFreqs != null) {
                return true;
            }
            return this.readFreqs(line.toString()) != null;
        }

        private IntArray extractClusterInfoFromProcFileFreqs() {
            IntArray numClusterFreqs = new IntArray();
            int freqsFound = 0;
            for (int i = 0; i < this.mFreqCount; ++i) {
                ++freqsFound;
                if (i + 1 != this.mFreqCount && this.mCpuFreqs[i + 1] > this.mCpuFreqs[i]) continue;
                numClusterFreqs.add(freqsFound);
                freqsFound = 0;
            }
            return numClusterFreqs;
        }
    }

    public static class KernelCpuUidUserSysTimeReader
    extends KernelCpuUidTimeReader<long[]> {
        private static final String REMOVE_UID_PROC_FILE = "/proc/uid_cputime/remove_uid_range";
        private final long[] mBuffer = new long[4];
        private final long[] mUsrSysTime = new long[2];

        public KernelCpuUidUserSysTimeReader(boolean throttle) {
            super(KernelCpuProcStringReader.getUserSysTimeReaderInstance(), throttle);
        }

        @VisibleForTesting
        public KernelCpuUidUserSysTimeReader(KernelCpuProcStringReader reader, boolean throttle) {
            super(reader, throttle);
        }

        @Override
        void readDeltaImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (iter == null) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) < 3) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    int uid = (int)this.mBuffer[0];
                    long[] lastTimes = (long[])this.mLastTimes.get(uid);
                    if (lastTimes == null) {
                        lastTimes = new long[2];
                        this.mLastTimes.put(uid, lastTimes);
                    }
                    long currUsrTimeUs = this.mBuffer[1];
                    long currSysTimeUs = this.mBuffer[2];
                    this.mUsrSysTime[0] = currUsrTimeUs - lastTimes[0];
                    this.mUsrSysTime[1] = currSysTimeUs - lastTimes[1];
                    if (this.mUsrSysTime[0] < 0L || this.mUsrSysTime[1] < 0L) {
                        Slog.e(this.mTag, "Negative user/sys time delta for UID=" + uid + "\nPrev times: u=" + lastTimes[0] + " s=" + lastTimes[1] + " Curr times: u=" + currUsrTimeUs + " s=" + currSysTimeUs);
                    } else if ((this.mUsrSysTime[0] > 0L || this.mUsrSysTime[1] > 0L) && cb != null) {
                        cb.onUidCpuTime(uid, this.mUsrSysTime);
                    }
                    lastTimes[0] = currUsrTimeUs;
                    lastTimes[1] = currSysTimeUs;
                }
            }
        }

        @Override
        void readAbsoluteImpl(Callback<long[]> cb) {
            try (KernelCpuProcStringReader.ProcFileIterator iter = this.mReader.open(!this.mThrottle);){
                CharBuffer buf;
                if (iter == null) {
                    return;
                }
                while ((buf = iter.nextLine()) != null) {
                    if (KernelCpuProcStringReader.asLongs(buf, this.mBuffer) < 3) {
                        Slog.wtf(this.mTag, "Invalid line: " + buf.toString());
                        continue;
                    }
                    this.mUsrSysTime[0] = this.mBuffer[1];
                    this.mUsrSysTime[1] = this.mBuffer[2];
                    cb.onUidCpuTime((int)this.mBuffer[0], this.mUsrSysTime);
                }
            }
        }

        @Override
        public void removeUid(int uid) {
            super.removeUid(uid);
            this.removeUidsFromKernelModule(uid, uid);
        }

        @Override
        public void removeUidsInRange(int startUid, int endUid) {
            super.removeUidsInRange(startUid, endUid);
            this.removeUidsFromKernelModule(startUid, endUid);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void removeUidsFromKernelModule(int startUid, int endUid) {
            Slog.d(this.mTag, "Removing uids " + startUid + "-" + endUid);
            int oldMask = StrictMode.allowThreadDiskWritesMask();
            try (FileWriter writer = new FileWriter(REMOVE_UID_PROC_FILE);){
                writer.write(startUid + "-" + endUid);
                writer.flush();
            }
            catch (IOException e) {
                Slog.e(this.mTag, "failed to remove uids " + startUid + " - " + endUid + " from uid_cputime module", e);
            }
            finally {
                StrictMode.setThreadPolicyMask(oldMask);
            }
        }
    }

    public static interface Callback<T> {
        public void onUidCpuTime(int var1, T var2);
    }
}

