/*
 * Decompiled with CFR 0.152.
 */
package org.gradle.internal.remote.internal.inet;

import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectStreamException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedSelectorException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import org.gradle.internal.UncheckedException;
import org.gradle.internal.concurrent.CompositeStoppable;
import org.gradle.internal.impldep.com.google.common.base.Objects;
import org.gradle.internal.io.BufferCaster;
import org.gradle.internal.remote.internal.MessageIOException;
import org.gradle.internal.remote.internal.MessageSerializer;
import org.gradle.internal.remote.internal.RecoverableMessageIOException;
import org.gradle.internal.remote.internal.RemoteConnection;
import org.gradle.internal.remote.internal.inet.SocketInetAddress;
import org.gradle.internal.serialize.FlushableEncoder;
import org.gradle.internal.serialize.ObjectReader;
import org.gradle.internal.serialize.ObjectWriter;
import org.gradle.internal.serialize.StatefulSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SocketConnection<T>
implements RemoteConnection<T> {
    private static final Logger LOGGER = LoggerFactory.getLogger(SocketConnection.class);
    private final SocketChannel socket;
    private final SocketInetAddress localAddress;
    private final SocketInetAddress remoteAddress;
    private final ObjectWriter<T> objectWriter;
    private final ObjectReader<T> objectReader;
    private final InputStream instr;
    private final OutputStream outstr;
    private final FlushableEncoder encoder;

    public SocketConnection(SocketChannel socket, MessageSerializer streamSerializer, StatefulSerializer<T> messageSerializer) {
        this.socket = socket;
        try {
            socket.configureBlocking(false);
            this.outstr = new SocketOutputStream(socket);
            this.instr = new SocketInputStream(socket);
        }
        catch (IOException e) {
            throw UncheckedException.throwAsUncheckedException(e);
        }
        InetSocketAddress localSocketAddress = (InetSocketAddress)socket.socket().getLocalSocketAddress();
        this.localAddress = new SocketInetAddress(localSocketAddress.getAddress(), localSocketAddress.getPort());
        InetSocketAddress remoteSocketAddress = (InetSocketAddress)socket.socket().getRemoteSocketAddress();
        this.remoteAddress = new SocketInetAddress(remoteSocketAddress.getAddress(), remoteSocketAddress.getPort());
        this.objectReader = messageSerializer.newReader(streamSerializer.newDecoder(this.instr));
        this.encoder = streamSerializer.newEncoder(this.outstr);
        this.objectWriter = messageSerializer.newWriter(this.encoder);
    }

    public String toString() {
        return "socket connection from " + this.localAddress + " to " + this.remoteAddress;
    }

    @Override
    public T receive() throws MessageIOException {
        try {
            return this.objectReader.read();
        }
        catch (EOFException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Discarding EOFException: {}", (Object)e.toString());
            }
            return null;
        }
        catch (ObjectStreamException e) {
            throw new RecoverableMessageIOException(String.format("Could not read message from '%s'.", this.remoteAddress), e);
        }
        catch (ClassNotFoundException e) {
            throw new RecoverableMessageIOException(String.format("Could not read message from '%s'.", this.remoteAddress), e);
        }
        catch (IOException e) {
            throw new RecoverableMessageIOException(String.format("Could not read message from '%s'.", this.remoteAddress), e);
        }
        catch (Throwable e) {
            throw new MessageIOException(String.format("Could not read message from '%s'.", this.remoteAddress), e);
        }
    }

    private static boolean isEndOfStream(Exception e) {
        if (e instanceof EOFException) {
            return true;
        }
        if (e instanceof IOException) {
            if (Objects.equal((Object)e.getMessage(), (Object)"An existing connection was forcibly closed by the remote host")) {
                return true;
            }
            if (Objects.equal((Object)e.getMessage(), (Object)"An established connection was aborted by the software in your host machine")) {
                return true;
            }
            if (Objects.equal((Object)e.getMessage(), (Object)"Connection reset by peer")) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void dispatch(T message) throws MessageIOException {
        try {
            this.objectWriter.write(message);
        }
        catch (ObjectStreamException e) {
            throw new RecoverableMessageIOException(String.format("Could not write message %s to '%s'.", message, this.remoteAddress), e);
        }
        catch (ClassNotFoundException e) {
            throw new RecoverableMessageIOException(String.format("Could not write message %s to '%s'.", message, this.remoteAddress), e);
        }
        catch (IOException e) {
            throw new RecoverableMessageIOException(String.format("Could not write message %s to '%s'.", message, this.remoteAddress), e);
        }
        catch (Throwable e) {
            throw new MessageIOException(String.format("Could not write message %s to '%s'.", message, this.remoteAddress), e);
        }
    }

    @Override
    public void flush() throws MessageIOException {
        try {
            this.encoder.flush();
            this.outstr.flush();
        }
        catch (Throwable e) {
            throw new MessageIOException(String.format("Could not write '%s'.", this.remoteAddress), e);
        }
    }

    @Override
    public void stop() {
        CompositeStoppable.stoppable(new Closeable(){

            @Override
            public void close() throws IOException {
                SocketConnection.this.flush();
            }
        }, this.instr, this.outstr, this.socket).stop();
    }

    private static class SocketOutputStream
    extends OutputStream {
        private static final int RETRIES_WHEN_BUFFER_FULL = 2;
        private Selector selector;
        private final SocketChannel socket;
        private final ByteBuffer buffer;
        private final byte[] writeBuffer = new byte[1];

        public SocketOutputStream(SocketChannel socket) throws IOException {
            this.socket = socket;
            this.buffer = ByteBuffer.allocateDirect(32768);
        }

        @Override
        public void write(int b) throws IOException {
            this.writeBuffer[0] = (byte)b;
            this.write(this.writeBuffer);
        }

        @Override
        public void write(byte[] src, int offset, int max) throws IOException {
            int remaining = max;
            int currentPos = offset;
            while (remaining > 0) {
                int count = Math.min(remaining, this.buffer.remaining());
                if (count > 0) {
                    this.buffer.put(src, currentPos, count);
                    remaining -= count;
                    currentPos += count;
                }
                while (this.buffer.remaining() == 0) {
                    this.writeBufferToChannel();
                }
            }
        }

        @Override
        public void flush() throws IOException {
            while (this.buffer.position() > 0) {
                this.writeBufferToChannel();
            }
        }

        private void writeBufferToChannel() throws IOException {
            BufferCaster.cast(this.buffer).flip();
            int count = this.writeWithNonBlockingRetry();
            if (count == 0) {
                this.waitForWriteBufferToDrain();
            }
            this.buffer.compact();
        }

        private int writeWithNonBlockingRetry() throws IOException {
            int count = 0;
            int retryCount = 0;
            while (count == 0 && retryCount++ < 2) {
                count = this.socket.write(this.buffer);
                if (count < 0) {
                    throw new EOFException();
                }
                if (count != 0) continue;
                Thread.yield();
            }
            return count;
        }

        private void waitForWriteBufferToDrain() throws IOException {
            if (this.selector == null) {
                this.selector = Selector.open();
            }
            SelectionKey key = this.socket.register(this.selector, 4);
            this.selector.select();
            key.cancel();
            this.selector.selectNow();
        }

        @Override
        public void close() throws IOException {
            if (this.selector != null) {
                this.selector.close();
                this.selector = null;
            }
        }
    }

    private static class SocketInputStream
    extends InputStream {
        private final Selector selector;
        private final ByteBuffer buffer;
        private final SocketChannel socket;
        private final byte[] readBuffer = new byte[1];

        public SocketInputStream(SocketChannel socket) throws IOException {
            this.socket = socket;
            this.selector = Selector.open();
            socket.register(this.selector, 1);
            this.buffer = ByteBuffer.allocateDirect(4096);
            BufferCaster.cast(this.buffer).limit(0);
        }

        @Override
        public int read() throws IOException {
            int nread = this.read(this.readBuffer, 0, 1);
            if (nread <= 0) {
                return nread;
            }
            return this.readBuffer[0];
        }

        @Override
        public int read(byte[] dest, int offset, int max) throws IOException {
            if (max == 0) {
                return 0;
            }
            if (this.buffer.remaining() == 0) {
                int nread;
                try {
                    this.selector.select();
                }
                catch (ClosedSelectorException e) {
                    return -1;
                }
                if (!this.selector.isOpen()) {
                    return -1;
                }
                BufferCaster.cast(this.buffer).clear();
                try {
                    nread = this.socket.read(this.buffer);
                }
                catch (IOException e) {
                    if (SocketConnection.isEndOfStream(e)) {
                        BufferCaster.cast(this.buffer).position(0);
                        BufferCaster.cast(this.buffer).limit(0);
                        return -1;
                    }
                    throw e;
                }
                BufferCaster.cast(this.buffer).flip();
                if (nread < 0) {
                    return -1;
                }
            }
            int count = Math.min(this.buffer.remaining(), max);
            this.buffer.get(dest, offset, count);
            return count;
        }

        @Override
        public void close() throws IOException {
            this.selector.close();
        }
    }
}

