"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConflictParser = void 0;

var _noopVisitor = require("./noop-visitor");

var _position = require("../position");

class Result {
  constructor(remainingSteps) {
    this.steps = remainingSteps;
  }

  wasSuccessful() {
    return this.steps.length === 0;
  }

}

class ConflictParser {
  constructor(adapter, visitor, isRebase) {
    this.adapter = adapter;
    this.visitor = visitor;
    this.isRebase = isRebase;
    this.lastBoundary = null;
    this.steps = [];

    if (this.isRebase) {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitTheirSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitOurSide'));
    } else {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitOurSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitTheirSide'));
    }
  }

  continueFrom(result) {
    this.steps = result.steps;
    return this.parse();
  }

  parse() {
    for (let i = 0; i < this.steps.length; i++) {
      if (!this.steps[i](this)) {
        return new Result(this.steps.slice(i));
      }
    }

    return new Result([]);
  } // Visit a side that begins with a banner and description as its first line.


  visitHeaderSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();

    if (this.advanceToBoundary('|=') === null) {
      return false;
    }

    const sideRowEnd = this.adapter.getCurrentRow();
    this.visitor[visitMethod](position, sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  } // Visit the base side from diff3 output, if one is present, then visit the separator.


  visitBaseAndSeparator() {
    if (this.lastBoundary === '|') {
      if (!this.visitBaseSide()) {
        return false;
      }
    }

    return this.visitSeparator();
  } // Visit a base side from diff3 output.


  visitBaseSide() {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();
    let b = this.advanceToBoundary('<=');

    if (b === null) {
      return false;
    }

    while (b === '<') {
      // Embedded recursive conflict within a base side, caused by a criss-cross merge.
      // Advance the input adapter beyond it without marking anything.
      const subParser = new ConflictParser(this.adapter, new _noopVisitor.NoopVisitor(), this.isRebase);

      if (!subParser.parse().wasSuccessful()) {
        return false;
      }

      b = this.advanceToBoundary('<=');

      if (b === null) {
        return false;
      }
    }

    const sideRowEnd = this.adapter.getCurrentRow();
    this.visitor.visitBaseSide(sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  } // Visit a "========" separator.


  visitSeparator() {
    const sepRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();
    const sepRowEnd = this.adapter.getCurrentRow();
    this.visitor.visitSeparator(sepRowStart, sepRowEnd);
    return true;
  } // Visit a side with a banner and description as its last line.


  visitFooterSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();

    if (this.advanceToBoundary('>') === null) {
      return false;
    }

    this.adapter.advanceRow();
    const sideRowEnd = this.adapter.getCurrentRow();
    this.visitor[visitMethod](position, sideRowEnd - 1, sideRowStart, sideRowEnd - 1);
    return true;
  } // Determine if the current row is a side boundary.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundary detected.
  //
  // Returns the matching boundaryKinds character, or `null` if none match.


  isAtBoundary(boundaryKinds = '<|=>') {
    const line = this.adapter.getCurrentLine();

    for (let i = 0; i < boundaryKinds.length; i++) {
      const b = boundaryKinds[i];

      if (line.startsWith(b.repeat(7))) {
        return b;
      }
    }

    return null;
  } // Increment the current row until the current line matches one of the provided boundary kinds, or until there are no
  // more lines in the editor.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundaries that halt the
  //   progression.
  //
  // Returns the matching boundaryKinds character, or 'null' if there are no matches to the end of the editor.


  advanceToBoundary(boundaryKinds = '<|=>') {
    let b = this.isAtBoundary(boundaryKinds);

    while (b === null) {
      this.adapter.advanceRow();

      if (this.adapter.isAtEnd()) {
        return null;
      }

      b = this.isAtBoundary(boundaryKinds);
    }

    this.lastBoundary = b;
    return b;
  }

}

exports.ConflictParser = ConflictParser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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