"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adoptToHapiAuthFormat = adoptToHapiAuthFormat;

var _boom = _interopRequireDefault(require("boom"));

var _lodash = require("lodash");

var _router = require("../router");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var ResultType;

(function (ResultType) {
  ResultType["authenticated"] = "authenticated";
  ResultType["redirected"] = "redirected";
  ResultType["rejected"] = "rejected";
})(ResultType || (ResultType = {}));

const authResult = {
  authenticated(data = {}) {
    return {
      type: ResultType.authenticated,
      state: data.state || {},
      headers: data.headers || {}
    };
  },

  redirected(url) {
    return {
      type: ResultType.redirected,
      url
    };
  },

  rejected(error, options = {}) {
    return {
      type: ResultType.rejected,
      error,
      statusCode: options.statusCode
    };
  },

  isValid(candidate) {
    return candidate && (candidate.type === ResultType.authenticated || candidate.type === ResultType.rejected || candidate.type === ResultType.redirected);
  },

  isAuthenticated(result) {
    return result.type === ResultType.authenticated;
  },

  isRedirected(result) {
    return result.type === ResultType.redirected;
  },

  isRejected(result) {
    return result.type === ResultType.rejected;
  }

};
/**
 * Auth Headers map
 * @public
 * */

const toolkit = {
  authenticated: authResult.authenticated,
  redirected: authResult.redirected,
  rejected: authResult.rejected
};
/** @public */

/** @public */
function adoptToHapiAuthFormat(fn, onSuccess = _lodash.noop) {
  return async function interceptAuth(req, h) {
    try {
      const result = await fn(_router.KibanaRequest.from(req, undefined, false), toolkit);

      if (!authResult.isValid(result)) {
        throw new Error(`Unexpected result from Authenticate. Expected AuthResult, but given: ${result}.`);
      }

      if (authResult.isAuthenticated(result)) {
        onSuccess(req, {
          state: result.state,
          headers: result.headers
        });
        return h.authenticated({
          credentials: result.state || {}
        });
      }

      if (authResult.isRedirected(result)) {
        return h.redirect(result.url).takeover();
      }

      const {
        error,
        statusCode
      } = result;
      return _boom.default.boomify(error, {
        statusCode
      });
    } catch (error) {
      return _boom.default.internal(error.message, {
        statusCode: 500
      });
    }
  };
}