/**
 * @file Adapted from https://github.com/purescript/purescript-exceptions
 */
import { left, right } from './Either';
import { IO, io } from './IO';
import { none, some } from './Option';
/**
 * Create a JavaScript error, specifying a message
 *
 * @since 1.0.0
 */
export const error = (message) => {
    return new Error(message);
};
/**
 * Get the error message from a JavaScript error
 *
 * @since 1.0.0
 */
export const message = (e) => {
    return e.message;
};
/**
 * Get the stack trace from a JavaScript error
 *
 * @since 1.0.0
 */
export const stack = (e) => {
    return typeof e.stack === 'string' ? some(e.stack) : none;
};
/**
 * Throw an exception
 *
 * @since 1.0.0
 */
export const throwError = (e) => {
    return new IO(() => {
        throw e;
    });
};
/**
 * Catch an exception by providing an exception handler
 *
 * @since 1.0.0
 */
export const catchError = (ma, handler) => {
    return new IO(() => {
        try {
            return ma.run();
        }
        catch (e) {
            if (e instanceof Error) {
                return handler(e).run();
            }
            else {
                return handler(new Error(e.toString())).run();
            }
        }
    });
};
/**
 * Runs an IO and returns eventual Exceptions as a `Left` value. If the computation succeeds the result gets wrapped in
 * a `Right`.
 *
 * @since 1.0.0
 */
export const tryCatch = (ma) => {
    return catchError(ma.map(right), e => io.of(left(e)));
};
