"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Hunk {
  constructor({
    oldStartRow,
    newStartRow,
    oldRowCount,
    newRowCount,
    sectionHeading,
    marker,
    regions
  }) {
    this.oldStartRow = oldStartRow;
    this.newStartRow = newStartRow;
    this.oldRowCount = oldRowCount;
    this.newRowCount = newRowCount;
    this.sectionHeading = sectionHeading;
    this.marker = marker;
    this.regions = regions;
  }

  getOldStartRow() {
    return this.oldStartRow;
  }

  getNewStartRow() {
    return this.newStartRow;
  }

  getOldRowCount() {
    return this.oldRowCount;
  }

  getNewRowCount() {
    return this.newRowCount;
  }

  getHeader() {
    return `@@ -${this.oldStartRow},${this.oldRowCount} +${this.newStartRow},${this.newRowCount} @@`;
  }

  getSectionHeading() {
    return this.sectionHeading;
  }

  getRegions() {
    return this.regions;
  }

  getChanges() {
    return this.regions.filter(change => change.isChange());
  }

  getMarker() {
    return this.marker;
  }

  getRange() {
    return this.getMarker().getRange();
  }

  getBufferRows() {
    return this.getRange().getRows();
  }

  bufferRowCount() {
    return this.getRange().getRowCount();
  }

  includesBufferRow(row) {
    return this.getRange().intersectsRow(row);
  }

  getOldRowAt(row) {
    let current = this.oldStartRow;

    for (const region of this.getRegions()) {
      if (region.includesBufferRow(row)) {
        const offset = row - region.getStartBufferRow();
        return region.when({
          unchanged: () => current + offset,
          addition: () => null,
          deletion: () => current + offset,
          nonewline: () => null
        });
      } else {
        current += region.when({
          unchanged: () => region.bufferRowCount(),
          addition: () => 0,
          deletion: () => region.bufferRowCount(),
          nonewline: () => 0
        });
      }
    }

    return null;
  }

  getNewRowAt(row) {
    let current = this.newStartRow;

    for (const region of this.getRegions()) {
      if (region.includesBufferRow(row)) {
        const offset = row - region.getStartBufferRow();
        return region.when({
          unchanged: () => current + offset,
          addition: () => current + offset,
          deletion: () => null,
          nonewline: () => null
        });
      } else {
        current += region.when({
          unchanged: () => region.bufferRowCount(),
          addition: () => region.bufferRowCount(),
          deletion: () => 0,
          nonewline: () => 0
        });
      }
    }

    return null;
  }

  getMaxLineNumberWidth() {
    return Math.max((this.oldStartRow + this.oldRowCount).toString().length, (this.newStartRow + this.newRowCount).toString().length);
  }

  changedLineCount() {
    return this.regions.filter(region => region.isChange()).reduce((count, change) => count + change.bufferRowCount(), 0);
  }

  updateMarkers(map) {
    this.marker = map.get(this.marker) || this.marker;

    for (const region of this.regions) {
      region.updateMarkers(map);
    }
  }

  destroyMarkers() {
    this.marker.destroy();

    for (const region of this.regions) {
      region.destroyMarkers();
    }
  }

  toStringIn(buffer) {
    return this.getRegions().reduce((str, region) => str + region.toStringIn(buffer), this.getHeader() + '\n');
  }
  /*
   * Construct a String containing internal diagnostic information.
   */

  /* istanbul ignore next */


  inspect(opts = {}) {
    const options = _objectSpread({
      indent: 0
    }, opts);

    let indentation = '';

    for (let i = 0; i < options.indent; i++) {
      indentation += ' ';
    }

    let inspectString = `${indentation}(Hunk marker=${this.marker.id}\n`;

    if (this.marker.isDestroyed()) {
      inspectString += ' [destroyed]';
    }

    if (!this.marker.isValid()) {
      inspectString += ' [invalid]';
    }

    for (const region of this.regions) {
      inspectString += region.inspect({
        indent: options.indent + 2
      });
    }

    inspectString += `${indentation})\n`;
    return inspectString;
  }

}

exports["default"] = Hunk;

_defineProperty(Hunk, "layerName", 'hunk');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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