"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

class Task {
  constructor(fn, parallel = true) {
    this.fn = fn;
    this.parallel = parallel;
    this.promise = new Promise((resolve, reject) => {
      this.resolve = resolve;
      this.reject = reject;
    });
  }

  async execute() {
    try {
      const value = await this.fn.call(undefined);
      this.resolve(value);
    } catch (err) {
      this.reject(err);
    }
  }

  runsInParallel() {
    return this.parallel;
  }

  runsInSerial() {
    return !this.parallel;
  }

  getPromise() {
    return this.promise;
  }

}

class AsyncQueue {
  constructor(options = {}) {
    this.parallelism = options.parallelism || 1;
    this.nonParallelizableOperation = false;
    this.tasksInProgress = 0;
    this.queue = [];
  }

  push(fn, {
    parallel
  } = {
    parallel: true
  }) {
    const task = new Task(fn, parallel);
    this.queue.push(task);
    this.processQueue();
    return task.getPromise();
  }

  processQueue() {
    if (!this.queue.length || this.nonParallelizableOperation || this.disposed) {
      return;
    }

    const task = this.queue[0];
    const canRunParallelOp = task.runsInParallel() && this.tasksInProgress < this.parallelism;
    const canRunSerialOp = task.runsInSerial() && this.tasksInProgress === 0;

    if (canRunSerialOp || canRunParallelOp) {
      this.processTask(task, task.runsInParallel());
      this.queue.shift();
      this.processQueue();
    }
  }

  async processTask(task, runsInParallel) {
    if (this.disposed) {
      return;
    }

    this.tasksInProgress++;

    if (!runsInParallel) {
      this.nonParallelizableOperation = true;
    }

    try {
      await task.execute();
    } finally {
      this.tasksInProgress--;
      this.nonParallelizableOperation = false;
      this.processQueue();
    }
  }

  dispose() {
    this.queue = [];
    this.disposed = true;
  }

}

exports["default"] = AsyncQueue;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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