import * as R from './Record';
import { getLastSemigroup } from './Semigroup';
import { fromEquals } from './Setoid';
export const URI = 'StrMap';
const liftSeparated = ({ left, right }) => {
    return {
        left: new StrMap(left),
        right: new StrMap(right)
    };
};
/**
 * @data
 * @constructor StrMap
 * @since 1.0.0
 */
export class StrMap {
    constructor(value) {
        this.value = value;
    }
    mapWithKey(f) {
        return new StrMap(R.mapWithKey(this.value, f));
    }
    map(f) {
        return this.mapWithKey((_, a) => f(a));
    }
    reduce(b, f) {
        return R.reduce(this.value, b, f);
    }
    /**
     * @since 1.12.0
     */
    foldr(b, f) {
        return R.foldr(this.value, b, f);
    }
    /**
     * @since 1.12.0
     */
    reduceWithKey(b, f) {
        return R.reduceWithKey(this.value, b, f);
    }
    /**
     * @since 1.12.0
     */
    foldrWithKey(b, f) {
        return R.foldrWithKey(this.value, b, f);
    }
    filter(p) {
        return this.filterWithKey((_, a) => p(a));
    }
    /**
     * @since 1.12.0
     */
    filterMap(f) {
        return this.filterMapWithKey((_, a) => f(a));
    }
    /**
     * @since 1.12.0
     */
    partition(p) {
        return this.partitionWithKey((_, a) => p(a));
    }
    /**
     * @since 1.12.0
     */
    partitionMap(f) {
        return this.partitionMapWithKey((_, a) => f(a));
    }
    /**
     * @since 1.12.0
     */
    separate() {
        return liftSeparated(R.separate(this.value));
    }
    /**
     * Use `partitionMapWithKey` instead
     * @since 1.12.0
     * @deprecated
     */
    partitionMapWithIndex(f) {
        return this.partitionMapWithKey(f);
    }
    /**
     * @since 1.14.0
     */
    partitionMapWithKey(f) {
        return liftSeparated(R.partitionMapWithKey(this.value, f));
    }
    /**
     * Use `partitionWithKey` instead
     * @since 1.12.0
     * @deprecated
     */
    partitionWithIndex(p) {
        return this.partitionWithKey(p);
    }
    /**
     * @since 1.14.0
     */
    partitionWithKey(p) {
        return liftSeparated(R.partitionWithKey(this.value, p));
    }
    /**
     * Use `filterMapWithKey` instead
     * @since 1.12.0
     * @deprecated
     */
    filterMapWithIndex(f) {
        return this.filterMapWithKey(f);
    }
    /**
     * @since 1.14.0
     */
    filterMapWithKey(f) {
        return new StrMap(R.filterMapWithKey(this.value, f));
    }
    /**
     * Use `filterWithKey` instead
     * @since 1.12.0
     * @deprecated
     */
    filterWithIndex(p) {
        return this.filterWithKey(p);
    }
    /**
     * @since 1.14.0
     */
    filterWithKey(p) {
        return new StrMap(R.filterWithKey(this.value, p));
    }
    /**
     * @since 1.14.0
     */
    every(predicate) {
        return R.every(this.value, predicate);
    }
    /**
     * @since 1.14.0
     */
    some(predicate) {
        return R.some(this.value, predicate);
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (S) => {
    const SR = R.getShow(S);
    return {
        show: s => `new StrMap(${SR.show(s.value)})`
    };
};
/**
 *
 * @since 1.10.0
 */
const empty = new StrMap(R.empty);
const concat = (S) => {
    const concat = R.getMonoid(S).concat;
    return (x, y) => new StrMap(concat(x.value, y.value));
};
/**
 *
 * @since 1.0.0
 */
export const getMonoid = (S = getLastSemigroup()) => {
    return {
        concat: concat(S),
        empty
    };
};
const map = (fa, f) => {
    return fa.map(f);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => {
    const foldMapM = R.foldMap(M);
    return (fa, f) => foldMapM(fa.value, f);
};
const foldr = (fa, b, f) => {
    return fa.foldr(b, f);
};
const reduceWithIndex = (fa, b, f) => {
    return fa.reduceWithKey(b, f);
};
const foldMapWithIndex = (M) => {
    const foldMapWithKey = R.foldMapWithKey(M);
    return (fa, f) => foldMapWithKey(fa.value, f);
};
const foldrWithIndex = (fa, b, f) => {
    return fa.foldrWithKey(b, f);
};
export function traverseWithKey(F) {
    const traverseWithKeyF = R.traverseWithKey(F);
    return (ta, f) => F.map(traverseWithKeyF(ta.value, f), d => new StrMap(d));
}
function traverse(F) {
    const traverseWithKeyF = traverseWithKey(F);
    return (ta, f) => traverseWithKeyF(ta, (_, a) => f(a));
}
function sequence(F) {
    const traverseWithKeyF = traverseWithKey(F);
    return ta => traverseWithKeyF(ta, (_, a) => a);
}
/**
 * Test whether one dictionary contains all of the keys and values contained in another dictionary
 *
 * @since 1.0.0
 */
export const isSubdictionary = (S) => {
    const isSubrecordS = R.isSubrecord(S);
    return (d1, d2) => isSubrecordS(d1.value, d2.value);
};
/**
 * Calculate the number of key/value pairs in a dictionary
 *
 * @since 1.0.0
 */
export const size = (d) => {
    return R.size(d.value);
};
/**
 * Test whether a dictionary is empty
 *
 * @since 1.0.0
 */
export const isEmpty = (d) => {
    return R.isEmpty(d.value);
};
/**
 *
 * @since 1.0.0
 */
export const getSetoid = (S) => {
    const isSubrecordS = R.isSubrecord(S);
    return fromEquals((x, y) => isSubrecordS(x.value, y.value) && isSubrecordS(y.value, x.value));
};
/**
 * Create a dictionary with one key/value pair
 *
 * @since 1.0.0
 */
export const singleton = (k, a) => {
    return new StrMap(R.singleton(k, a));
};
/**
 * Lookup the value for a key in a dictionary
 *
 * @since 1.0.0
 */
export const lookup = (k, d) => {
    return R.lookup(k, d.value);
};
export function fromFoldable(
// tslint:disable-next-line: deprecation
F) {
    const fromFoldableF = R.fromFoldable(F);
    return (ta, onConflict) => new StrMap(fromFoldableF(ta, onConflict));
}
/**
 *
 * @since 1.0.0
 */
export const collect = (d, f) => {
    return R.collect(d.value, f);
};
/**
 *
 * @since 1.0.0
 */
export const toArray = (d) => {
    return R.toArray(d.value);
};
export function toUnfoldable(U) {
    const toUnfoldableU = R.toUnfoldable(U);
    return (d) => toUnfoldableU(d.value);
}
/**
 * Insert or replace a key/value pair in a map
 *
 * @since 1.0.0
 */
export const insert = (k, a, d) => {
    const value = R.insert(k, a, d.value);
    return value === d.value ? d : new StrMap(value);
};
/**
 * Delete a key and value from a map
 *
 * @since 1.0.0
 */
export const remove = (k, d) => {
    const value = R.remove(k, d.value);
    return value === d.value ? d : new StrMap(value);
};
/**
 * Delete a key and value from a map, returning the value as well as the subsequent map
 *
 * @since 1.0.0
 */
export const pop = (k, d) => {
    return R.pop(k, d.value).map(([a, d]) => [a, new StrMap(d)]);
};
/**
 * @since 1.14.0
 */
export function elem(S) {
    return (a, fa) => fa.some(x => S.equals(x, a));
}
const filterMap = (fa, f) => {
    return fa.filterMap(f);
};
const filter = (fa, p) => {
    return fa.filter(p);
};
const compact = (fa) => {
    return new StrMap(R.compact(fa.value));
};
const separate = (fa) => {
    return fa.separate();
};
const partitionMap = (fa, f) => {
    return fa.partitionMap(f);
};
const partition = (fa, p) => {
    return fa.partition(p);
};
const wither = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), compact);
};
const wilt = (F) => {
    const traverseF = traverse(F);
    return (wa, f) => F.map(traverseF(wa, f), separate);
};
const mapWithIndex = (fa, f) => {
    return fa.mapWithKey(f);
};
// tslint:disable-next-line: deprecation
const traverseWithIndex = traverseWithKey;
const partitionMapWithIndex = (fa, f) => {
    return fa.partitionMapWithKey(f);
};
const partitionWithIndex = (fa, p) => {
    return fa.partitionWithKey(p);
};
const filterMapWithIndex = (fa, f) => {
    return fa.filterMapWithKey(f);
};
const filterWithIndex = (fa, p) => {
    return fa.filterWithKey(p);
};
/**
 * @since 1.0.0
 */
export const strmap = {
    URI,
    map,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence,
    compact,
    separate,
    filter,
    filterMap,
    partition,
    partitionMap,
    wither,
    wilt,
    mapWithIndex,
    reduceWithIndex,
    foldMapWithIndex,
    foldrWithIndex,
    traverseWithIndex,
    partitionMapWithIndex,
    partitionWithIndex,
    filterMapWithIndex,
    filterWithIndex
};
