import { identity } from './function';
import { left as eitherLeft, right as eitherRight } from './Either';
export const URI = 'Reader';
/**
 * @since 1.0.0
 */
export class Reader {
    constructor(run) {
        this.run = run;
    }
    map(f) {
        return new Reader((e) => f(this.run(e)));
    }
    ap(fab) {
        return new Reader((e) => fab.run(e)(this.run(e)));
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return new Reader((e) => f(this.run(e)).run(e));
    }
    /**
     * @since 1.6.1
     */
    local(f) {
        return new Reader(e => this.run(f(e)));
    }
}
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new Reader(() => a);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
/**
 * reads the current context
 *
 * @since 1.0.0
 */
export const ask = () => {
    return new Reader(identity);
};
/**
 * Projects a value from the global context in a Reader
 *
 * @since 1.0.0
 */
export const asks = (f) => {
    return new Reader(f);
};
/**
 * changes the value of the local context during the execution of the action `fa`
 *
 * @since 1.0.0
 */
export const local = (f) => (fa) => {
    return fa.local(f);
};
const promap = (fbc, f, g) => {
    return new Reader(a => g(fbc.run(f(a))));
};
const compose = (ab, la) => {
    return new Reader(l => ab.run(la.run(l)));
};
const id = () => {
    return new Reader(identity);
};
const first = (pab) => {
    return new Reader(([a, c]) => [pab.run(a), c]);
};
const second = (pbc) => {
    return new Reader(([a, b]) => [a, pbc.run(b)]);
};
const left = (pab) => {
    return new Reader(e => e.fold(a => eitherLeft(pab.run(a)), eitherRight));
};
const right = (pbc) => {
    return new Reader(e => e.fold(eitherLeft, b => eitherRight(pbc.run(b))));
};
/**
 * @since 1.14.0
 */
export const getSemigroup = (S) => {
    return {
        concat: (x, y) => new Reader(e => S.concat(x.run(e), y.run(e)))
    };
};
/**
 * @since 1.14.0
 */
export const getMonoid = (M) => {
    return Object.assign({}, getSemigroup(M), { empty: of(M.empty) });
};
/**
 * @since 1.0.0
 */
export const reader = {
    URI,
    map,
    of,
    ap,
    chain,
    promap,
    compose,
    id,
    first,
    second,
    left,
    right
};
