/**
 * @file `Task<A>` represents an asynchronous computation that yields a value of type `A` and **never fails**.
 * If you want to represent an asynchronous computation that may fail, please see `TaskEither`.
 */
import { left, right } from './Either';
import { constant, constIdentity, identity, toString } from './function';
export const URI = 'Task';
/**
 * @since 1.0.0
 */
export class Task {
    constructor(run) {
        this.run = run;
    }
    map(f) {
        return new Task(() => this.run().then(f));
    }
    ap(fab) {
        return new Task(() => Promise.all([fab.run(), this.run()]).then(([f, a]) => f(a)));
    }
    /**
     * Flipped version of `ap`
     */
    ap_(fb) {
        return fb.ap(this);
    }
    /**
     * Combine two effectful actions, keeping only the result of the first
     * @since 1.6.0
     */
    applyFirst(fb) {
        return fb.ap(this.map(constant));
    }
    /**
     * Combine two effectful actions, keeping only the result of the second
     * @since 1.5.0
     */
    applySecond(fb) {
        return fb.ap(this.map(constIdentity));
    }
    chain(f) {
        return new Task(() => this.run().then(a => f(a).run()));
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `new Task(${toString(this.run)})`;
    }
}
const map = (fa, f) => {
    return fa.map(f);
};
const of = (a) => {
    return new Task(() => Promise.resolve(a));
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
/**
 * @since 1.0.0
 */
export const getRaceMonoid = () => {
    return {
        concat: (x, y) => new Task(() => new Promise((resolve, reject) => {
            let running = true;
            const resolveFirst = (a) => {
                if (running) {
                    running = false;
                    resolve(a);
                }
            };
            const rejectFirst = (e) => {
                if (running) {
                    running = false;
                    reject(e);
                }
            };
            x.run().then(resolveFirst, rejectFirst);
            y.run().then(resolveFirst, rejectFirst);
        })),
        empty: never
    };
};
const never = new Task(() => new Promise(_ => undefined));
/**
 * @since 1.0.0
 */
export const getSemigroup = (S) => {
    return {
        concat: (x, y) => new Task(() => x.run().then(rx => y.run().then(ry => S.concat(rx, ry))))
    };
};
/**
 * @since 1.0.0
 */
export const getMonoid = (M) => {
    return Object.assign({}, getSemigroup(M), { empty: of(M.empty) });
};
/**
 * @since 1.0.0
 */
export const tryCatch = (f, onrejected) => {
    return new Task(() => f().then(right, reason => left(onrejected(reason))));
};
/**
 * Lifts an IO action into a Task
 *
 * @since 1.0.0
 */
export const fromIO = (io) => {
    return new Task(() => Promise.resolve(io.run()));
};
/**
 * @since 1.7.0
 */
export const delay = (millis, a) => {
    return new Task(() => new Promise(resolve => {
        setTimeout(() => {
            resolve(a);
        }, millis);
    }));
};
const fromTask = identity;
/**
 * @since 1.0.0
 */
export const task = {
    URI,
    map,
    of,
    ap,
    chain,
    fromIO,
    fromTask
};
/**
 * Like `Task` but `ap` is sequential
 *
 * @since 1.10.0
 */
export const taskSeq = Object.assign({}, task, { ap: (fab, fa) => fab.chain(f => fa.map(f)) });
