"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var scale_band_1 = require("../utils/scales/scale_band");
var scale_continuous_1 = require("../utils/scales/scale_continuous");
/**
 * Count the max number of bars in cluster value.
 * Doesn't take in consideration areas, lines or points.
 * @param stacked all the stacked formatted dataseries
 * @param nonStacked all the non-stacked formatted dataseries
 */
function countBarsInCluster(stacked, nonStacked) {
    // along x axis, we count one "space" per bar series.
    // we ignore the points, areas, lines as they are
    // aligned with the x value and doesn't occupy space
    var nonStackedBarsInCluster = nonStacked.reduce(function (acc, ns) {
        return acc + ns.counts.barSeries;
    }, 0);
    // count stacked bars groups as 1 per group
    var stackedBarsInCluster = stacked.reduce(function (acc, ns) {
        return acc + (ns.counts.barSeries > 0 ? 1 : 0);
    }, 0);
    var totalBarsInCluster = nonStackedBarsInCluster + stackedBarsInCluster;
    return {
        nonStackedBarsInCluster: nonStackedBarsInCluster,
        stackedBarsInCluster: stackedBarsInCluster,
        totalBarsInCluster: totalBarsInCluster,
    };
}
exports.countBarsInCluster = countBarsInCluster;
/**
 * Compute the x scale used to align geometries to the x axis.
 * @param xDomain the x domain
 * @param totalBarsInCluster the total number of grouped series
 * @param axisLength the length of the x axis
 */
function computeXScale(xDomain, totalBarsInCluster, minRange, maxRange, barsPadding) {
    var scaleType = xDomain.scaleType, minInterval = xDomain.minInterval, domain = xDomain.domain, isBandScale = xDomain.isBandScale, timeZone = xDomain.timeZone;
    var rangeDiff = Math.abs(maxRange - minRange);
    var isInverse = maxRange < minRange;
    if (scaleType === "ordinal" /* Ordinal */) {
        var dividend = totalBarsInCluster > 0 ? totalBarsInCluster : 1;
        var bandwidth = rangeDiff / (domain.length * dividend);
        return new scale_band_1.ScaleBand(domain, [minRange, maxRange], bandwidth, barsPadding);
    }
    else {
        if (isBandScale) {
            var intervalCount = (domain[1] - domain[0]) / minInterval;
            var bandwidth = rangeDiff / (intervalCount + 1);
            var start = isInverse ? minRange - bandwidth : minRange;
            var end = isInverse ? maxRange : maxRange - bandwidth;
            return new scale_continuous_1.ScaleContinuous(scaleType, domain, [start, end], bandwidth / totalBarsInCluster, minInterval, timeZone, totalBarsInCluster, barsPadding);
        }
        else {
            return new scale_continuous_1.ScaleContinuous(scaleType, domain, [minRange, maxRange], 0, minInterval, timeZone, totalBarsInCluster, barsPadding);
        }
    }
}
exports.computeXScale = computeXScale;
/**
 * Compute the y scales, one per groupId for the y axis.
 * @param yDomains the y domains
 * @param axisLength the axisLength of the y axis
 */
function computeYScales(yDomains, minRange, maxRange) {
    var yScales = new Map();
    yDomains.forEach(function (yDomain) {
        var yScale = new scale_continuous_1.ScaleContinuous(yDomain.scaleType, yDomain.domain, [minRange, maxRange]);
        yScales.set(yDomain.groupId, yScale);
    });
    return yScales;
}
exports.computeYScales = computeYScales;
//# sourceMappingURL=scales.js.map