"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _atom = require("atom");

var _eventKit = require("event-kit");

var _electron = require("electron");

var _react = _interopRequireDefault(require("react"));

var _reactDom = _interopRequireDefault(require("react-dom"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _lodash = _interopRequireDefault(require("lodash.memoize"));

var _fsExtra = _interopRequireDefault(require("fs-extra"));

var _octicon = _interopRequireDefault(require("../atom/octicon"));

var _helpers = require("../helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _objectWithoutProperties(source, excluded) { if (source == null) return {}; var target = _objectWithoutPropertiesLoose(source, excluded); var key, i; if (Object.getOwnPropertySymbols) { var sourceSymbolKeys = Object.getOwnPropertySymbols(source); for (i = 0; i < sourceSymbolKeys.length; i++) { key = sourceSymbolKeys[i]; if (excluded.indexOf(key) >= 0) continue; if (!Object.prototype.propertyIsEnumerable.call(source, key)) continue; target[key] = source[key]; } } return target; }

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; var ownKeys = Object.keys(source); if (typeof Object.getOwnPropertySymbols === 'function') { ownKeys = ownKeys.concat(Object.getOwnPropertySymbols(source).filter(function (sym) { return Object.getOwnPropertyDescriptor(source, sym).enumerable; })); } ownKeys.forEach(function (key) { _defineProperty(target, key, source[key]); }); } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const {
  dialog
} = _electron.remote;
const genArray = (0, _lodash["default"])(function genArray(interval, count) {
  const arr = [];

  for (let i = 1; i <= count; i++) {
    arr.push(interval * i);
  }

  return arr;
}, (interval, count) => `${interval}:${count}`);

class Marker {
  static deserialize(data) {
    const marker = new Marker(data.label, () => {});
    marker.end = data.end;
    marker.markers = data.markers;
    return marker;
  }

  constructor(label, didUpdate) {
    this.label = label;
    this.didUpdate = didUpdate;
    this.end = null;
    this.markers = [];
  }

  getStart() {
    return this.markers.length ? this.markers[0].start : null;
  }

  getEnd() {
    return this.end;
  }

  mark(sectionName, start) {
    this.markers.push({
      name: sectionName,
      start: start || performance.now()
    });
  }

  finalize() {
    this.end = performance.now();
    this.didUpdate();
  }

  getTimings() {
    return this.markers.map((timing, idx, ary) => {
      const next = ary[idx + 1];
      const end = next ? next.start : this.getEnd();
      return _objectSpread({}, timing, {
        end
      });
    });
  }

  serialize() {
    return {
      label: this.label,
      end: this.end,
      markers: this.markers.slice()
    };
  }

}

class MarkerTooltip extends _react["default"].Component {
  render() {
    const {
      marker
    } = this.props;
    const timings = marker.getTimings();
    return _react["default"].createElement("div", {
      style: {
        textAlign: 'left',
        maxWidth: 300,
        whiteSpace: 'initial'
      }
    }, _react["default"].createElement("strong", null, _react["default"].createElement("tt", null, marker.label)), _react["default"].createElement("ul", {
      style: {
        paddingLeft: 20,
        marginTop: 10
      }
    }, timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return _react["default"].createElement("li", {
        key: name
      }, name, ": ", Math.floor(duration * 100) / 100, "ms");
    })));
  }

}

_defineProperty(MarkerTooltip, "propTypes", {
  marker: _propTypes["default"].instanceOf(Marker).isRequired
});

const COLORS = {
  queued: 'red',
  prepare: 'cyan',
  nexttick: 'yellow',
  execute: 'green',
  ipc: 'pink'
};

class MarkerSpan extends _react["default"].Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleMouseOver', 'handleMouseOut');
  }

  render() {
    const _this$props = this.props,
          {
      marker
    } = _this$props,
          others = _objectWithoutProperties(_this$props, ["marker"]);

    const timings = marker.getTimings();
    const totalTime = marker.getEnd() - marker.getStart();
    const percentages = timings.map(({
      name,
      start,
      end
    }) => {
      const duration = end - start;
      return {
        color: COLORS[name],
        percent: duration / totalTime * 100
      };
    });
    return _react["default"].createElement("span", _extends({}, others, {
      ref: c => {
        this.element = c;
      },
      onMouseOver: this.handleMouseOver,
      onMouseOut: this.handleMouseOut
    }), percentages.map(({
      color,
      percent
    }, i) => {
      const style = {
        width: `${percent}%`,
        background: color
      };
      return _react["default"].createElement("span", {
        className: "waterfall-marker-section",
        key: i,
        style: style
      });
    }));
  }

  handleMouseOver(e) {
    const elem = document.createElement('div');

    _reactDom["default"].render(_react["default"].createElement(MarkerTooltip, {
      marker: this.props.marker
    }), elem);

    this.tooltipDisposable = atom.tooltips.add(this.element, {
      item: elem,
      placement: 'auto bottom',
      trigger: 'manual'
    });
  }

  closeTooltip() {
    this.tooltipDisposable && this.tooltipDisposable.dispose();
    this.tooltipDisposable = null;
  }

  handleMouseOut(e) {
    this.closeTooltip();
  }

  componentWillUnmount() {
    this.closeTooltip();
  }

}

_defineProperty(MarkerSpan, "propTypes", {
  marker: _propTypes["default"].instanceOf(Marker).isRequired
});

class Waterfall extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'renderMarker');
    this.state = this.getNextState(props);
  }

  componentWillReceiveProps(nextProps) {
    this.setState(this.getNextState(nextProps));
  }

  getNextState(props) {
    const {
      markers
    } = props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const totalDuration = endTime - startTime;
    let timelineMarkInterval = null;

    if (props.zoomFactor <= 0.15) {
      timelineMarkInterval = 1000;
    } else if (props.zoomFactor <= 0.3) {
      timelineMarkInterval = 500;
    } else if (props.zoomFactor <= 0.6) {
      timelineMarkInterval = 250;
    } else {
      timelineMarkInterval = 100;
    }

    const timelineMarks = genArray(timelineMarkInterval, Math.ceil(totalDuration / timelineMarkInterval));
    return {
      firstMarker,
      lastMarker,
      startTime,
      endTime,
      totalDuration,
      timelineMarks
    };
  }

  render() {
    return _react["default"].createElement("div", {
      className: "waterfall-scroller"
    }, _react["default"].createElement("div", {
      className: "waterfall-container"
    }, this.renderTimeMarkers(), this.renderTimeline(), this.props.markers.map(this.renderMarker)));
  }

  renderTimeline() {
    return _react["default"].createElement("div", {
      className: "waterfall-timeline"
    }, "\xA0", this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react["default"].createElement("span", {
        className: "waterfall-timeline-label",
        style: style,
        key: `tl:${time}`
      }, time, "ms");
    }));
  }

  renderTimeMarkers() {
    return _react["default"].createElement("div", {
      className: "waterfall-time-markers"
    }, this.state.timelineMarks.map(time => {
      const leftPos = time * this.props.zoomFactor;
      const style = {
        left: leftPos
      };
      return _react["default"].createElement("span", {
        className: "waterfall-time-marker",
        style: style,
        key: `tm:${time}`
      });
    }));
  }

  renderMarker(marker, i) {
    if (marker.getStart() === null || marker.getEnd() === null) {
      return _react["default"].createElement("div", {
        key: i
      });
    }

    const startOffset = marker.getStart() - this.state.startTime;
    const duration = marker.getEnd() - marker.getStart();
    const markerStyle = {
      left: startOffset * this.props.zoomFactor,
      width: duration * this.props.zoomFactor
    };
    return _react["default"].createElement("div", {
      className: "waterfall-row",
      key: i
    }, _react["default"].createElement("span", {
      className: "waterfall-row-label",
      style: {
        paddingLeft: markerStyle.left + markerStyle.width
      }
    }, marker.label), _react["default"].createElement(MarkerSpan, {
      className: "waterfall-marker",
      style: markerStyle,
      marker: marker
    }));
  }

}

_defineProperty(Waterfall, "propTypes", {
  markers: _propTypes["default"].arrayOf(_propTypes["default"].instanceOf(Marker)).isRequired,
  zoomFactor: _propTypes["default"].number.isRequired
});

class WaterfallWidget extends _react["default"].Component {
  constructor(props, context) {
    super(props, context);
    (0, _helpers.autobind)(this, 'handleZoomFactorChange', 'handleCollapseClick', 'handleExportClick');
    this.state = {
      zoomFactor: 0.3,
      collapsed: false
    };
  }

  render() {
    const {
      markers
    } = this.props;
    const firstMarker = markers[0];
    const lastMarker = markers[markers.length - 1];
    const startTime = firstMarker.getStart();
    const endTime = lastMarker.getEnd();
    const duration = endTime - startTime;
    return _react["default"].createElement("div", {
      className: "waterfall-widget inset-pannel"
    }, _react["default"].createElement("div", {
      className: "waterfall-header"
    }, _react["default"].createElement("div", {
      className: "waterfall-header-text"
    }, _react["default"].createElement("span", {
      onClick: this.handleCollapseClick,
      className: "collapse-toggle"
    }, this.state.collapsed ? '\u25b6' : '\u25bc'), this.props.markers.length, " event(s) over ", Math.floor(duration), "ms"), _react["default"].createElement("div", {
      className: "waterfall-header-controls"
    }, _react["default"].createElement("button", {
      className: "waterfall-export-button btn btn-sm",
      onClick: this.handleExportClick
    }, "Export"), _react["default"].createElement(_octicon["default"], {
      icon: "search"
    }), _react["default"].createElement("input", {
      type: "range",
      className: "input-range",
      min: 0.1,
      max: 1,
      step: 0.01,
      value: this.state.zoomFactor,
      onChange: this.handleZoomFactorChange
    }))), this.state.collapsed ? null : _react["default"].createElement(Waterfall, {
      markers: this.props.markers,
      zoomFactor: this.state.zoomFactor
    }));
  }

  handleZoomFactorChange(e) {
    this.setState({
      zoomFactor: parseFloat(e.target.value)
    });
  }

  handleCollapseClick(e) {
    this.setState(s => ({
      collapsed: !s.collapsed
    }));
  }

  handleExportClick(e) {
    e.preventDefault();
    const json = JSON.stringify(this.props.markers.map(m => m.serialize()), null, '  ');
    const buffer = new _atom.TextBuffer({
      text: json
    });
    dialog.showSaveDialog({
      defaultPath: 'git-timings.json'
    }, filename => {
      if (!filename) {
        return;
      }

      buffer.saveAs(filename);
    });
  }

}

_defineProperty(WaterfallWidget, "propTypes", {
  markers: _propTypes["default"].arrayOf(_propTypes["default"].instanceOf(Marker)).isRequired
});

let markers = null;
let groupId = 0;
const groups = [];
let lastMarkerTime = null;
let updateTimer = null;

class GitTimingsView extends _react["default"].Component {
  static buildURI() {
    return this.uriPattern;
  }

  static generateMarker(label) {
    const marker = new Marker(label, () => {
      GitTimingsView.scheduleUpdate();
    });
    const now = performance.now();

    if (!markers || lastMarkerTime && Math.abs(now - lastMarkerTime) >= 5000) {
      groupId++;
      markers = [];
      groups.unshift({
        id: groupId,
        markers
      });

      if (groups.length > 100) {
        groups.pop();
      }
    }

    lastMarkerTime = now;
    markers.push(marker);
    GitTimingsView.scheduleUpdate();
    return marker;
  }

  static restoreGroup(group) {
    groupId++;
    groups.unshift({
      id: groupId,
      markers: group
    });
    GitTimingsView.scheduleUpdate(true);
  }

  static scheduleUpdate(immediate = false) {
    if (updateTimer) {
      clearTimeout(updateTimer);
    }

    updateTimer = setTimeout(() => {
      GitTimingsView.emitter.emit('did-update');
    }, immediate ? 0 : 1000);
  }

  static onDidUpdate(callback) {
    return GitTimingsView.emitter.on('did-update', callback);
  }

  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'handleImportClick');
  }

  componentDidMount() {
    this.subscriptions = new _eventKit.CompositeDisposable(GitTimingsView.onDidUpdate(() => this.forceUpdate()));
  }

  componentWillUnmount() {
    this.subscriptions.dispose();
  }

  render() {
    return _react["default"].createElement("div", {
      className: "github-GitTimingsView"
    }, _react["default"].createElement("div", {
      className: "github-GitTimingsView-header"
    }, _react["default"].createElement("button", {
      className: "import-button btn",
      onClick: this.handleImportClick
    }, "Import")), groups.map((group, idx) => _react["default"].createElement(WaterfallWidget, {
      key: group.id,
      markers: group.markers
    })));
  }

  handleImportClick(e) {
    e.preventDefault();
    dialog.showOpenDialog({
      properties: ['openFile']
    }, async filenames => {
      if (!filenames) {
        return;
      }

      const filename = filenames[0];

      try {
        const contents = await _fsExtra["default"].readFile(filename, {
          encoding: 'utf8'
        });
        const data = JSON.parse(contents);
        const restoredMarkers = data.map(item => Marker.deserialize(item));
        GitTimingsView.restoreGroup(restoredMarkers);
      } catch (_err) {
        atom.notifications.addError(`Could not import timings from ${filename}`);
      }
    });
  }

  serialize() {
    return {
      deserializer: 'GitTimingsView'
    };
  }

  getURI() {
    return this.constructor.buildURI();
  }

  getTitle() {
    return 'GitHub Package Timings View';
  }

}

exports["default"] = GitTimingsView;

_defineProperty(GitTimingsView, "uriPattern", 'atom-github://debug/timings');

_defineProperty(GitTimingsView, "emitter", new _eventKit.Emitter());
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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