"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HttpServer = void 0;

var _http_tools = require("./http_tools");

var _auth = require("./lifecycle/auth");

var _on_post_auth = require("./lifecycle/on_post_auth");

var _on_pre_auth = require("./lifecycle/on_pre_auth");

var _cookie_session_storage = require("./cookie_session_storage");

var _auth_state_storage = require("./auth_state_storage");

var _auth_headers_storage = require("./auth_headers_storage");

var _base_path_service = require("./base_path_service");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class HttpServer {
  constructor(logger, name) {
    this.logger = logger;
    this.name = name;

    _defineProperty(this, "server", void 0);

    _defineProperty(this, "config", void 0);

    _defineProperty(this, "registeredRouters", new Set());

    _defineProperty(this, "authRegistered", false);

    _defineProperty(this, "log", void 0);

    _defineProperty(this, "authState", void 0);

    _defineProperty(this, "authHeaders", void 0);

    this.authState = new _auth_state_storage.AuthStateStorage(() => this.authRegistered);
    this.authHeaders = new _auth_headers_storage.AuthHeadersStorage();
    this.log = logger.get('http', 'server', name);
  }

  isListening() {
    return this.server !== undefined && this.server.listener.listening;
  }

  registerRouter(router) {
    if (this.isListening()) {
      throw new Error('Routers can be registered only when HTTP server is stopped.');
    }

    this.log.debug(`registering route handler for [${router.path}]`);
    this.registeredRouters.add(router);
  }

  setup(config) {
    const serverOptions = (0, _http_tools.getServerOptions)(config);
    this.server = (0, _http_tools.createServer)(serverOptions);
    this.config = config;
    const basePathService = new _base_path_service.BasePath(config.basePath);
    this.setupBasePathRewrite(config, basePathService);
    return {
      registerRouter: this.registerRouter.bind(this),
      registerOnPreAuth: this.registerOnPreAuth.bind(this),
      registerOnPostAuth: this.registerOnPostAuth.bind(this),
      registerAuth: (fn, cookieOptions) => this.registerAuth(fn, cookieOptions, config.basePath),
      basePath: basePathService,
      auth: {
        get: this.authState.get,
        isAuthenticated: this.authState.isAuthenticated,
        getAuthHeaders: this.authHeaders.get
      },
      // Return server instance with the connection options so that we can properly
      // bridge core and the "legacy" Kibana internally. Once this bridge isn't
      // needed anymore we shouldn't return the instance from this method.
      server: this.server
    };
  }

  async start() {
    if (this.server === undefined) {
      throw new Error('Http server is not setup up yet');
    }

    this.log.debug('starting http server');

    for (const router of this.registeredRouters) {
      for (const route of router.getRoutes()) {
        const {
          authRequired = true,
          tags
        } = route.options;
        this.server.route({
          handler: route.handler,
          method: route.method,
          path: this.getRouteFullPath(router.path, route.path),
          options: {
            auth: authRequired ? undefined : false,
            tags: tags ? Array.from(tags) : undefined
          }
        });
      }
    }

    await this.server.start();
    const serverPath = this.config.rewriteBasePath || this.config.basePath || '';
    this.log.info('http server running');
    this.log.debug(`http server listening on ${this.server.info.uri}${serverPath}`);
  }

  async stop() {
    if (this.server === undefined) {
      return;
    }

    this.log.debug('stopping http server');
    await this.server.stop();
    this.server = undefined;
  }

  setupBasePathRewrite(config, basePathService) {
    if (config.basePath === undefined || !config.rewriteBasePath) {
      return;
    }

    this.registerOnPreAuth((request, toolkit) => {
      const oldUrl = request.url.href;
      const newURL = basePathService.remove(oldUrl);
      const shouldRedirect = newURL !== oldUrl;

      if (shouldRedirect) {
        return toolkit.redirected(newURL, {
          forward: true
        });
      }

      return toolkit.rejected(new Error('not found'), {
        statusCode: 404
      });
    });
  }

  getRouteFullPath(routerPath, routePath) {
    // If router's path ends with slash and route's path starts with slash,
    // we should omit one of them to have a valid concatenated path.
    const routePathStartIndex = routerPath.endsWith('/') && routePath.startsWith('/') ? 1 : 0;
    return `${routerPath}${routePath.slice(routePathStartIndex)}`;
  }

  registerOnPostAuth(fn) {
    if (this.server === undefined) {
      throw new Error('Server is not created yet');
    }

    this.server.ext('onPostAuth', (0, _on_post_auth.adoptToHapiOnPostAuthFormat)(fn));
  }

  registerOnPreAuth(fn) {
    if (this.server === undefined) {
      throw new Error('Server is not created yet');
    }

    this.server.ext('onRequest', (0, _on_pre_auth.adoptToHapiOnPreAuthFormat)(fn));
  }

  async registerAuth(fn, cookieOptions, basePath) {
    if (this.server === undefined) {
      throw new Error('Server is not created yet');
    }

    if (this.authRegistered) {
      throw new Error('Auth interceptor was already registered');
    }

    this.authRegistered = true;
    const sessionStorageFactory = await (0, _cookie_session_storage.createCookieSessionStorageFactory)(this.logger.get('http', 'server', this.name, 'cookie-session-storage'), this.server, cookieOptions, basePath);
    this.server.auth.scheme('login', () => ({
      authenticate: (0, _auth.adoptToHapiAuthFormat)(fn, (req, {
        state,
        headers
      }) => {
        this.authState.set(req, state);
        this.authHeaders.set(req, headers); // we mutate headers only for the backward compatibility with the legacy platform.
        // where some plugin read directly from headers to identify whether a user is authenticated.

        Object.assign(req.headers, headers);
      })
    }));
    this.server.auth.strategy('session', 'login'); // The default means that the `session` strategy that is based on `login` schema defined above will be
    // automatically assigned to all routes that don't contain an auth config.
    // should be applied for all routes if they don't specify auth strategy in route declaration
    // https://github.com/hapijs/hapi/blob/master/API.md#-serverauthdefaultoptions

    this.server.auth.default('session');
    return {
      sessionStorageFactory
    };
  }

}

exports.HttpServer = HttpServer;