"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;

var _react = _interopRequireDefault(require("react"));

var _propTypes = _interopRequireDefault(require("prop-types"));

var _path = _interopRequireDefault(require("path"));

var _helpers = require("../helpers");

var _reporterProxy = require("../reporter-proxy");

var _propTypes2 = require("../prop-types");

var _changedFileItem = _interopRequireDefault(require("../items/changed-file-item"));

var _multiFilePatchView = _interopRequireDefault(require("../views/multi-file-patch-view"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { "default": obj }; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MultiFilePatchController extends _react["default"].Component {
  constructor(props) {
    super(props);
    (0, _helpers.autobind)(this, 'selectedRowsChanged', 'undoLastDiscard', 'diveIntoMirrorPatch', 'openFile', 'toggleFile', 'toggleRows', 'toggleModeChange', 'toggleSymlinkChange', 'discardRows');
    this.state = {
      selectionMode: 'hunk',
      selectedRows: new Set(),
      hasMultipleFileSelections: false
    };
    this.mouseSelectionInProgress = false;
    this.stagingOperationInProgress = false;
    this.lastPatchString = null;
    this.patchChangePromise = new Promise(resolve => {
      this.resolvePatchChangePromise = resolve;
    });
  }

  componentDidUpdate(prevProps) {
    if (this.lastPatchString !== null && this.lastPatchString !== this.props.multiFilePatch.toString()) {
      this.resolvePatchChangePromise();
      this.patchChangePromise = new Promise(resolve => {
        this.resolvePatchChangePromise = resolve;
      });
    }
  }

  render() {
    return _react["default"].createElement(_multiFilePatchView["default"], _extends({}, this.props, {
      selectedRows: this.state.selectedRows,
      selectionMode: this.state.selectionMode,
      hasMultipleFileSelections: this.state.hasMultipleFileSelections,
      selectedRowsChanged: this.selectedRowsChanged,
      diveIntoMirrorPatch: this.diveIntoMirrorPatch,
      openFile: this.openFile,
      toggleFile: this.toggleFile,
      toggleRows: this.toggleRows,
      toggleModeChange: this.toggleModeChange,
      toggleSymlinkChange: this.toggleSymlinkChange,
      undoLastDiscard: this.undoLastDiscard,
      discardRows: this.discardRows,
      selectNextHunk: this.selectNextHunk,
      selectPreviousHunk: this.selectPreviousHunk,
      switchToIssueish: this.props.switchToIssueish
    }));
  }

  undoLastDiscard(filePatch, {
    eventSource
  } = {}) {
    (0, _reporterProxy.addEvent)('undo-last-discard', {
      "package": 'github',
      component: this.constructor.name,
      eventSource
    });
    return this.props.undoLastDiscard(filePatch.getPath(), this.props.repository);
  }

  diveIntoMirrorPatch(filePatch) {
    const mirrorStatus = this.withStagingStatus({
      staged: 'unstaged',
      unstaged: 'staged'
    });
    const workingDirectory = this.props.repository.getWorkingDirectoryPath();

    const uri = _changedFileItem["default"].buildURI(filePatch.getPath(), workingDirectory, mirrorStatus);

    this.props.destroy();
    return this.props.workspace.open(uri);
  }

  async openFile(filePatch, positions, pending) {
    const absolutePath = _path["default"].join(this.props.repository.getWorkingDirectoryPath(), filePatch.getPath());

    const editor = await this.props.workspace.open(absolutePath, {
      pending
    });

    if (positions.length > 0) {
      editor.setCursorBufferPosition(positions[0], {
        autoscroll: false
      });

      for (const position of positions.slice(1)) {
        editor.addCursorAtBufferPosition(position);
      }

      editor.scrollToBufferPosition(positions[positions.length - 1], {
        center: true
      });
    }

    return editor;
  }

  toggleFile(filePatch) {
    return this.stagingOperation(() => {
      const methodName = this.withStagingStatus({
        staged: 'unstageFiles',
        unstaged: 'stageFiles'
      });
      return this.props.repository[methodName]([filePatch.getPath()]);
    });
  }

  async toggleRows(rowSet, nextSelectionMode) {
    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    if (chosenRows.size === 0) {
      return Promise.resolve();
    }

    return this.stagingOperation(() => {
      const patch = this.withStagingStatus({
        staged: () => this.props.multiFilePatch.getUnstagePatchForLines(chosenRows),
        unstaged: () => this.props.multiFilePatch.getStagePatchForLines(chosenRows)
      });
      return this.props.repository.applyPatchToIndex(patch);
    });
  }

  toggleModeChange(filePatch) {
    return this.stagingOperation(() => {
      const targetMode = this.withStagingStatus({
        unstaged: filePatch.getNewMode(),
        staged: filePatch.getOldMode()
      });
      return this.props.repository.stageFileModeChange(filePatch.getPath(), targetMode);
    });
  }

  toggleSymlinkChange(filePatch) {
    return this.stagingOperation(() => {
      const relPath = filePatch.getPath();
      const repository = this.props.repository;
      return this.withStagingStatus({
        unstaged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'added') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.stageFiles([relPath]);
        },
        staged: () => {
          if (filePatch.hasTypechange() && filePatch.getStatus() === 'deleted') {
            return repository.stageFileSymlinkChange(relPath);
          }

          return repository.unstageFiles([relPath]);
        }
      });
    });
  }

  async discardRows(rowSet, nextSelectionMode, {
    eventSource
  } = {}) {
    // (kuychaco) For now we only support discarding rows for MultiFilePatches that contain a single file patch
    // The only way to access this method from the UI is to be in a ChangedFileItem, which only has a single file patch
    // This check is duplicated in RootController#discardLines. We also want it here to prevent us from sending metrics
    // unnecessarily
    if (this.props.multiFilePatch.getFilePatches().length !== 1) {
      return Promise.resolve(null);
    }

    let chosenRows = rowSet;

    if (chosenRows) {
      const nextMultipleFileSelections = this.props.multiFilePatch.spansMultipleFiles(chosenRows);
      await this.selectedRowsChanged(chosenRows, nextSelectionMode, nextMultipleFileSelections);
    } else {
      chosenRows = this.state.selectedRows;
    }

    (0, _reporterProxy.addEvent)('discard-unstaged-changes', {
      "package": 'github',
      component: this.constructor.name,
      lineCount: chosenRows.size,
      eventSource
    });
    return this.props.discardLines(this.props.multiFilePatch, chosenRows, this.props.repository);
  }

  selectedRowsChanged(rows, nextSelectionMode, nextMultipleFileSelections) {
    if ((0, _helpers.equalSets)(this.state.selectedRows, rows) && this.state.selectionMode === nextSelectionMode && this.state.hasMultipleFileSelections === nextMultipleFileSelections) {
      return Promise.resolve();
    }

    return new Promise(resolve => {
      this.setState({
        selectedRows: rows,
        selectionMode: nextSelectionMode,
        hasMultipleFileSelections: nextMultipleFileSelections
      }, resolve);
    });
  }

  withStagingStatus(callbacks) {
    const callback = callbacks[this.props.stagingStatus];
    /* istanbul ignore if */

    if (!callback) {
      throw new Error(`Unknown staging status: ${this.props.stagingStatus}`);
    }

    return callback instanceof Function ? callback() : callback;
  }

  stagingOperation(fn) {
    if (this.stagingOperationInProgress) {
      return null;
    }

    this.stagingOperationInProgress = true;
    this.lastPatchString = this.props.multiFilePatch.toString();
    const operationPromise = fn();
    operationPromise.then(() => this.patchChangePromise).then(() => {
      this.stagingOperationInProgress = false;
      this.lastPatchString = null;
    });
    return operationPromise;
  }

}

exports["default"] = MultiFilePatchController;

_defineProperty(MultiFilePatchController, "propTypes", {
  repository: _propTypes["default"].object.isRequired,
  stagingStatus: _propTypes["default"].oneOf(['staged', 'unstaged']),
  multiFilePatch: _propTypes2.MultiFilePatchPropType.isRequired,
  hasUndoHistory: _propTypes["default"].bool,
  reviewCommentsLoading: _propTypes["default"].bool,
  reviewCommentThreads: _propTypes["default"].arrayOf(_propTypes["default"].shape({
    thread: _propTypes["default"].object.isRequired,
    comments: _propTypes["default"].arrayOf(_propTypes["default"].object).isRequired
  })),
  workspace: _propTypes["default"].object.isRequired,
  commands: _propTypes["default"].object.isRequired,
  keymaps: _propTypes["default"].object.isRequired,
  tooltips: _propTypes["default"].object.isRequired,
  config: _propTypes["default"].object.isRequired,
  destroy: _propTypes["default"].func.isRequired,
  discardLines: _propTypes["default"].func,
  undoLastDiscard: _propTypes["default"].func,
  surface: _propTypes["default"].func,
  switchToIssueish: _propTypes["default"].func
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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