/***************************************************************************
 *   Copyright (C) 2011 by Francesco Nwokeka <francesco.nwokeka@gmail.com> *
 *   Copyright (C) 2014 by Aleix Pol Gonzalez <aleixpol@kde.org>           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA            *
 ***************************************************************************/

import QtQuick 2.1
import org.kde.telepathy 0.1 as KTp
import org.kde.plasma.plasmoid 2.0
import org.kde.ktpcontactlist 0.1 as KTpContactList

Item
{
    id: root

    Plasmoid.switchWidth: 200
    Plasmoid.switchHeight: 300

    Plasmoid.fullRepresentation: ContactList {}

    Plasmoid.busy : true;

    KTpContactList.RegisterContactApplet {
        id: registerApplet
    }

    KTp.PresenceModel {
        id: presenceModel
        onRowsInserted: {
            updateMenu();
        }
        onRowsRemoved: {
            updateMenu();
        }
    }

    KTp.GlobalPresence {
        id: ktpPresence
        onAccountManagerChanged: {
            update();
            updateMenu();
        }

        onCurrentPresenceChanged: {
            update();
        }

        onIsChangingPresenceChanged: {
            update();
        }

        onConnectionStatusChanged: {
            update();
            updateMenu();
        }
    }

    function setPresence(row) {
        ktpPresence.requestedPresence = presenceModel.get(row, "presence");
    }

    function update() {
        Plasmoid.busy = ktpPresence.isChangingPresence;
        Plasmoid.icon = ktpPresence.currentPresenceIconName;
        Plasmoid.toolTipMainText = ktpPresence.currentPresenceName;
        Plasmoid.toolTipSubText = ktpPresence.presenceMessage;

        if (ktpPresence.connectionStatus == KTp.GlobalPresence.Connecting) {
            Plasmoid.toolTipSubText = i18nc("Means 'Connecting your IM accounts', it's in the applet tooltip", "Connecting...");
        } else if (ktpPresence.isChangingPresence && !ktpPresence.hasConnectionError) {
            Plasmoid.toolTipSubText = i18nc("The arg is the presence name (as is in ktp-common-internals.po, eg. Changing Presence to Away..., it's in the applet tooltip",
                                            "Changing Presence to %1...", ktpPresence.requestedPresenceName);
        }
    }

    function updateMenu() {
        // remove any already existing actions
        plasmoid.clearActions();

        for(var i=0; i<presenceModel.count; ++i) {
            plasmoid.setAction(i, presenceModel.get(i, "display"), presenceModel.get(i, "iconName"));
        }
        plasmoid.setActionSeparator("statuses");

        // application actions
        plasmoid.setAction("openIMSettings", i18n("Instant Messaging Settings..."), "configure");
        plasmoid.setAction("openContactList", i18n("Contact List..."), "telepathy-kde");
        plasmoid.setActionSeparator("applications");

        if (ktpPresence.connectionStatus < KTp.GlobalPresence.Connected) {
            return;
        }

        plasmoid.setAction("addContact", i18n("Add New Contacts..."), "list-add-user");
        plasmoid.setAction("joinChatRoom", i18n("Join Chat Room..."), "im-irc");

        if (telepathyManager.canDial) {
            plasmoid.setAction("dial", i18n("Make a Call..."), "internet-telephony");
        }
        if (telepathyManager.canSendFile) {
            plasmoid.setAction("sendFile", i18n("Send a File..."), "mail-attachment");
        }
        plasmoid.setActionSeparator("actions");
    }

    Component.onCompleted: {
        telepathyManager.addContactListFeatures();
        ktpPresence.accountManager = telepathyManager.accountManager;
    }

    function action_dial() { telepathyManager.openDialUi(); }
    function action_sendFile() { telepathyManager.openSendFileUi(); }
    function action_addContact() { telepathyManager.addContact(); }
    function action_joinChatRoom() { telepathyManager.joinChatRoom(); }
    function action_openContactList() { telepathyManager.toggleContactList(); }
    function action_openIMSettings() {telepathyManager.showSettingsKCM(); }

    function actionTriggered(actionName) {
        var number = parseInt(actionName)
        if (number !== null) {
            root.setPresence(number)
        } else {
            console.log("Unknown action", actionName);
        }
    }
}
