(function() {
  var AddDialog, BufferedProcess, CompositeDisposable, CopyDialog, Directory, DirectoryView, Emitter, IgnoredNames, MoveDialog, RootDragAndDrop, TREE_VIEW_URI, TreeView, _, fs, getFullExtension, getStyleObject, nextId, path, ref, ref1, repoForPath, shell, toggleConfig,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  shell = require('electron').shell;

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  ref1 = require("./helpers"), repoForPath = ref1.repoForPath, getStyleObject = ref1.getStyleObject, getFullExtension = ref1.getFullExtension;

  fs = require('fs-plus');

  AddDialog = require('./add-dialog');

  MoveDialog = require('./move-dialog');

  CopyDialog = require('./copy-dialog');

  IgnoredNames = null;

  Directory = require('./directory');

  DirectoryView = require('./directory-view');

  RootDragAndDrop = require('./root-drag-and-drop');

  TREE_VIEW_URI = 'atom://tree-view';

  toggleConfig = function(keyPath) {
    return atom.config.set(keyPath, !atom.config.get(keyPath));
  };

  nextId = 1;

  module.exports = TreeView = (function() {
    function TreeView(state) {
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragEnter = bind(this.onDragEnter, this);
      this.onStylesheetsChanged = bind(this.onStylesheetsChanged, this);
      this.moveConflictingEntry = bind(this.moveConflictingEntry, this);
      var j, len, observer, ref2, ref3, selectedPath;
      this.id = nextId++;
      this.element = document.createElement('div');
      this.element.classList.add('tool-panel', 'tree-view');
      this.element.tabIndex = -1;
      this.list = document.createElement('ol');
      this.list.classList.add('tree-view-root', 'full-menu', 'list-tree', 'has-collapsable-children', 'focusable-panel');
      this.element.appendChild(this.list);
      this.disposables = new CompositeDisposable;
      this.emitter = new Emitter;
      this.roots = [];
      this.selectedPath = null;
      this.selectOnMouseUp = null;
      this.lastFocusedEntry = null;
      this.ignoredPatterns = [];
      this.useSyncFS = false;
      this.currentlyOpening = new Map;
      this.editorsToMove = [];
      this.editorsToDestroy = [];
      this.dragEventCounts = new WeakMap;
      this.rootDragAndDrop = new RootDragAndDrop(this);
      this.handleEvents();
      process.nextTick((function(_this) {
        return function() {
          var onStylesheetsChanged;
          _this.onStylesheetsChanged();
          onStylesheetsChanged = _.debounce(_this.onStylesheetsChanged, 100);
          _this.disposables.add(atom.styles.onDidAddStyleElement(onStylesheetsChanged));
          _this.disposables.add(atom.styles.onDidRemoveStyleElement(onStylesheetsChanged));
          return _this.disposables.add(atom.styles.onDidUpdateStyleElement(onStylesheetsChanged));
        };
      })(this));
      this.updateRoots(state.directoryExpansionStates);
      if (((ref2 = state.selectedPaths) != null ? ref2.length : void 0) > 0) {
        ref3 = state.selectedPaths;
        for (j = 0, len = ref3.length; j < len; j++) {
          selectedPath = ref3[j];
          this.selectMultipleEntries(this.entryForPath(selectedPath));
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      if ((state.scrollTop != null) || (state.scrollLeft != null)) {
        observer = new IntersectionObserver((function(_this) {
          return function() {
            if (_this.isVisible()) {
              _this.element.scrollTop = state.scrollTop;
              _this.element.scrollLeft = state.scrollLeft;
              return observer.disconnect();
            }
          };
        })(this));
        observer.observe(this.element);
      }
      if (state.width > 0) {
        this.element.style.width = state.width + "px";
      }
      this.disposables.add(this.onWillMoveEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, initialPath, k, l, len1, len2, newPath, results, results1;
          initialPath = arg.initialPath, newPath = arg.newPath;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(initialPath)) {
            initialPath += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if (filePath != null ? filePath.startsWith(initialPath) : void 0) {
                results.push(_this.editorsToMove.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === initialPath) {
                results1.push(_this.editorsToMove.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryMoved((function(_this) {
        return function(arg) {
          var editor, filePath, index, initialPath, k, len1, newPath, ref4, results;
          initialPath = arg.initialPath, newPath = arg.newPath;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            filePath = editor.getPath();
            index = _this.editorsToMove.indexOf(filePath);
            if (index !== -1) {
              editor.getBuffer().setPath(filePath.replace(initialPath, newPath));
              results.push(_this.editorsToMove.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onMoveEntryFailed((function(_this) {
        return function(arg) {
          var index, initialPath, newPath;
          initialPath = arg.initialPath, newPath = arg.newPath;
          index = _this.editorsToMove.indexOf(initialPath);
          if (index !== -1) {
            return _this.editorsToMove.splice(index, 1);
          }
        };
      })(this)));
      this.disposables.add(this.onWillDeleteEntry((function(_this) {
        return function(arg) {
          var editor, editors, filePath, k, l, len1, len2, pathToDelete, results, results1;
          pathToDelete = arg.pathToDelete;
          editors = atom.workspace.getTextEditors();
          if (fs.isDirectorySync(pathToDelete)) {
            pathToDelete += path.sep;
            results = [];
            for (k = 0, len1 = editors.length; k < len1; k++) {
              editor = editors[k];
              filePath = editor.getPath();
              if ((filePath != null ? filePath.startsWith(pathToDelete) : void 0) && !editor.isModified()) {
                results.push(_this.editorsToDestroy.push(filePath));
              } else {
                results.push(void 0);
              }
            }
            return results;
          } else {
            results1 = [];
            for (l = 0, len2 = editors.length; l < len2; l++) {
              editor = editors[l];
              filePath = editor.getPath();
              if (filePath === pathToDelete && !editor.isModified()) {
                results1.push(_this.editorsToDestroy.push(filePath));
              } else {
                results1.push(void 0);
              }
            }
            return results1;
          }
        };
      })(this)));
      this.disposables.add(this.onEntryDeleted((function(_this) {
        return function(arg) {
          var editor, index, k, len1, pathToDelete, ref4, results;
          pathToDelete = arg.pathToDelete;
          ref4 = atom.workspace.getTextEditors();
          results = [];
          for (k = 0, len1 = ref4.length; k < len1; k++) {
            editor = ref4[k];
            index = _this.editorsToDestroy.indexOf(editor.getPath());
            if (index !== -1) {
              editor.destroy();
              results.push(_this.editorsToDestroy.splice(index, 1));
            } else {
              results.push(void 0);
            }
          }
          return results;
        };
      })(this)));
      this.disposables.add(this.onDeleteEntryFailed((function(_this) {
        return function(arg) {
          var index, pathToDelete;
          pathToDelete = arg.pathToDelete;
          index = _this.editorsToDestroy.indexOf(pathToDelete);
          if (index !== -1) {
            return _this.editorsToDestroy.splice(index, 1);
          }
        };
      })(this)));
    }

    TreeView.prototype.serialize = function() {
      return {
        directoryExpansionStates: new (function(roots) {
          var j, len, root;
          for (j = 0, len = roots.length; j < len; j++) {
            root = roots[j];
            this[root.directory.path] = root.directory.serializeExpansionState();
          }
          return this;
        })(this.roots),
        deserializer: 'TreeView',
        selectedPaths: Array.from(this.getSelectedEntries(), function(entry) {
          return entry.getPath();
        }),
        scrollLeft: this.element.scrollLeft,
        scrollTop: this.element.scrollTop,
        width: parseInt(this.element.style.width || 0)
      };
    };

    TreeView.prototype.destroy = function() {
      var j, len, ref2, root;
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        root.directory.destroy();
      }
      this.disposables.dispose();
      this.rootDragAndDrop.dispose();
      return this.emitter.emit('did-destroy');
    };

    TreeView.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TreeView.prototype.getTitle = function() {
      return "Project";
    };

    TreeView.prototype.getURI = function() {
      return TREE_VIEW_URI;
    };

    TreeView.prototype.getPreferredLocation = function() {
      if (atom.config.get('tree-view.showOnRightSide')) {
        return 'right';
      } else {
        return 'left';
      }
    };

    TreeView.prototype.getAllowedLocations = function() {
      return ["left", "right"];
    };

    TreeView.prototype.isPermanentDockItem = function() {
      return true;
    };

    TreeView.prototype.getPreferredWidth = function() {
      var result;
      this.list.style.width = 'min-content';
      result = this.list.offsetWidth;
      this.list.style.width = '';
      return result;
    };

    TreeView.prototype.onDirectoryCreated = function(callback) {
      return this.emitter.on('directory-created', callback);
    };

    TreeView.prototype.onEntryCopied = function(callback) {
      return this.emitter.on('entry-copied', callback);
    };

    TreeView.prototype.onWillDeleteEntry = function(callback) {
      return this.emitter.on('will-delete-entry', callback);
    };

    TreeView.prototype.onEntryDeleted = function(callback) {
      return this.emitter.on('entry-deleted', callback);
    };

    TreeView.prototype.onDeleteEntryFailed = function(callback) {
      return this.emitter.on('delete-entry-failed', callback);
    };

    TreeView.prototype.onWillMoveEntry = function(callback) {
      return this.emitter.on('will-move-entry', callback);
    };

    TreeView.prototype.onEntryMoved = function(callback) {
      return this.emitter.on('entry-moved', callback);
    };

    TreeView.prototype.onMoveEntryFailed = function(callback) {
      return this.emitter.on('move-entry-failed', callback);
    };

    TreeView.prototype.onFileCreated = function(callback) {
      return this.emitter.on('file-created', callback);
    };

    TreeView.prototype.handleEvents = function() {
      this.element.addEventListener('click', (function(_this) {
        return function(e) {
          if (e.target.classList.contains('entries')) {
            return;
          }
          if (!(e.shiftKey || e.metaKey || e.ctrlKey)) {
            return _this.entryClicked(e);
          }
        };
      })(this));
      this.element.addEventListener('mousedown', (function(_this) {
        return function(e) {
          return _this.onMouseDown(e);
        };
      })(this));
      this.element.addEventListener('mouseup', (function(_this) {
        return function(e) {
          return _this.onMouseUp(e);
        };
      })(this));
      this.element.addEventListener('dragstart', (function(_this) {
        return function(e) {
          return _this.onDragStart(e);
        };
      })(this));
      this.element.addEventListener('dragenter', (function(_this) {
        return function(e) {
          return _this.onDragEnter(e);
        };
      })(this));
      this.element.addEventListener('dragleave', (function(_this) {
        return function(e) {
          return _this.onDragLeave(e);
        };
      })(this));
      this.element.addEventListener('dragover', (function(_this) {
        return function(e) {
          return _this.onDragOver(e);
        };
      })(this));
      this.element.addEventListener('drop', (function(_this) {
        return function(e) {
          return _this.onDrop(e);
        };
      })(this));
      atom.commands.add(this.element, {
        'core:move-up': (function(_this) {
          return function(e) {
            return _this.moveUp(e);
          };
        })(this),
        'core:move-down': (function(_this) {
          return function(e) {
            return _this.moveDown(e);
          };
        })(this),
        'core:page-up': (function(_this) {
          return function() {
            return _this.pageUp();
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            return _this.pageDown();
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            return _this.scrollToTop();
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            return _this.scrollToBottom();
          };
        })(this),
        'tree-view:expand-item': (function(_this) {
          return function() {
            return _this.openSelectedEntry({
              pending: true
            }, true);
          };
        })(this),
        'tree-view:recursive-expand-directory': (function(_this) {
          return function() {
            return _this.expandDirectory(true);
          };
        })(this),
        'tree-view:collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory();
          };
        })(this),
        'tree-view:recursive-collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory(true);
          };
        })(this),
        'tree-view:collapse-all': (function(_this) {
          return function() {
            return _this.collapseDirectory(true, true);
          };
        })(this),
        'tree-view:open-selected-entry': (function(_this) {
          return function() {
            return _this.openSelectedEntry();
          };
        })(this),
        'tree-view:open-selected-entry-right': (function(_this) {
          return function() {
            return _this.openSelectedEntryRight();
          };
        })(this),
        'tree-view:open-selected-entry-left': (function(_this) {
          return function() {
            return _this.openSelectedEntryLeft();
          };
        })(this),
        'tree-view:open-selected-entry-up': (function(_this) {
          return function() {
            return _this.openSelectedEntryUp();
          };
        })(this),
        'tree-view:open-selected-entry-down': (function(_this) {
          return function() {
            return _this.openSelectedEntryDown();
          };
        })(this),
        'tree-view:move': (function(_this) {
          return function() {
            return _this.moveSelectedEntry();
          };
        })(this),
        'tree-view:copy': (function(_this) {
          return function() {
            return _this.copySelectedEntries();
          };
        })(this),
        'tree-view:cut': (function(_this) {
          return function() {
            return _this.cutSelectedEntries();
          };
        })(this),
        'tree-view:paste': (function(_this) {
          return function() {
            return _this.pasteEntries();
          };
        })(this),
        'tree-view:copy-full-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(false);
          };
        })(this),
        'tree-view:show-in-file-manager': (function(_this) {
          return function() {
            return _this.showSelectedEntryInFileManager();
          };
        })(this),
        'tree-view:open-in-new-window': (function(_this) {
          return function() {
            return _this.openSelectedEntryInNewWindow();
          };
        })(this),
        'tree-view:copy-project-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(true);
          };
        })(this),
        'tree-view:unfocus': (function(_this) {
          return function() {
            return _this.unfocus();
          };
        })(this),
        'tree-view:toggle-vcs-ignored-files': function() {
          return toggleConfig('tree-view.hideVcsIgnoredFiles');
        },
        'tree-view:toggle-ignored-names': function() {
          return toggleConfig('tree-view.hideIgnoredNames');
        },
        'tree-view:remove-project-folder': (function(_this) {
          return function(e) {
            return _this.removeProjectFolder(e);
          };
        })(this)
      });
      [0, 1, 2, 3, 4, 5, 6, 7, 8].forEach((function(_this) {
        return function(index) {
          return atom.commands.add(_this.element, "tree-view:open-selected-entry-in-pane-" + (index + 1), function() {
            return _this.openSelectedEntryInPane(index);
          });
        };
      })(this));
      this.disposables.add(atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          _this.selectActiveFile();
          if (atom.config.get('tree-view.autoReveal')) {
            return _this.revealActiveFile({
              show: false,
              focus: false
            });
          }
        };
      })(this)));
      this.disposables.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideVcsIgnoredFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideIgnoredNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('core.ignoredNames', (function(_this) {
        return function() {
          if (atom.config.get('tree-view.hideIgnoredNames')) {
            return _this.updateRoots();
          }
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.sortFoldersBeforeFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      return this.disposables.add(atom.config.onDidChange('tree-view.squashDirectoryNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
    };

    TreeView.prototype.toggle = function() {
      return atom.workspace.toggle(this);
    };

    TreeView.prototype.show = function(focus) {
      return atom.workspace.open(this, {
        searchAllPanes: true,
        activatePane: false,
        activateItem: false
      }).then((function(_this) {
        return function() {
          atom.workspace.paneContainerForURI(_this.getURI()).show();
          if (focus) {
            return _this.focus();
          }
        };
      })(this));
    };

    TreeView.prototype.hide = function() {
      return atom.workspace.hide(this);
    };

    TreeView.prototype.focus = function() {
      return this.element.focus();
    };

    TreeView.prototype.unfocus = function() {
      return atom.workspace.getCenter().activate();
    };

    TreeView.prototype.hasFocus = function() {
      return document.activeElement === this.element;
    };

    TreeView.prototype.toggleFocus = function() {
      if (this.hasFocus()) {
        return this.unfocus();
      } else {
        return this.show(true);
      }
    };

    TreeView.prototype.entryClicked = function(e) {
      var entry, isRecursive;
      if (entry = e.target.closest('.entry')) {
        isRecursive = e.altKey || false;
        this.selectEntry(entry);
        if (entry.classList.contains('directory')) {
          return entry.toggleExpansion(isRecursive);
        } else if (entry.classList.contains('file')) {
          return this.fileViewEntryClicked(e);
        }
      }
    };

    TreeView.prototype.fileViewEntryClicked = function(e) {
      var alwaysOpenExisting, detail, filePath, openPromise, ref2;
      filePath = e.target.closest('.entry').getPath();
      detail = (ref2 = e.detail) != null ? ref2 : 1;
      alwaysOpenExisting = atom.config.get('tree-view.alwaysOpenExisting');
      if (detail === 1) {
        if (atom.config.get('core.allowPendingPaneItems')) {
          openPromise = atom.workspace.open(filePath, {
            pending: true,
            activatePane: false,
            searchAllPanes: alwaysOpenExisting
          });
          this.currentlyOpening.set(filePath, openPromise);
          return openPromise.then((function(_this) {
            return function() {
              return _this.currentlyOpening["delete"](filePath);
            };
          })(this));
        }
      } else if (detail === 2) {
        return this.openAfterPromise(filePath, {
          searchAllPanes: alwaysOpenExisting
        });
      }
    };

    TreeView.prototype.openAfterPromise = function(uri, options) {
      var promise;
      if (promise = this.currentlyOpening.get(uri)) {
        return promise.then(function() {
          return atom.workspace.open(uri, options);
        });
      } else {
        return atom.workspace.open(uri, options);
      }
    };

    TreeView.prototype.updateRoots = function(expansionStates) {
      var directory, j, k, key, len, len1, oldExpansionStates, projectPath, ref2, results, root, selectedPath, selectedPaths, stats;
      if (expansionStates == null) {
        expansionStates = {};
      }
      selectedPaths = this.selectedPaths();
      oldExpansionStates = {};
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        oldExpansionStates[root.directory.path] = root.directory.serializeExpansionState();
        root.directory.destroy();
        root.remove();
      }
      if (IgnoredNames == null) {
        IgnoredNames = require('./ignored-names');
      }
      this.roots = (function() {
        var k, l, len1, len2, ref3, ref4, ref5, ref6, results;
        ref3 = atom.project.getPaths();
        results = [];
        for (k = 0, len1 = ref3.length; k < len1; k++) {
          projectPath = ref3[k];
          stats = fs.lstatSyncNoException(projectPath);
          if (!stats) {
            continue;
          }
          stats = _.pick.apply(_, [stats].concat(slice.call(_.keys(stats))));
          ref4 = ["atime", "birthtime", "ctime", "mtime"];
          for (l = 0, len2 = ref4.length; l < len2; l++) {
            key = ref4[l];
            stats[key] = stats[key].getTime();
          }
          directory = new Directory({
            name: path.basename(projectPath),
            fullPath: projectPath,
            symlink: false,
            isRoot: true,
            expansionState: (ref5 = (ref6 = expansionStates[projectPath]) != null ? ref6 : oldExpansionStates[projectPath]) != null ? ref5 : {
              isExpanded: true
            },
            ignoredNames: new IgnoredNames(),
            useSyncFS: this.useSyncFS,
            stats: stats
          });
          root = new DirectoryView(directory).element;
          this.list.appendChild(root);
          results.push(root);
        }
        return results;
      }).call(this);
      results = [];
      for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
        selectedPath = selectedPaths[k];
        results.push(this.selectMultipleEntries(this.entryForPath(selectedPath)));
      }
      return results;
    };

    TreeView.prototype.getActivePath = function() {
      var ref2;
      return (ref2 = atom.workspace.getCenter().getActivePaneItem()) != null ? typeof ref2.getPath === "function" ? ref2.getPath() : void 0 : void 0;
    };

    TreeView.prototype.selectActiveFile = function() {
      var activeFilePath;
      activeFilePath = this.getActivePath();
      if (this.entryForPath(activeFilePath)) {
        return this.selectEntryForPath(activeFilePath);
      } else {
        return this.deselect();
      }
    };

    TreeView.prototype.revealActiveFile = function(options) {
      var focus, promise, show;
      if (options == null) {
        options = {};
      }
      if (!atom.project.getPaths().length) {
        return Promise.resolve();
      }
      show = options.show, focus = options.focus;
      if (focus == null) {
        focus = atom.config.get('tree-view.focusOnReveal');
      }
      promise = show || focus ? this.show(focus) : Promise.resolve();
      return promise.then((function(_this) {
        return function() {
          var activeFilePath, activePathComponents, currentPath, entry, j, len, pathComponent, ref2, relativePath, results, rootPath;
          if (!(activeFilePath = _this.getActivePath())) {
            return;
          }
          ref2 = atom.project.relativizePath(activeFilePath), rootPath = ref2[0], relativePath = ref2[1];
          if (rootPath == null) {
            return;
          }
          activePathComponents = relativePath.split(path.sep);
          activePathComponents.unshift(rootPath.substr(rootPath.lastIndexOf(path.sep) + 1));
          currentPath = rootPath.substr(0, rootPath.lastIndexOf(path.sep));
          results = [];
          for (j = 0, len = activePathComponents.length; j < len; j++) {
            pathComponent = activePathComponents[j];
            currentPath += path.sep + pathComponent;
            entry = _this.entryForPath(currentPath);
            if (entry.classList.contains('directory')) {
              results.push(entry.expand());
            } else {
              _this.selectEntry(entry);
              results.push(_this.scrollToEntry(entry));
            }
          }
          return results;
        };
      })(this));
    };

    TreeView.prototype.copySelectedEntryPath = function(relativePath) {
      var pathToCopy;
      if (relativePath == null) {
        relativePath = false;
      }
      if (pathToCopy = this.selectedPath) {
        if (relativePath) {
          pathToCopy = atom.project.relativize(pathToCopy);
        }
        return atom.clipboard.write(pathToCopy);
      }
    };

    TreeView.prototype.entryForPath = function(entryPath) {
      var bestMatchEntry, bestMatchLength, entry, entryLength, j, len, ref2, ref3;
      bestMatchEntry = null;
      bestMatchLength = 0;
      ref2 = this.list.querySelectorAll('.entry');
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        if (entry.isPathEqual(entryPath)) {
          return entry;
        }
        entryLength = entry.getPath().length;
        if (((ref3 = entry.directory) != null ? ref3.contains(entryPath) : void 0) && entryLength > bestMatchLength) {
          bestMatchEntry = entry;
          bestMatchLength = entryLength;
        }
      }
      return bestMatchEntry;
    };

    TreeView.prototype.selectEntryForPath = function(entryPath) {
      return this.selectEntry(this.entryForPath(entryPath));
    };

    TreeView.prototype.moveDown = function(event) {
      var nextEntry, selectedEntry;
      if (event != null) {
        event.stopImmediatePropagation();
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (selectedEntry.classList.contains('directory')) {
          if (this.selectEntry(selectedEntry.entries.children[0])) {
            this.scrollToEntry(this.selectedEntry(), false);
            return;
          }
        }
        if (nextEntry = this.nextEntry(selectedEntry)) {
          this.selectEntry(nextEntry);
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.moveUp = function(event) {
      var entries, previousEntry, selectedEntry;
      event.stopImmediatePropagation();
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (previousEntry = this.previousEntry(selectedEntry)) {
          this.selectEntry(previousEntry);
        } else {
          this.selectEntry(selectedEntry.parentElement.closest('.directory'));
        }
      } else {
        entries = this.list.querySelectorAll('.entry');
        this.selectEntry(entries[entries.length - 1]);
      }
      return this.scrollToEntry(this.selectedEntry(), false);
    };

    TreeView.prototype.nextEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        if (currentEntry.nextSibling != null) {
          currentEntry = currentEntry.nextSibling;
          if (currentEntry.matches('.entry')) {
            return currentEntry;
          }
        } else {
          currentEntry = currentEntry.parentElement.closest('.directory');
        }
      }
      return null;
    };

    TreeView.prototype.previousEntry = function(entry) {
      var entries, previousEntry;
      previousEntry = entry.previousSibling;
      while ((previousEntry != null) && !previousEntry.matches('.entry')) {
        previousEntry = previousEntry.previousSibling;
      }
      if (previousEntry == null) {
        return null;
      }
      if (previousEntry.matches('.directory.expanded')) {
        entries = previousEntry.querySelectorAll('.entry');
        if (entries.length > 0) {
          return entries[entries.length - 1];
        }
      }
      return previousEntry;
    };

    TreeView.prototype.expandDirectory = function(isRecursive) {
      var directory, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      directory = selectedEntry.closest('.directory');
      if (isRecursive === false && directory.isExpanded) {
        if (directory.directory.getEntries().length > 0) {
          return this.moveDown();
        }
      } else {
        return directory.expand(isRecursive);
      }
    };

    TreeView.prototype.collapseDirectory = function(isRecursive, allDirectories) {
      var directory, j, len, ref2, root, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      if (allDirectories == null) {
        allDirectories = false;
      }
      if (allDirectories) {
        ref2 = this.roots;
        for (j = 0, len = ref2.length; j < len; j++) {
          root = ref2[j];
          root.collapse(true);
        }
        return;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (directory = selectedEntry.closest('.expanded.directory')) {
        directory.collapse(isRecursive);
        return this.selectEntry(directory);
      }
    };

    TreeView.prototype.openSelectedEntry = function(options, expandDirectory) {
      var selectedEntry;
      if (options == null) {
        options = {};
      }
      if (expandDirectory == null) {
        expandDirectory = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (selectedEntry.classList.contains('directory')) {
        if (expandDirectory) {
          return this.expandDirectory(false);
        } else {
          return selectedEntry.toggleExpansion();
        }
      } else if (selectedEntry.classList.contains('file')) {
        if (atom.config.get('tree-view.alwaysOpenExisting')) {
          options = Object.assign({
            searchAllPanes: true
          }, options);
        }
        return this.openAfterPromise(selectedEntry.getPath(), options);
      }
    };

    TreeView.prototype.openSelectedEntrySplit = function(orientation, side) {
      var pane, selectedEntry, split;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getActivePane();
      if (pane && selectedEntry.classList.contains('file')) {
        if (atom.workspace.getCenter().getActivePaneItem()) {
          split = pane.split(orientation, side);
          return atom.workspace.openURIInPane(selectedEntry.getPath(), split);
        } else {
          return this.openSelectedEntry(true);
        }
      }
    };

    TreeView.prototype.openSelectedEntryRight = function() {
      return this.openSelectedEntrySplit('horizontal', 'after');
    };

    TreeView.prototype.openSelectedEntryLeft = function() {
      return this.openSelectedEntrySplit('horizontal', 'before');
    };

    TreeView.prototype.openSelectedEntryUp = function() {
      return this.openSelectedEntrySplit('vertical', 'before');
    };

    TreeView.prototype.openSelectedEntryDown = function() {
      return this.openSelectedEntrySplit('vertical', 'after');
    };

    TreeView.prototype.openSelectedEntryInPane = function(index) {
      var pane, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getPanes()[index];
      if (pane && selectedEntry.classList.contains('file')) {
        return atom.workspace.openURIInPane(selectedEntry.getPath(), pane);
      }
    };

    TreeView.prototype.moveSelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if ((entry == null) || indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry.getPath();
      } else {
        oldPath = this.getActivePath();
      }
      if (oldPath) {
        dialog = new MoveDialog(oldPath, {
          willMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('will-move-entry', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this),
          onMoveFailed: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('move-entry-failed', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this)
        });
        return dialog.attach();
      }
    };

    TreeView.prototype.showSelectedEntryInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in " + (this.getFileManagerName()));
      }
    };

    TreeView.prototype.showCurrentFileInFileManager = function() {
      var filePath, ref2;
      if (!(filePath = (ref2 = atom.workspace.getCenter().getActiveTextEditor()) != null ? ref2.getPath() : void 0)) {
        return;
      }
      if (!shell.showItemInFolder(filePath)) {
        return atom.notifications.addWarning("Unable to show " + filePath + " in " + (this.getFileManagerName()));
      }
    };

    TreeView.prototype.getFileManagerName = function() {
      switch (process.platform) {
        case 'darwin':
          return 'Finder';
        case 'win32':
          return 'Explorer';
        default:
          return 'File Manager';
      }
    };

    TreeView.prototype.openSelectedEntryInNewWindow = function() {
      var pathToOpen, ref2;
      if (pathToOpen = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0) {
        return atom.open({
          pathsToOpen: [pathToOpen],
          newWindow: true
        });
      }
    };

    TreeView.prototype.copySelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if (indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry != null ? entry.getPath() : void 0;
      } else {
        oldPath = this.getActivePath();
      }
      if (!oldPath) {
        return;
      }
      dialog = new CopyDialog(oldPath, {
        onCopy: (function(_this) {
          return function(arg) {
            var initialPath, newPath;
            initialPath = arg.initialPath, newPath = arg.newPath;
            return _this.emitter.emit('entry-copied', {
              initialPath: initialPath,
              newPath: newPath
            });
          };
        })(this)
      });
      return dialog.attach();
    };

    TreeView.prototype.removeSelectedEntries = function() {
      var activePath, j, len, ref2, ref3, root, selectedEntries, selectedPaths;
      if (this.hasFocus()) {
        selectedPaths = this.selectedPaths();
        selectedEntries = this.getSelectedEntries();
      } else if (activePath = this.getActivePath()) {
        selectedPaths = [activePath];
        selectedEntries = [this.entryForPath(activePath)];
      }
      if (!((selectedPaths != null ? selectedPaths.length : void 0) > 0)) {
        return;
      }
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        if (ref3 = root.getPath(), indexOf.call(selectedPaths, ref3) >= 0) {
          atom.confirm({
            message: "The root directory '" + root.directory.name + "' can't be removed.",
            buttons: ['OK']
          }, function() {});
          return;
        }
      }
      return atom.confirm({
        message: "Are you sure you want to delete the selected " + (selectedPaths.length > 1 ? 'items' : 'item') + "?",
        detailedMessage: "You are deleting:\n" + (selectedPaths.join('\n')),
        buttons: ['Move to Trash', 'Cancel']
      }, (function(_this) {
        return function(response) {
          var failedDeletions, firstSelectedEntry, k, len1, repo, selectedPath;
          if (response === 0) {
            failedDeletions = [];
            for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
              selectedPath = selectedPaths[k];
              if (!fs.existsSync(selectedPath)) {
                continue;
              }
              _this.emitter.emit('will-delete-entry', {
                pathToDelete: selectedPath
              });
              if (shell.moveItemToTrash(selectedPath)) {
                _this.emitter.emit('entry-deleted', {
                  pathToDelete: selectedPath
                });
              } else {
                _this.emitter.emit('delete-entry-failed', {
                  pathToDelete: selectedPath
                });
                failedDeletions.push(selectedPath);
              }
              if (repo = repoForPath(selectedPath)) {
                repo.getPathStatus(selectedPath);
              }
            }
            if (failedDeletions.length > 0) {
              atom.notifications.addError(_this.formatTrashFailureMessage(failedDeletions), {
                description: _this.formatTrashEnabledMessage(),
                detail: "" + (failedDeletions.join('\n')),
                dismissable: true
              });
            }
            if (firstSelectedEntry = selectedEntries[0]) {
              _this.selectEntry(firstSelectedEntry.closest('.directory:not(.selected)'));
            }
            if (atom.config.get('tree-view.squashDirectoryNames')) {
              return _this.updateRoots();
            }
          }
        };
      })(this));
    };

    TreeView.prototype.formatTrashFailureMessage = function(failedDeletions) {
      var fileText;
      fileText = failedDeletions.length > 1 ? 'files' : 'file';
      return "The following " + fileText + " couldn't be moved to the trash.";
    };

    TreeView.prototype.formatTrashEnabledMessage = function() {
      switch (process.platform) {
        case 'linux':
          return 'Is `gvfs-trash` installed?';
        case 'darwin':
          return 'Is Trash enabled on the volume where the files are stored?';
        case 'win32':
          return 'Is there a Recycle Bin on the drive where the files are stored?';
      }
    };

    TreeView.prototype.copySelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:cutPath');
      return window.localStorage['tree-view:copyPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.cutSelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:copyPath');
      return window.localStorage['tree-view:cutPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.pasteEntries = function() {
      var basePath, catchAndShowFileErrors, copiedPaths, cutPaths, error, extension, fileCounter, filePath, initialPath, initialPathIsDirectory, initialPaths, j, len, newPath, originalNewPath, realBasePath, realInitialPath, ref2, results, selectedEntry;
      selectedEntry = this.selectedEntry();
      cutPaths = window.localStorage['tree-view:cutPath'] ? JSON.parse(window.localStorage['tree-view:cutPath']) : null;
      copiedPaths = window.localStorage['tree-view:copyPath'] ? JSON.parse(window.localStorage['tree-view:copyPath']) : null;
      initialPaths = copiedPaths || cutPaths;
      catchAndShowFileErrors = function(operation) {
        var error;
        try {
          return operation();
        } catch (error1) {
          error = error1;
          return atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
            detail: error.message
          });
        }
      };
      ref2 = initialPaths != null ? initialPaths : [];
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        initialPath = ref2[j];
        initialPathIsDirectory = fs.isDirectorySync(initialPath);
        if (selectedEntry && initialPath && fs.existsSync(initialPath)) {
          basePath = selectedEntry.getPath();
          if (selectedEntry.classList.contains('file')) {
            basePath = path.dirname(basePath);
          }
          newPath = path.join(basePath, path.basename(initialPath));
          realBasePath = fs.realpathSync(basePath) + path.sep;
          realInitialPath = fs.realpathSync(initialPath) + path.sep;
          if (initialPathIsDirectory && realBasePath.startsWith(realInitialPath)) {
            if (!fs.isSymbolicLinkSync(initialPath)) {
              atom.notifications.addWarning('Cannot paste a folder into itself');
              continue;
            }
          }
          if (copiedPaths) {
            fileCounter = 0;
            originalNewPath = newPath;
            while (fs.existsSync(newPath)) {
              if (initialPathIsDirectory) {
                newPath = "" + originalNewPath + fileCounter;
              } else {
                extension = getFullExtension(originalNewPath);
                filePath = path.join(path.dirname(originalNewPath), path.basename(originalNewPath, extension));
                newPath = "" + filePath + fileCounter + extension;
              }
              fileCounter += 1;
            }
            if (initialPathIsDirectory) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.copySync(initialPath, newPath);
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.writeFileSync(newPath, fs.readFileSync(initialPath));
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            }
          } else if (cutPaths) {
            try {
              this.emitter.emit('will-move-entry', {
                initialPath: initialPath,
                newPath: newPath
              });
              fs.moveSync(initialPath, newPath);
              results.push(this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              }));
            } catch (error1) {
              error = error1;
              this.emitter.emit('move-entry-failed', {
                initialPath: initialPath,
                newPath: newPath
              });
              results.push(atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
                detail: error.message
              }));
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TreeView.prototype.add = function(isCreatingFile) {
      var dialog, ref2, ref3, selectedEntry, selectedPath;
      selectedEntry = (ref2 = this.selectedEntry()) != null ? ref2 : this.roots[0];
      selectedPath = (ref3 = selectedEntry != null ? selectedEntry.getPath() : void 0) != null ? ref3 : '';
      dialog = new AddDialog(selectedPath, isCreatingFile);
      dialog.onDidCreateDirectory((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          _this.selectEntryForPath(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('directory-created', {
            path: createdPath
          });
        };
      })(this));
      dialog.onDidCreateFile((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          atom.workspace.open(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('file-created', {
            path: createdPath
          });
        };
      })(this));
      return dialog.attach();
    };

    TreeView.prototype.removeProjectFolder = function(e) {
      var pathToRemove, ref2, ref3, ref4, ref5, ref6, ref7, ref8, ref9;
      pathToRemove = (ref2 = e.target.closest(".project-root > .header")) != null ? (ref3 = ref2.querySelector(".name")) != null ? ref3.dataset.path : void 0 : void 0;
      if (pathToRemove == null) {
        pathToRemove = (ref4 = this.selectedEntry()) != null ? (ref5 = ref4.closest(".project-root")) != null ? (ref6 = ref5.querySelector(".header")) != null ? (ref7 = ref6.querySelector(".name")) != null ? ref7.dataset.path : void 0 : void 0 : void 0 : void 0;
      }
      if (this.roots.length === 1) {
        if (pathToRemove == null) {
          pathToRemove = (ref8 = this.roots[0].querySelector(".header")) != null ? (ref9 = ref8.querySelector(".name")) != null ? ref9.dataset.path : void 0 : void 0;
        }
      }
      if (pathToRemove != null) {
        return atom.project.removePath(pathToRemove);
      }
    };

    TreeView.prototype.selectedEntry = function() {
      return this.list.querySelector('.selected');
    };

    TreeView.prototype.selectEntry = function(entry) {
      var selectedEntries;
      if (entry == null) {
        return;
      }
      this.selectedPath = entry.getPath();
      this.lastFocusedEntry = entry;
      selectedEntries = this.getSelectedEntries();
      if (selectedEntries.length > 1 || selectedEntries[0] !== entry) {
        this.deselect(selectedEntries);
        entry.classList.add('selected');
      }
      return entry;
    };

    TreeView.prototype.getSelectedEntries = function() {
      return this.list.querySelectorAll('.selected');
    };

    TreeView.prototype.deselect = function(elementsToDeselect) {
      var j, len, selected;
      if (elementsToDeselect == null) {
        elementsToDeselect = this.getSelectedEntries();
      }
      for (j = 0, len = elementsToDeselect.length; j < len; j++) {
        selected = elementsToDeselect[j];
        selected.classList.remove('selected');
      }
      return void 0;
    };

    TreeView.prototype.scrollTop = function(top) {
      if (top != null) {
        return this.element.scrollTop = top;
      } else {
        return this.element.scrollTop;
      }
    };

    TreeView.prototype.scrollBottom = function(bottom) {
      if (bottom != null) {
        return this.element.scrollTop = bottom - this.element.offsetHeight;
      } else {
        return this.element.scrollTop + this.element.offsetHeight;
      }
    };

    TreeView.prototype.scrollToEntry = function(entry, center) {
      var element;
      if (center == null) {
        center = true;
      }
      element = (entry != null ? entry.classList.contains('directory') : void 0) ? entry.header : entry;
      return element != null ? element.scrollIntoViewIfNeeded(center) : void 0;
    };

    TreeView.prototype.scrollToBottom = function() {
      var lastEntry;
      if (lastEntry = _.last(this.list.querySelectorAll('.entry'))) {
        this.selectEntry(lastEntry);
        return this.scrollToEntry(lastEntry);
      }
    };

    TreeView.prototype.scrollToTop = function() {
      if (this.roots[0] != null) {
        this.selectEntry(this.roots[0]);
      }
      return this.element.scrollTop = 0;
    };

    TreeView.prototype.pageUp = function() {
      return this.element.scrollTop -= this.element.offsetHeight;
    };

    TreeView.prototype.pageDown = function() {
      return this.element.scrollTop += this.element.offsetHeight;
    };

    TreeView.prototype.moveEntry = function(initialPath, newDirectoryPath) {
      var entryName, error, newPath, realInitialPath, realNewDirectoryPath, repo;
      if (initialPath === newDirectoryPath) {
        return;
      }
      realNewDirectoryPath = fs.realpathSync(newDirectoryPath) + path.sep;
      realInitialPath = fs.realpathSync(initialPath) + path.sep;
      if (fs.isDirectorySync(initialPath) && realNewDirectoryPath.startsWith(realInitialPath)) {
        if (!fs.isSymbolicLinkSync(initialPath)) {
          atom.notifications.addWarning('Cannot move a folder into itself');
          return;
        }
      }
      entryName = path.basename(initialPath);
      newPath = path.join(newDirectoryPath, entryName);
      try {
        this.emitter.emit('will-move-entry', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (!fs.existsSync(newDirectoryPath)) {
          fs.makeTreeSync(newDirectoryPath);
        }
        fs.moveSync(initialPath, newPath);
        this.emitter.emit('entry-moved', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        if (error.code === 'EEXIST') {
          return this.moveConflictingEntry(initialPath, newPath, newDirectoryPath);
        } else {
          this.emitter.emit('move-entry-failed', {
            initialPath: initialPath,
            newPath: newPath
          });
          atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newDirectoryPath, {
            detail: error.message
          });
        }
      }
      return true;
    };

    TreeView.prototype.moveConflictingEntry = function(initialPath, newPath, newDirectoryPath) {
      var chosen, entries, entry, error, j, len, repo;
      try {
        if (!fs.isDirectorySync(initialPath)) {
          chosen = atom.confirm({
            message: "'" + (path.relative(newDirectoryPath, newPath)) + "' already exists",
            detailedMessage: 'Do you want to replace it?',
            buttons: ['Replace file', 'Skip', 'Cancel']
          });
          switch (chosen) {
            case 0:
              fs.renameSync(initialPath, newPath);
              this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
              if (repo = repoForPath(newPath)) {
                repo.getPathStatus(initialPath);
                repo.getPathStatus(newPath);
              }
              break;
            case 2:
              return false;
          }
        } else {
          entries = fs.readdirSync(initialPath);
          for (j = 0, len = entries.length; j < len; j++) {
            entry = entries[j];
            if (fs.existsSync(path.join(newPath, entry))) {
              if (!this.moveConflictingEntry(path.join(initialPath, entry), path.join(newPath, entry), newDirectoryPath)) {
                return false;
              }
            } else {
              this.moveEntry(path.join(initialPath, entry), newPath);
            }
          }
          if (!fs.readdirSync(initialPath).length) {
            fs.rmdirSync(initialPath);
          }
        }
      } catch (error1) {
        error = error1;
        this.emitter.emit('move-entry-failed', {
          initialPath: initialPath,
          newPath: newPath
        });
        atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newPath, {
          detail: error.message
        });
      }
      return true;
    };

    TreeView.prototype.onStylesheetsChanged = function() {
      if (!this.isVisible()) {
        return;
      }
      this.element.style.display = 'none';
      this.element.offsetWidth;
      return this.element.style.display = '';
    };

    TreeView.prototype.onMouseDown = function(e) {
      var cmdKey, entryToSelect, shiftKey;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      cmdKey = e.metaKey || (e.ctrlKey && process.platform !== 'darwin');
      if (entryToSelect.classList.contains('selected')) {
        if (e.button === 2 || (e.ctrlKey && process.platform === 'darwin')) {
          return;
        } else {
          shiftKey = e.shiftKey;
          this.selectOnMouseUp = {
            shiftKey: shiftKey,
            cmdKey: cmdKey
          };
          return;
        }
      }
      if (e.shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (e.shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.selectMultipleEntries(entryToSelect);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.onMouseUp = function(e) {
      var cmdKey, entryToSelect, ref2, shiftKey;
      if (this.selectOnMouseUp == null) {
        return;
      }
      ref2 = this.selectOnMouseUp, shiftKey = ref2.shiftKey, cmdKey = ref2.cmdKey;
      this.selectOnMouseUp = null;
      if (!(entryToSelect = e.target.closest('.entry'))) {
        return;
      }
      e.stopPropagation();
      if (shiftKey && cmdKey) {
        this.selectContinuousEntries(entryToSelect, false);
        return this.showMultiSelectMenuIfNecessary();
      } else if (shiftKey) {
        this.selectContinuousEntries(entryToSelect);
        return this.showMultiSelectMenuIfNecessary();
      } else if (cmdKey) {
        this.deselect([entryToSelect]);
        this.lastFocusedEntry = entryToSelect;
        return this.showMultiSelectMenuIfNecessary();
      } else {
        this.selectEntry(entryToSelect);
        return this.showFullMenu();
      }
    };

    TreeView.prototype.selectedPaths = function() {
      var entry, j, len, ref2, results;
      ref2 = this.getSelectedEntries();
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        results.push(entry.getPath());
      }
      return results;
    };

    TreeView.prototype.selectContinuousEntries = function(entry, deselectOthers) {
      var currentSelectedEntry, element, elements, entries, entryIndex, i, j, len, parentContainer, ref2, selectedIndex;
      if (deselectOthers == null) {
        deselectOthers = true;
      }
      currentSelectedEntry = (ref2 = this.lastFocusedEntry) != null ? ref2 : this.selectedEntry();
      parentContainer = entry.parentElement;
      elements = [];
      if (parentContainer === currentSelectedEntry.parentElement) {
        entries = Array.from(parentContainer.querySelectorAll('.entry'));
        entryIndex = entries.indexOf(entry);
        selectedIndex = entries.indexOf(currentSelectedEntry);
        elements = (function() {
          var j, ref3, ref4, results;
          results = [];
          for (i = j = ref3 = entryIndex, ref4 = selectedIndex; ref3 <= ref4 ? j <= ref4 : j >= ref4; i = ref3 <= ref4 ? ++j : --j) {
            results.push(entries[i]);
          }
          return results;
        })();
        if (deselectOthers) {
          this.deselect();
        }
        for (j = 0, len = elements.length; j < len; j++) {
          element = elements[j];
          element.classList.add('selected');
        }
      }
      return elements;
    };

    TreeView.prototype.selectMultipleEntries = function(entry) {
      if (entry != null) {
        entry.classList.toggle('selected');
      }
      return entry;
    };

    TreeView.prototype.showFullMenu = function() {
      this.list.classList.remove('multi-select');
      return this.list.classList.add('full-menu');
    };

    TreeView.prototype.showMultiSelectMenu = function() {
      this.list.classList.remove('full-menu');
      return this.list.classList.add('multi-select');
    };

    TreeView.prototype.showMultiSelectMenuIfNecessary = function() {
      if (this.getSelectedEntries().length > 1) {
        return this.showMultiSelectMenu();
      } else {
        return this.showFullMenu();
      }
    };

    TreeView.prototype.multiSelectEnabled = function() {
      return this.list.classList.contains('multi-select');
    };

    TreeView.prototype.onDragEnter = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        if (!this.dragEventCounts.get(entry)) {
          this.dragEventCounts.set(entry, 0);
        }
        if (!(this.dragEventCounts.get(entry) !== 0 || entry.classList.contains('selected'))) {
          entry.classList.add('drag-over', 'selected');
        }
        return this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) + 1);
      }
    };

    TreeView.prototype.onDragLeave = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) - 1);
        if (this.dragEventCounts.get(entry) === 0 && entry.classList.contains('drag-over')) {
          return entry.classList.remove('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDragStart = function(e) {
      var dragImage, entry, entryPath, initialPaths, j, key, len, newElement, parentSelected, ref2, ref3, target, value;
      this.dragEventCounts = new WeakMap;
      this.selectOnMouseUp = null;
      if (entry = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.rootDragAndDrop.canDragStart(e)) {
          return this.rootDragAndDrop.onDragStart(e);
        }
        dragImage = document.createElement("ol");
        dragImage.classList.add("entries", "list-tree");
        dragImage.style.position = "absolute";
        dragImage.style.top = 0;
        dragImage.style.left = 0;
        dragImage.style.willChange = "transform";
        initialPaths = [];
        ref2 = this.getSelectedEntries();
        for (j = 0, len = ref2.length; j < len; j++) {
          target = ref2[j];
          entryPath = target.querySelector(".name").dataset.path;
          parentSelected = target.parentNode.closest(".entry.selected");
          if (!parentSelected) {
            initialPaths.push(entryPath);
            newElement = target.cloneNode(true);
            if (newElement.classList.contains("directory")) {
              newElement.querySelector(".entries").remove();
            }
            ref3 = getStyleObject(target);
            for (key in ref3) {
              value = ref3[key];
              newElement.style[key] = value;
            }
            newElement.style.paddingLeft = "1em";
            newElement.style.paddingRight = "1em";
            dragImage.append(newElement);
          }
        }
        document.body.appendChild(dragImage);
        e.dataTransfer.effectAllowed = "move";
        e.dataTransfer.setDragImage(dragImage, 0, 0);
        e.dataTransfer.setData("initialPaths", initialPaths);
        return window.requestAnimationFrame(function() {
          return dragImage.remove();
        });
      }
    };

    TreeView.prototype.onDragOver = function(e) {
      var entry;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.dragEventCounts.get(entry) > 0 && !entry.classList.contains('selected')) {
          return entry.classList.add('drag-over', 'selected');
        }
      }
    };

    TreeView.prototype.onDrop = function(e) {
      var entry, file, initialPath, initialPaths, j, k, l, len, len1, newDirectoryPath, ref2, ref3, ref4, ref5, results, results1, results2;
      this.dragEventCounts = new WeakMap;
      if (entry = e.target.closest('.entry.directory')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        newDirectoryPath = (ref2 = entry.querySelector('.name')) != null ? ref2.dataset.path : void 0;
        if (!newDirectoryPath) {
          return false;
        }
        initialPaths = e.dataTransfer.getData('initialPaths');
        if (initialPaths) {
          initialPaths = initialPaths.split(',');
          if (initialPaths.includes(newDirectoryPath)) {
            return;
          }
          entry.classList.remove('drag-over', 'selected');
          results = [];
          for (j = initialPaths.length - 1; j >= 0; j += -1) {
            initialPath = initialPaths[j];
            if ((ref3 = this.entryForPath(initialPath)) != null) {
              if (typeof ref3.collapse === "function") {
                ref3.collapse();
              }
            }
            if (!this.moveEntry(initialPath, newDirectoryPath)) {
              break;
            } else {
              results.push(void 0);
            }
          }
          return results;
        } else {
          entry.classList.remove('selected');
          ref4 = e.dataTransfer.files;
          results1 = [];
          for (k = 0, len = ref4.length; k < len; k++) {
            file = ref4[k];
            if (!this.moveEntry(file.path, newDirectoryPath)) {
              break;
            } else {
              results1.push(void 0);
            }
          }
          return results1;
        }
      } else if (e.dataTransfer.files.length) {
        ref5 = e.dataTransfer.files;
        results2 = [];
        for (l = 0, len1 = ref5.length; l < len1; l++) {
          entry = ref5[l];
          results2.push(atom.project.addPath(entry.path));
        }
        return results2;
      }
    };

    TreeView.prototype.isVisible = function() {
      return this.element.offsetWidth !== 0 || this.element.offsetHeight !== 0;
    };

    return TreeView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
