"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nullRemote = exports["default"] = void 0;

var _endpoint = require("./endpoint");

class Remote {
  constructor(name, url) {
    this.name = name;
    this.url = url;
    const {
      isGithubRepo,
      domain,
      protocol,
      owner,
      repo
    } = githubInfoFromRemote(url);
    this.githubRepo = isGithubRepo;
    this.domain = domain;
    this.protocol = protocol;
    this.owner = owner;
    this.repo = repo;
  }

  getName() {
    return this.name;
  }

  getUrl() {
    return this.url;
  }

  isGithubRepo() {
    return this.githubRepo;
  }

  getProtocol() {
    return this.protocol;
  }

  getDomain() {
    return this.domain;
  }

  getOwner() {
    return this.owner;
  }

  getRepo() {
    return this.repo;
  }

  getNameOr() {
    return this.getName();
  }

  getSlug() {
    if (this.owner === null || this.repo === null) {
      return null;
    }

    return `${this.owner}/${this.repo}`;
  }

  getEndpoint() {
    return this.domain === null ? null : (0, _endpoint.getEndpoint)(this.domain);
  }

  isPresent() {
    return true;
  }

}

exports["default"] = Remote;

function githubInfoFromRemote(remoteUrl) {
  if (!remoteUrl) {
    return {
      isGithubRepo: false,
      domain: null,
      owner: null,
      repo: null
    };
  } //             proto         login   domain           owner    repo


  const regex = /(?:(.+):\/\/)?(?:.+@)?(github\.com)[:/]\/?([^/]+)\/(.+)/;
  const match = remoteUrl.match(regex);

  if (match) {
    return {
      isGithubRepo: true,
      protocol: match[1] || 'ssh',
      domain: match[2],
      owner: match[3],
      repo: match[4].replace(/\.git$/, '')
    };
  } else {
    return {
      isGithubRepo: false,
      protocol: null,
      domain: null,
      owner: null,
      repo: null
    };
  }
}

const nullRemote = {
  getName() {
    return '';
  },

  getUrl() {
    return '';
  },

  isGithubRepo() {
    return false;
  },

  getDomain() {
    return null;
  },

  getProtocol() {
    return null;
  },

  getOwner() {
    return null;
  },

  getRepo() {
    return null;
  },

  getNameOr(fallback) {
    return fallback;
  },

  getSlug() {
    return null;
  },

  getEndpoint() {
    return null;
  },

  isPresent() {
    return false;
  }

};
exports.nullRemote = nullRemote;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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