"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports["default"] = void 0;
const DISABLEMENT = Symbol('disablement');
const ENABLED = Symbol('enabled');
const NO_REASON = Symbol('no-reason'); // Track an operation that may be either enabled or disabled with a message and a reason. EnableableOperation instances
// are immutable to aid passing them as React component props; call `.enable()` or `.disable()` to derive a new
// operation instance with the same callback.

class EnableableOperation {
  constructor(op, options = {}) {
    this.beforeOp = null;
    this.op = op;
    this.afterOp = null;
    this.disablement = options[DISABLEMENT] || ENABLED;
  }

  toggleState(component, stateKey) {
    this.beforeOp = () => {
      component.setState(prevState => {
        return !prevState[stateKey] ? {
          [stateKey]: true
        } : {};
      });
    };

    this.afterOp = () => {
      return new Promise(resolve => {
        component.setState(prevState => {
          return prevState[stateKey] ? {
            [stateKey]: false
          } : {};
        }, resolve);
      });
    };
  }

  isEnabled() {
    return this.disablement === ENABLED;
  }

  async run() {
    if (!this.isEnabled()) {
      throw new Error(this.disablement.message);
    }

    if (this.beforeOp) {
      this.beforeOp();
    }

    let result = undefined;

    try {
      result = await this.op();
    } finally {
      if (this.afterOp) {
        await this.afterOp();
      }
    }

    return result;
  }

  getMessage() {
    return this.disablement.message;
  }

  why() {
    return this.disablement.reason;
  }

  disable(reason = NO_REASON, message = 'disabled') {
    if (!this.isEnabled() && this.disablement.reason === reason && this.disablement.message === message) {
      return this;
    }

    return new this.constructor(this.op, {
      [DISABLEMENT]: {
        reason,
        message
      }
    });
  }

  enable() {
    if (this.isEnabled()) {
      return this;
    }

    return new this.constructor(this.op, {
      [DISABLEMENT]: ENABLED
    });
  }

}

exports["default"] = EnableableOperation;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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