"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.adoptToHapiAuthFormat = adoptToHapiAuthFormat;

var _boom = _interopRequireDefault(require("boom"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
var ResultType;
/** @internal */

(function (ResultType) {
  ResultType["authenticated"] = "authenticated";
  ResultType["redirected"] = "redirected";
  ResultType["rejected"] = "rejected";
})(ResultType || (ResultType = {}));

class AuthResult {
  static authenticated(credentials) {
    return new AuthResult(ResultType.authenticated, credentials);
  }

  static redirected(url) {
    return new AuthResult(ResultType.redirected, url);
  }

  static rejected(error, options = {}) {
    return new AuthResult(ResultType.rejected, {
      error,
      statusCode: options.statusCode
    });
  }

  static isValidResult(candidate) {
    return candidate instanceof AuthResult;
  }

  constructor(type, payload) {
    this.type = type;
    this.payload = payload;
  }

  isAuthenticated() {
    return this.type === ResultType.authenticated;
  }

  isRedirected() {
    return this.type === ResultType.redirected;
  }

  isRejected() {
    return this.type === ResultType.rejected;
  }

}
/**
 * @public
 * A tool set defining an outcome of Auth interceptor for incoming request.
 */


const toolkit = {
  authenticated: AuthResult.authenticated,
  redirected: AuthResult.redirected,
  rejected: AuthResult.rejected
};
/** @public */

/** @public */
function adoptToHapiAuthFormat(fn, sessionStorage) {
  return async function interceptAuth(req, h) {
    try {
      const result = await fn(req, sessionStorage.asScoped(req), toolkit);

      if (AuthResult.isValidResult(result)) {
        if (result.isAuthenticated()) {
          return h.authenticated({
            credentials: result.payload
          });
        }

        if (result.isRedirected()) {
          return h.redirect(result.payload).takeover();
        }

        if (result.isRejected()) {
          const {
            error,
            statusCode
          } = result.payload;
          return _boom.default.boomify(error, {
            statusCode
          });
        }
      }

      throw new Error(`Unexpected result from Authenticate. Expected AuthResult, but given: ${result}.`);
    } catch (error) {
      return _boom.default.internal(error.message, {
        statusCode: 500
      });
    }
  };
}