/**
 * @file The free group generated by elements of `A`, up to equality. Note that the `Setoid` and `Monoid` instances differ
 * from the standard such instances for `Array<Either<A, A>>`; two elements of the free group are equal iff they are equal
 * after being reduced to "canonical form", i.e., cancelling adjacent inverses.
 *
 * Adapted from https://hackage.haskell.org/package/free-algebras-0.0.7.0/docs/Data-Group-Free.html
 */
import { empty as emptyArray, getMonoid as getArrayMonoid, getSetoid as getArraySetoid, array } from './Array';
import { getSetoid as getEitherSetoid, left, right } from './Either';
import { fromEquals } from './Setoid';
export const URI = 'FreeGroup';
/**
 * @since 1.13.0
 */
export class FreeGroup {
    constructor(value) {
        this.value = value;
    }
    map(f) {
        return new FreeGroup(this.value.map(e => e.bimap(f, f)));
    }
    ap(fab) {
        return fab.chain(f => this.map(f)); // <- derived
    }
    ap_(fb) {
        return fb.ap(this);
    }
    chain(f) {
        return new FreeGroup(array.chain(this.value, e => e.bimap(f, f).value.value));
    }
}
const of = (a) => {
    return new FreeGroup([right(a)]);
};
const map = (fa, f) => {
    return fa.map(f);
};
const ap = (fab, fa) => {
    return fa.ap(fab);
};
const chain = (fa, f) => {
    return fa.chain(f);
};
/**
 * Smart constructor which normalizes an array
 *
 * @since 1.13.0
 */
export const fromArray = (S) => {
    const normalizeS = normalize(S);
    return as => new FreeGroup(normalizeS(as));
};
/**
 * Reduce a term of a free group to canonical form, i.e. cancelling adjacent inverses.
 *
 * @since 1.13.0
 */
export const normalize = (S) => (g) => {
    return g.reduceRight((acc, s) => {
        if (acc.length > 0) {
            const head = acc[0];
            const tail = acc.slice(1);
            if (head._tag !== s._tag && S.equals(head.value, s.value)) {
                return tail;
            }
        }
        acc.unshift(s);
        return acc;
    }, []);
};
/**
 * @since 1.13.0
 */
export const getSetoid = (S) => {
    const AS = getArraySetoid(getEitherSetoid(S, S));
    const normalizeS = normalize(S);
    return fromEquals((x, y) => AS.equals(normalizeS(x.value), normalizeS(y.value)));
};
/**
 * @since 1.13.0
 */
export const empty = new FreeGroup(emptyArray);
/**
 * @since 1.13.0
 */
export const getGroup = (S) => {
    const M = getArrayMonoid();
    const normalizeS = normalize(S);
    return {
        concat: (x, y) => new FreeGroup(normalizeS(M.concat(x.value, y.value))),
        empty,
        inverse: x => new FreeGroup(x.value.reverse().map(s => (s.isLeft() ? right(s.value) : left(s.value))))
    };
};
/**
 * @since 1.13.0
 */
export const freeGroup = {
    URI,
    of,
    map,
    ap,
    chain
};
