"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var _a;
var d3_array_1 = require("d3-array");
var d3_scale_1 = require("d3-scale");
var luxon_1 = require("luxon");
var commons_1 = require("../commons");
var SCALES = (_a = {},
    _a["linear" /* Linear */] = d3_scale_1.scaleLinear,
    _a["log" /* Log */] = d3_scale_1.scaleLog,
    _a["sqrt" /* Sqrt */] = d3_scale_1.scaleSqrt,
    _a["time" /* Time */] = d3_scale_1.scaleUtc,
    _a);
function limitToMin(value, positive) {
    if (value === 0) {
        return positive ? 1 : -1;
    }
    return value;
}
exports.limitToMin = limitToMin;
/**
 * As log(0) = -Infinite, a log scale domain must be strictly-positive
 * or strictly-negative; the domain must not include or cross zero value.
 * We need to limit the domain scale to the right value on all possible cases.
 * @param domain the domain to limit
 */
function limitLogScaleDomain(domain) {
    if (domain[0] === 0) {
        if (domain[1] > 0) {
            return [1, domain[1]];
        }
        else if (domain[1] < 0) {
            return [-1, domain[1]];
        }
        else {
            return [1, 1];
        }
    }
    if (domain[1] === 0) {
        if (domain[0] > 0) {
            return [domain[0], 1];
        }
        else if (domain[0] < 0) {
            return [domain[0], -1];
        }
        else {
            return [1, 1];
        }
    }
    if (domain[0] < 0 && domain[1] > 0) {
        var isD0Min = Math.abs(domain[1]) - Math.abs(domain[0]) >= 0;
        if (isD0Min) {
            return [1, domain[1]];
        }
        else {
            return [domain[0], -1];
        }
    }
    if (domain[0] > 0 && domain[1] < 0) {
        var isD0Max = Math.abs(domain[0]) - Math.abs(domain[1]) >= 0;
        if (isD0Max) {
            return [domain[0], 1];
        }
        else {
            return [-1, domain[1]];
        }
    }
    return domain;
}
exports.limitLogScaleDomain = limitLogScaleDomain;
var ScaleContinuous = /** @class */ (function () {
    function ScaleContinuous(type, domain, range, 
    /**
     * The desidered bandwidth for a linear band scale.
     * @default 0
     */
    bandwidth, 
    /**
     * The min interval computed on the XDomain. Not available for yDomains.
     * @default 0
     */
    minInterval, 
    /**
     * A time zone identifier. Can be any IANA zone supported by he host environment,
     * or a fixed-offset name of the form 'utc+3', or the strings 'local' or 'utc'.
     * @default 'utc'
     */
    timeZone, 
    /**
     * The number of bars in the cluster. Used to correctly compute scales when
     * using padding between bars.
     * @default 1
     */
    totalBarsInCluster, 
    /**
     * The proportion of the range that is reserved for blank space between bands
     * A number between 0 and 1.
     * @default 0
     */
    barsPadding) {
        var _this = this;
        if (bandwidth === void 0) { bandwidth = 0; }
        if (minInterval === void 0) { minInterval = 0; }
        if (timeZone === void 0) { timeZone = 'utc'; }
        if (totalBarsInCluster === void 0) { totalBarsInCluster = 1; }
        if (barsPadding === void 0) { barsPadding = 0; }
        this.d3Scale = SCALES[type]();
        if (type === "log" /* Log */) {
            this.domain = limitLogScaleDomain(domain);
            this.d3Scale.domain(this.domain);
        }
        else {
            this.domain = domain;
            this.d3Scale.domain(domain);
        }
        var safeBarPadding = commons_1.clamp(barsPadding, 0, 1);
        this.barsPadding = safeBarPadding;
        this.bandwidth = bandwidth * (1 - safeBarPadding);
        this.bandwidthPadding = bandwidth * safeBarPadding;
        this.d3Scale.range(range);
        this.step = 0;
        this.type = type;
        this.range = range;
        this.minInterval = minInterval;
        this.isInverted = this.domain[0] > this.domain[1];
        this.timeZone = timeZone;
        this.totalBarsInCluster = totalBarsInCluster;
        if (type === "time" /* Time */) {
            var startDomain = luxon_1.DateTime.fromMillis(this.domain[0], { zone: this.timeZone });
            var endDomain = luxon_1.DateTime.fromMillis(this.domain[1], { zone: this.timeZone });
            var offset_1 = startDomain.offset;
            var shiftedDomainMin = startDomain.plus({ minutes: offset_1 }).toMillis();
            var shiftedDomainMax = endDomain.plus({ minutes: offset_1 }).toMillis();
            var tzShiftedScale = d3_scale_1.scaleUtc().domain([shiftedDomainMin, shiftedDomainMax]);
            var rawTicks = tzShiftedScale.ticks();
            var timePerTick = (shiftedDomainMax - shiftedDomainMin) / rawTicks.length;
            var hasHourTicks_1 = timePerTick < 1000 * 60 * 60 * 12;
            this.tickValues = rawTicks.map(function (d) {
                var currentDateTime = luxon_1.DateTime.fromJSDate(d, { zone: _this.timeZone });
                var currentOffset = hasHourTicks_1 ? offset_1 : currentDateTime.offset;
                return currentDateTime
                    .minus({ minutes: currentOffset })
                    .toMillis();
            });
        }
        else {
            if (this.minInterval > 0) {
                var intervalCount = Math.floor((this.domain[1] - this.domain[0]) / this.minInterval);
                this.tickValues = new Array(intervalCount + 1).fill(0).map(function (d, i) {
                    return _this.domain[0] + i * _this.minInterval;
                });
            }
            else {
                this.tickValues = this.d3Scale.ticks();
            }
        }
    }
    ScaleContinuous.prototype.scale = function (value) {
        return this.d3Scale(value) + (this.bandwidthPadding / 2) * this.totalBarsInCluster;
    };
    ScaleContinuous.prototype.ticks = function () {
        return this.tickValues;
    };
    ScaleContinuous.prototype.invert = function (value) {
        var invertedValue = this.d3Scale.invert(value);
        if (this.type === "time" /* Time */) {
            invertedValue = luxon_1.DateTime.fromJSDate(invertedValue).toMillis();
        }
        return invertedValue;
    };
    ScaleContinuous.prototype.invertWithStep = function (value, data) {
        var invertedValue = this.invert(value - this.bandwidth / 2);
        var leftIndex = d3_array_1.bisectLeft(data, invertedValue);
        if (leftIndex === 0) {
            // is equal or less than the first value
            var prevValue1 = data[leftIndex];
            if (data.length === 0) {
                return prevValue1;
            }
            var nextValue1 = data[leftIndex + 1];
            var nextDiff1 = Math.abs(nextValue1 - invertedValue);
            var prevDiff1 = Math.abs(invertedValue - prevValue1);
            if (nextDiff1 < prevDiff1) {
                return nextValue1;
            }
            return prevValue1;
        }
        if (leftIndex === data.length) {
            return data[leftIndex - 1];
        }
        var nextValue = data[leftIndex];
        var prevValue = data[leftIndex - 1];
        var nextDiff = Math.abs(nextValue - invertedValue);
        var prevDiff = Math.abs(invertedValue - prevValue);
        if (nextDiff <= prevDiff) {
            return nextValue;
        }
        return prevValue;
    };
    return ScaleContinuous;
}());
exports.ScaleContinuous = ScaleContinuous;
function isContinuousScale(scale) {
    return scale.type !== "ordinal" /* Ordinal */;
}
exports.isContinuousScale = isContinuousScale;
function isLogarithmicScale(scale) {
    return scale.type === "log" /* Log */;
}
exports.isLogarithmicScale = isLogarithmicScale;
//# sourceMappingURL=scale_continuous.js.map