import { phantom, toString } from './function';
import { none, some } from './Option';
import { fromEquals } from './Setoid';
export const URI = 'These';
export class This {
    constructor(value) {
        this.value = value;
        this._tag = 'This';
    }
    map(f) {
        return this;
    }
    bimap(f, g) {
        return new This(f(this.value));
    }
    reduce(b, f) {
        return b;
    }
    /** Applies a function to each case in the data structure */
    fold(this_, that, both) {
        return this_(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `this_(${toString(this.value)})`;
    }
    /** Returns `true` if the these is `This`, `false` otherwise */
    isThis() {
        return true;
    }
    /** Returns `true` if the these is `That`, `false` otherwise */
    isThat() {
        return false;
    }
    /** Returns `true` if the these is `Both`, `false` otherwise */
    isBoth() {
        return false;
    }
}
export class That {
    constructor(value) {
        this.value = value;
        this._tag = 'That';
    }
    map(f) {
        return new That(f(this.value));
    }
    bimap(f, g) {
        return new That(g(this.value));
    }
    reduce(b, f) {
        return f(b, this.value);
    }
    fold(this_, that, both) {
        return that(this.value);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `that(${toString(this.value)})`;
    }
    isThis() {
        return false;
    }
    isThat() {
        return true;
    }
    isBoth() {
        return false;
    }
}
export class Both {
    constructor(l, a) {
        this.l = l;
        this.a = a;
        this._tag = 'Both';
    }
    map(f) {
        return new Both(this.l, f(this.a));
    }
    bimap(f, g) {
        return new Both(f(this.l), g(this.a));
    }
    reduce(b, f) {
        return f(b, this.a);
    }
    fold(this_, that, both) {
        return both(this.l, this.a);
    }
    inspect() {
        return this.toString();
    }
    toString() {
        return `both(${toString(this.l)}, ${toString(this.a)})`;
    }
    isThis() {
        return false;
    }
    isThat() {
        return false;
    }
    isBoth() {
        return true;
    }
}
/**
 * @since 1.17.0
 */
export const getShow = (SL, SA) => {
    return {
        show: t => t.fold(l => `this_(${SL.show(l)})`, a => `that(${SA.show(a)})`, (l, a) => `both(${SL.show(l)}, ${SA.show(a)})`)
    };
};
/**
 * @since 1.0.0
 */
export const getSetoid = (SL, SA) => {
    return fromEquals((x, y) => x.isThis()
        ? y.isThis() && SL.equals(x.value, y.value)
        : x.isThat()
            ? y.isThat() && SA.equals(x.value, y.value)
            : y.isBoth() && SL.equals(x.l, y.l) && SA.equals(x.a, y.a));
};
/**
 * @since 1.0.0
 */
export const getSemigroup = (SL, SA) => {
    return {
        concat: (x, y) => x.isThis()
            ? y.isThis()
                ? this_(SL.concat(x.value, y.value))
                : y.isThat()
                    ? both(x.value, y.value)
                    : both(SL.concat(x.value, y.l), y.a)
            : x.isThat()
                ? y.isThis()
                    ? both(y.value, x.value)
                    : y.isThat()
                        ? that(SA.concat(x.value, y.value))
                        : both(y.l, SA.concat(x.value, y.a))
                : y.isThis()
                    ? both(SL.concat(x.l, y.value), x.a)
                    : y.isThat()
                        ? both(x.l, SA.concat(x.a, y.value))
                        : both(SL.concat(x.l, y.l), SA.concat(x.a, y.a))
    };
};
const map = (fa, f) => {
    return fa.map(f);
};
/**
 * @since 1.0.0
 */
export const that = (a) => {
    return new That(a);
};
const of = that;
const ap = (S) => (fab, fa) => {
    return chain(S)(fab, f => map(fa, f));
};
const chain = (S) => (fa, f) => {
    if (fa.isThis()) {
        return this_(fa.value);
    }
    else if (fa.isThat()) {
        return f(fa.value);
    }
    else {
        const fb = f(fa.a);
        return fb.isThis()
            ? this_(S.concat(fa.l, fb.value))
            : fb.isThat()
                ? both(fa.l, fb.value)
                : both(S.concat(fa.l, fb.l), fb.a);
    }
};
/**
 * @since 1.0.0
 */
export const getMonad = (S) => {
    return {
        URI,
        _L: phantom,
        map,
        of,
        ap: ap(S),
        chain: chain(S)
    };
};
const bimap = (fla, f, g) => {
    return fla.bimap(f, g);
};
const reduce = (fa, b, f) => {
    return fa.reduce(b, f);
};
const foldMap = (M) => (fa, f) => {
    return fa.isThis() ? M.empty : fa.isThat() ? f(fa.value) : f(fa.a);
};
const foldr = (fa, b, f) => {
    return fa.isThis() ? b : fa.isThat() ? f(fa.value, b) : f(fa.a, b);
};
const traverse = (F) => (ta, f) => {
    return ta.isThis()
        ? F.of(this_(ta.value))
        : ta.isThat()
            ? F.map(f(ta.value), that)
            : F.map(f(ta.a), b => both(ta.l, b));
};
const sequence = (F) => (ta) => {
    return ta.isThis()
        ? F.of(this_(ta.value))
        : ta.isThat()
            ? F.map(ta.value, that)
            : F.map(ta.a, b => both(ta.l, b));
};
/**
 * @since 1.0.0
 */
export const this_ = (l) => {
    return new This(l);
};
/**
 * @since 1.0.0
 */
export const both = (l, a) => {
    return new Both(l, a);
};
/**
 *
 * @example
 * import { fromThese, this_, that, both } from 'fp-ts/lib/These'
 *
 * const from = fromThese('a', 1)
 * assert.deepStrictEqual(from(this_('b')), ['b', 1])
 * assert.deepStrictEqual(from(that(2)), ['a', 2])
 * assert.deepStrictEqual(from(both('b', 2)), ['b', 2])
 *
 * @since 1.0.0
 */
export const fromThese = (defaultThis, defaultThat) => (fa) => {
    return fa.isThis() ? [fa.value, defaultThat] : fa.isThat() ? [defaultThis, fa.value] : [fa.l, fa.a];
};
/**
 * Returns an `L` value if possible
 *
 * @example
 * import { theseLeft, this_, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(theseLeft(this_('a')), some('a'))
 * assert.deepStrictEqual(theseLeft(that(1)), none)
 * assert.deepStrictEqual(theseLeft(both('a', 1)), some('a'))
 *
 * @since 1.0.0
 */
export const theseLeft = (fa) => {
    return fa.isThis() ? some(fa.value) : fa.isThat() ? none : some(fa.l);
};
/**
 * Returns an `A` value if possible
 *
 * @example
 * import { theseRight, this_, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(theseRight(this_('a')), none)
 * assert.deepStrictEqual(theseRight(that(1)), some(1))
 * assert.deepStrictEqual(theseRight(both('a', 1)), some(1))
 *
 * @since 1.0.0
 */
export const theseRight = (fa) => {
    return fa.isThis() ? none : fa.isThat() ? some(fa.value) : some(fa.a);
};
/**
 * Returns `true` if the these is an instance of `This`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isThis = (fa) => {
    return fa.isThis();
};
/**
 * Returns `true` if the these is an instance of `That`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isThat = (fa) => {
    return fa.isThat();
};
/**
 * Returns `true` if the these is an instance of `Both`, `false` otherwise
 *
 * @since 1.0.0
 */
export const isBoth = (fa) => {
    return fa.isBoth();
};
/**
 * @example
 * import { thisOrBoth, this_, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(thisOrBoth('a', none), this_('a'))
 * assert.deepStrictEqual(thisOrBoth('a', some(1)), both('a', 1))
 *
 * @since 1.13.0
 */
export const thisOrBoth = (defaultThis, ma) => {
    return ma.isNone() ? this_(defaultThis) : both(defaultThis, ma.value);
};
/**
 * @example
 * import { thatOrBoth, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(thatOrBoth(1, none), that(1))
 * assert.deepStrictEqual(thatOrBoth(1, some('a')), both('a', 1))
 *
 * @since 1.13.0
 */
export const thatOrBoth = (defaultThat, ml) => {
    return ml.isNone() ? that(defaultThat) : both(ml.value, defaultThat);
};
/**
 * Returns the `L` value if and only if the value is constructed with `This`
 *
 * @example
 * import { theseThis, this_, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(theseThis(this_('a')), some('a'))
 * assert.deepStrictEqual(theseThis(that(1)), none)
 * assert.deepStrictEqual(theseThis(both('a', 1)), none)
 *
 * @since 1.13.0
 */
export const theseThis = (fa) => {
    return fa.isThis() ? some(fa.value) : none;
};
/**
 * Returns the `A` value if and only if the value is constructed with `That`
 *
 * @example
 * import { theseThat, this_, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(theseThat(this_('a')), none)
 * assert.deepStrictEqual(theseThat(that(1)), some(1))
 * assert.deepStrictEqual(theseThat(both('a', 1)), none)
 *
 *
 * @since 1.13.0
 */
export const theseThat = (fa) => {
    return fa.isThat() ? some(fa.value) : none;
};
/**
 * Takes a pair of `Option`s and attempts to create a `These` from them
 *
 * @example
 * import { fromOptions, this_, that, both } from 'fp-ts/lib/These'
 * import { none, some } from 'fp-ts/lib/Option'
 *
 * assert.deepStrictEqual(fromOptions(none, none), none)
 * assert.deepStrictEqual(fromOptions(some('a'), none), some(this_('a')))
 * assert.deepStrictEqual(fromOptions(none, some(1)), some(that(1)))
 * assert.deepStrictEqual(fromOptions(some('a'), some(1)), some(both('a', 1)))
 *
 * @since 1.13.0
 */
export const fromOptions = (fl, fa) => {
    return fl.foldL(() => fa.fold(none, a => some(that(a))), l => fa.foldL(() => some(this_(l)), a => some(both(l, a))));
};
/**
 * @example
 * import { fromEither, this_, that } from 'fp-ts/lib/These'
 * import { left, right } from 'fp-ts/lib/Either'
 *
 * assert.deepStrictEqual(fromEither(left('a')), this_('a'))
 * assert.deepStrictEqual(fromEither(right(1)), that(1))
 *
 * @since 1.13.0
 */
export const fromEither = (fa) => {
    return fa.isLeft() ? this_(fa.value) : that(fa.value);
};
/**
 * @since 1.0.0
 */
export const these = {
    URI,
    map,
    bimap,
    reduce,
    foldMap,
    foldr,
    traverse,
    sequence
};
