import { fromEquals } from './Setoid';
import { not, identity } from './function';
/**
 * @since 1.17.0
 */
export const getShow = (SA) => {
    return {
        show: s => {
            let elements = '';
            s.forEach(a => {
                elements += SA.show(a) + ', ';
            });
            if (elements !== '') {
                elements = elements.substring(0, elements.length - 2);
            }
            return `new Set([${elements}])`;
        }
    };
};
/**
 * @since 1.14.0
 */
export const empty = new Set();
/**
 * @since 1.0.0
 */
export const toArray = (O) => (x) => {
    const r = [];
    x.forEach(e => r.push(e));
    return r.sort(O.compare);
};
/**
 * @since 1.0.0
 */
export const getSetoid = (S) => {
    const subsetS = subset(S);
    return fromEquals((x, y) => subsetS(x, y) && subsetS(y, x));
};
/**
 * @since 1.0.0
 */
export const some = (x, predicate) => {
    const values = x.values();
    let e;
    let found = false;
    while (!found && !(e = values.next()).done) {
        found = predicate(e.value);
    }
    return found;
};
/**
 * Projects a Set through a function
 *
 * @since 1.2.0
 */
export const map = (S) => {
    const has = elem(S);
    return (set, f) => {
        const r = new Set();
        set.forEach(e => {
            const v = f(e);
            if (!has(v, r)) {
                r.add(v);
            }
        });
        return r;
    };
};
/**
 * @since 1.0.0
 */
export const every = (x, predicate) => {
    return !some(x, not(predicate));
};
/**
 * @since 1.2.0
 */
export const chain = (S) => {
    const has = elem(S);
    return (set, f) => {
        let r = new Set();
        set.forEach(e => {
            f(e).forEach(e => {
                if (!has(e, r)) {
                    r.add(e);
                }
            });
        });
        return r;
    };
};
/**
 * `true` if and only if every element in the first set is an element of the second set
 *
 * @since 1.0.0
 */
export const subset = (S) => {
    const has = elem(S);
    return (x, y) => every(x, a => has(a, y));
};
export function filter(x, predicate) {
    const values = x.values();
    let e;
    let r = new Set();
    while (!(e = values.next()).done) {
        const value = e.value;
        if (predicate(value)) {
            r.add(value);
        }
    }
    return r;
}
export function partition(x, predicate) {
    const values = x.values();
    let e;
    let right = new Set();
    let left = new Set();
    while (!(e = values.next()).done) {
        const value = e.value;
        if (predicate(value)) {
            right.add(value);
        }
        else {
            left.add(value);
        }
    }
    return { left, right };
}
/**
 * Use `elem` instead
 * @since 1.0.0
 * @deprecated
 */
export const member = (S) => {
    const has = elem(S);
    return set => a => has(a, set);
};
/**
 * Test if a value is a member of a set
 *
 * @since 1.14.0
 */
export const elem = (S) => (a, x) => {
    return some(x, (ax) => S.equals(a, ax));
};
/**
 * Form the union of two sets
 *
 * @since 1.0.0
 */
export const union = (S) => {
    const has = elem(S);
    return (x, y) => {
        const r = new Set(x);
        y.forEach(e => {
            if (!has(e, r)) {
                r.add(e);
            }
        });
        return r;
    };
};
/**
 * The set of elements which are in both the first and second set
 *
 * @since 1.0.0
 */
export const intersection = (S) => {
    const has = elem(S);
    return (x, y) => {
        const r = new Set();
        x.forEach(e => {
            if (has(e, y)) {
                r.add(e);
            }
        });
        return r;
    };
};
/**
 * @since 1.2.0
 */
export const partitionMap = (SL, SR) => (x, f) => {
    const values = x.values();
    let e;
    let left = new Set();
    let right = new Set();
    const hasL = elem(SL);
    const hasR = elem(SR);
    while (!(e = values.next()).done) {
        const v = f(e.value);
        if (v.isLeft()) {
            if (!hasL(v.value, left)) {
                left.add(v.value);
            }
        }
        else {
            if (!hasR(v.value, right)) {
                right.add(v.value);
            }
        }
    }
    return { left, right };
};
/**
 * Use `difference2v` instead
 *
 * @since 1.0.0
 * @deprecated
 */
export const difference = (S) => {
    const d = difference2v(S);
    return (x, y) => d(y, x);
};
/**
 * Form the set difference (`x` - `y`)
 *
 * @example
 * import { difference2v } from 'fp-ts/lib/Set'
 * import { setoidNumber } from 'fp-ts/lib/Setoid'
 *
 * assert.deepStrictEqual(difference2v(setoidNumber)(new Set([1, 2]), new Set([1, 3])), new Set([2]))
 *
 *
 * @since 1.12.0
 */
export const difference2v = (S) => {
    const has = elem(S);
    return (x, y) => filter(x, a => !has(a, y));
};
/**
 * @since 1.0.0
 */
export const getUnionMonoid = (S) => {
    return {
        concat: union(S),
        empty
    };
};
/**
 * @since 1.0.0
 */
export const getIntersectionSemigroup = (S) => {
    return {
        concat: intersection(S)
    };
};
/**
 * @since 1.0.0
 */
export const reduce = (O) => {
    const toArrayO = toArray(O);
    return (fa, b, f) => toArrayO(fa).reduce(f, b);
};
/**
 * @since 1.14.0
 */
export const foldMap = (O, M) => {
    const toArrayO = toArray(O);
    return (fa, f) => toArrayO(fa).reduce((b, a) => M.concat(b, f(a)), M.empty);
};
/**
 * Create a set with one element
 *
 * @since 1.0.0
 */
export const singleton = (a) => {
    return new Set([a]);
};
/**
 * Insert a value into a set
 *
 * @since 1.0.0
 */
export const insert = (S) => {
    const has = elem(S);
    return (a, x) => {
        if (!has(a, x)) {
            const r = new Set(x);
            r.add(a);
            return r;
        }
        else {
            return x;
        }
    };
};
/**
 * Delete a value from a set
 *
 * @since 1.0.0
 */
export const remove = (S) => (a, x) => {
    return filter(x, (ax) => !S.equals(a, ax));
};
/**
 * Create a set from an array
 *
 * @since 1.2.0
 */
export const fromArray = (S) => (as) => {
    const len = as.length;
    const r = new Set();
    const has = elem(S);
    for (let i = 0; i < len; i++) {
        const a = as[i];
        if (!has(a, r)) {
            r.add(a);
        }
    }
    return r;
};
/**
 * @since 1.12.0
 */
export const compact = (S) => {
    const filterMapS = filterMap(S);
    return fa => filterMapS(fa, identity);
};
/**
 * @since 1.12.0
 */
export const separate = (SL, SR) => (fa) => {
    const hasL = elem(SL);
    const hasR = elem(SR);
    const left = new Set();
    const right = new Set();
    fa.forEach(e => {
        if (e.isLeft()) {
            if (!hasL(e.value, left)) {
                left.add(e.value);
            }
        }
        else {
            if (!hasR(e.value, right)) {
                right.add(e.value);
            }
        }
    });
    return { left, right };
};
/**
 * @since 1.12.0
 */
export const filterMap = (S) => {
    const has = elem(S);
    return (fa, f) => {
        const r = new Set();
        fa.forEach(a => {
            const ob = f(a);
            if (ob.isSome() && !has(ob.value, r)) {
                r.add(ob.value);
            }
        });
        return r;
    };
};
