"use strict";
var __extends = (this && this.__extends) || (function () {
    var extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
var natural = require("natural");
var SpellingManager_1 = require("./SpellingManager");
var TokenCheckStatus_1 = require("./TokenCheckStatus");
/**
 * A token-based spelling manager that uses non-processed list of words to
 * provides correctness testing and suggestions. This has both case-sensitive
 * and -insensitive methods along with suggestions that are capitalized based
 * on the incorrect word.
 */
var TokenSpellingManager = /** @class */ (function (_super) {
    __extends(TokenSpellingManager, _super);
    function TokenSpellingManager() {
        var _this = _super !== null && _super.apply(this, arguments) || this;
        _this.maximumDistance = 0.9;
        _this.sensitive = {};
        _this.insensitive = {};
        return _this;
    }
    /**
     * Adds a word to the manager. If the word is in all lowercase, then it is
     * added as a case insensitive word, otherwise it is added as a case
     * sensitive result. If a token starts with "!", then it is automatically
     * case-sensitive.
     */
    TokenSpellingManager.prototype.add = function (token) {
        // If we aren't an array, then wrap it in an array.
        var tokens;
        if (typeof token === "string") {
            tokens = [token];
        }
        else {
            tokens = token;
        }
        // Loop through all the tokens and add each one.
        for (var _i = 0, tokens_1 = tokens; _i < tokens_1.length; _i++) {
            var t = tokens_1[_i];
            if (t && t.trim() !== "") {
                // If we have at least one uppercase character, we are
                // considered case sensitive. We don't test for lowercase
                // because we want to ignore things like single quotes for
                // posessives or contractions.
                if (/[A-Z]/.test(t)) {
                    this.addCaseSensitive(t);
                }
                else if (/^!/.test(t)) {
                    this.addCaseSensitive(t.substring(1));
                }
                else {
                    this.addCaseInsensitive(t);
                }
            }
        }
    };
    /**
     * Adds a case-sensitive token, if it hasn't already been added.
     *
     * There is no check to see if this token is already in the case-insensitive
     * list.
     */
    TokenSpellingManager.prototype.addCaseSensitive = function (token) {
        if (token && token.trim() !== "") {
            this.sensitive[token] = true;
        }
    };
    /**
     * Adds a case-insensitive token, if it hasn't already been added.
     *
     * There is no check to see if this token is already in the case-sensitive
     * list.
     */
    TokenSpellingManager.prototype.addCaseInsensitive = function (token) {
        if (token && token.trim() !== "") {
            this.insensitive[token.toLowerCase()] = true;
        }
    };
    /**
     * Checks the token to determine if it is correct or incorrect.
     */
    TokenSpellingManager.prototype.check = function (token) {
        if (token in this.sensitive) {
            return TokenCheckStatus_1.TokenCheckStatus.Correct;
        }
        if (token.toLowerCase() in this.insensitive) {
            return TokenCheckStatus_1.TokenCheckStatus.Correct;
        }
        return TokenCheckStatus_1.TokenCheckStatus.Unknown;
    };
    /**
     * Lists all of the words in a combined list appropriate for adding back
     * into the manager.
     */
    TokenSpellingManager.prototype.list = function () {
        // Gather up the list of sensitive items, prefixing with "!" for those
        // which would normally be in the case-insensitive list if they were
        // re-add()ed.
        var list = new Array();
        for (var token in this.sensitive) {
            if (token === token.toLowerCase()) {
                list.push("!" + token);
            }
            else {
                list.push(token);
            }
        }
        // Add in the insensitive items.
        for (var token in this.insensitive) {
            list.push(token);
        }
        // Sort the results because we always produce sorted results. Then
        // return it.
        list.sort();
        return list;
    };
    /**
     * Removes tokens, if it has been added.
     */
    TokenSpellingManager.prototype.remove = function (token) {
        if (token && token.trim() !== "") {
            this.removeCaseSensitive(token);
            this.removeCaseInsensitive(token.toLowerCase());
        }
    };
    /**
     * Removes a case-sensitive token, if it has been added.
     */
    TokenSpellingManager.prototype.removeCaseSensitive = function (token) {
        if (token && token.trim() !== "") {
            delete this.sensitive[token];
        }
    };
    /**
     * Removes a case-insensitive token, if it has been added.
     */
    TokenSpellingManager.prototype.removeCaseInsensitive = function (token) {
        if (token && token.trim() !== "") {
            delete this.insensitive[token];
        }
    };
    /**
     * Gives a suggestion for a token, sorted by likelyhood with the first item
     * in the resulting array being the most likely.
     */
    TokenSpellingManager.prototype.suggest = function (input) {
        // If the input is blank or null, then we don't have a suggestion.
        if (!input || input.trim().length === 0) {
            return [];
        }
        // Gather up all the suggestions from the case-sensitive list.
        var weights = [];
        for (var token in this.sensitive) {
            var distance = natural.JaroWinklerDistance(input, token);
            if (distance >= this.maximumDistance) {
                weights.push({ distance: distance, token: token });
            }
        }
        // Also gather up the weights from the insensitive list. When we go
        // through this one, we try to find the "best" approach which means if
        // the input is all uppercase, then we compare that. Otherwise, we try
        // initial capital, and finally we see if lowercase would work better.
        for (var token in this.insensitive) {
            // Figure out the best approah.
            var test_1 = token;
            if (/[A-Z].*[A-Z]/.test(input)) {
                test_1 = test_1.toUpperCase();
            }
            else if (/^[A-Z]/.test(input)) {
                test_1 = test_1.charAt(0).toUpperCase() + test_1.slice(1);
            }
            // Figure out the distance as above.
            var distance = natural.JaroWinklerDistance(input, test_1);
            if (distance >= this.maximumDistance) {
                weights.push({ distance: distance, token: test_1 });
            }
        }
        // Sort the list based on the distances. This will have the first key
        // be the highest distance.
        var keys = Object.keys(weights).sort(function (key1, key2) {
            var value1 = weights[key1];
            var value2 = weights[key2];
            if (value1.distance !== value2.distance) {
                return value1.distance - value2.distance;
            }
            return value1.token.localeCompare(value2.token);
        });
        // Go through the resulting items and pull out an ordered list.
        var results = [];
        for (var _i = 0, keys_1 = keys; _i < keys_1.length; _i++) {
            var key = keys_1[_i];
            results.push(weights[key].token);
        }
        return results;
    };
    return TokenSpellingManager;
}(SpellingManager_1.SpellingManager));
exports.TokenSpellingManager = TokenSpellingManager;
//# sourceMappingURL=TokenSpellingManager.js.map